//
// MODULE NAME:  CONFIG.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles configuration support for the analyzers, such
//	as setting options, and loading and saving those options to the
//	configuration file, ETHERPRO.CFG or SNOOPER.CFG.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/12/29.	2.0, original.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

#define BANNER_SECONDS	2		// number of seconds to display banner.

#ifdef ALARMS
extern USHORT GetAlarmTableSize (USHORT *RecordSize, USHORT *NumberOfRecords);
extern VOID WriteAlarmEntry (PVOID Buffer, USHORT Index);
extern VOID ReadAlarmEntry (PVOID Buffer, USHORT Index);
#endif

#ifdef ETHERPROBE
#define LOG_FILE_NAME		".\\ETHERPRO.LOG"
#ifdef DEMO
#define CONFIG_FILE_NAME	"ETHERPRO.DCF"
#else
#define CONFIG_FILE_NAME	"ETHERPRO.CFG"
#endif
#endif

#ifdef SNOOPER
#define LOG_FILE_NAME		".\\SNOOPER.LOG"
#ifdef DEMO
#define CONFIG_FILE_NAME	"SNOOPER.DCF"
#else
#define CONFIG_FILE_NAME	"SNOOPER.CFG"
#endif
#endif

VOID SaveConfiguration ()
{
    FILE *File;
    CONFIG_RECORD Config;

#ifdef ALARMS
    USHORT NumberOfAlarms, AlarmRecordSize, i;
#endif

    if ((File=fopen (CONFIG_FILE_NAME, "wb")) == NULL) {
	return;
    }

    Config.DisplayFilter = DisplayFilter;
    Config.CaptureFilter = CaptureFilter;
    Config.HighestLevelOnly = HighestLevelOnly;
    Config.DisplayAbsoluteTimes = DisplayAbsoluteTimes;
    Config.DisplayHexData = DisplayHexData;
    Config.DisplayProtocolDetail = DisplayProtocolDetail;
    Config.DisplaySummaryLevel = DisplaySummaryLevel;
    Config.TriggerOnData = TriggerOnData;
    Config.TriggerOnTime = TriggerOnTime;
    Config.TriggerOnAlarm = TriggerOnAlarm;
    Config.TriggerOnCollision = TriggerOnCollision;
    Config.TriggerOnOversize = TriggerOnOversize;
    Config.XlateAddresses = XlateAddresses;
    Config.SpeakerOn = SpeakerOn;
    Config.StopOnBufferFull = StopOnBufferFull;
    strcpy (Config.CaptureDirectory, CaptureDirectory);
    strcpy (Config.NameDirectory, NameDirectory);
    strcpy (Config.Printer, Printer);
    strcpy (Config.LogFileName, LogFileName);
    strcpy (Config.LicensedTo, LicensedTo);

    fwrite (&Config, sizeof (CONFIG_RECORD), 1, File);

    //
    // Now write-out the alarm statuses, too.
    //

#ifdef ALARMS
    GetAlarmTableSize (&AlarmRecordSize, &NumberOfAlarms);

    for (i=0; i<NumberOfAlarms; i++) {
	ReadAlarmEntry (ScratchBuf, i);
	fwrite (ScratchBuf, AlarmRecordSize, 1, File);
    }
#endif

    fclose (File);
} // SaveConfiguration

VOID DisplayOwner ()
{
    PWINDOW Window;
    time_t time1, time2;
    UCHAR *p="Copyright (C) 1992-1993 General Software, Inc.";
    UCHAR *q="All rights reserved.";

    sprintf (ScratchBuf, "%s Ethernet Network Analyzer V%u.%u %s",
	     PRODUCT_NAME,		// as defined in MAKEFILE.
	     PRODUCT_MAJVER,		// as defined in MAKEFILE.
	     PRODUCT_MINVER,		// as defined in MAKEFILE.
#ifdef DEMO				// as defined in MAKEFILE.
	     "Demo");
#else
	     "");
#endif

    Window = WinCreate (ScratchBuf, 40, 12, 60, 10,
			(WINDOW_BORDER_SINGLE | WINDOW_BORDER_NOHILITE),
			HELP_PALETTE);
    if (Window == NULL) {
	UtilEnd ();
	exit (1);			// we need memory to do this.
    }

    WinWrite (Window, LicensedTo, (60/2)-(strlen (LicensedTo)/2), 1, 60);
    WinWrite (Window, p, (60/2)-(strlen (p)/2), 3, 60);
    WinWrite (Window, q, (60/2)-(strlen (q)/2), 4, 60);

    //
    // Delay a bit to read copyright & registration.
    //

    time (&time1);
    while (TRUE) {
	time (&time2);
	if (time2-time1 >= BANNER_SECONDS) {
	    break;
	}
    }

    WinDestroy (Window);
} // DisplayOwner

VOID RegisterOwner ()
{
    PSCREEN Screen;
    PFIELD f_name;
    ACTION a;

    Screen = ScreenCreate ("Enter Your Name and press F1", 40, 12, 47, 5);
    if (Screen == NULL) {
	UtilEnd ();
	return;
    }

    f_name = FieldCreate ("Your Name: ", FIELD_TYPE_STRING,
			  1, 0, 44, FIELD_FLAGS_EDITABLE);
    strcpy (f_name->Data.String, "");
    ScreenAddField (Screen, f_name);

    if (ScreenEdit (Screen) == ACTION_SAVE) {
	if (strlen (f_name->Data.String) < 3) {
	    PopupWarning ("You must specify your full name when registering.");
	    ScreenDestroy (Screen);
	    UtilEnd ();
	    exit (3);
	}
	sprintf (LicensedTo, "This copy licensed to %s", f_name->Data.String);
	SaveConfiguration ();
    } else {
	PopupWarning ("You must register this product before using it.");
	ScreenDestroy (Screen);
	UtilEnd ();
	exit (2);
    }
    ScreenDestroy (Screen);
} // RegisterOwner

VOID LoadConfiguration ()
{
    FILE *File;
    CONFIG_RECORD Config;

#ifdef ALARMS
    USHORT AlarmRecordSize, NumberOfAlarms, i;
#endif

    strcpy (CaptureDirectory, ".");
    strcpy (NameDirectory, ".");
    strcpy (Printer, "PRN");
    strcpy (LogFileName, LOG_FILE_NAME);

#ifdef DEMO
    strcpy (LicensedTo, "Your demo is limited to 5 seconds of continuous capture.");
#else
    strcpy (LicensedTo, "Unlicensed Commercial Copy");
#endif

    if ((File=fopen (CONFIG_FILE_NAME, "rb")) == NULL) {
#ifdef DEMO
	DisplayOwner ();
#else
	RegisterOwner ();
#endif
	return;
    }
    if (fread (&Config, sizeof (CONFIG_RECORD), 1, File) != 1) {
	fclose (File);
	return;
    }

    //
    // Now write-out the alarm statuses, too.
    //

#ifdef ALARMS
    GetAlarmTableSize (&AlarmRecordSize, &NumberOfAlarms);

    for (i=0; i<NumberOfAlarms; i++) {
	fread (ScratchBuf, AlarmRecordSize, 1, File);
	WriteAlarmEntry (ScratchBuf, i);
    }
#endif

    fclose (File);

    DisplayFilter = Config.DisplayFilter;
    CaptureFilter = Config.CaptureFilter;
    HighestLevelOnly = Config.HighestLevelOnly;
    DisplayAbsoluteTimes = Config.DisplayAbsoluteTimes;
    DisplayHexData = Config.DisplayHexData;
    DisplayProtocolDetail = Config.DisplayProtocolDetail;
    DisplaySummaryLevel = Config.DisplaySummaryLevel;
    TriggerOnData = Config.TriggerOnData;
    TriggerOnTime = Config.TriggerOnTime;
    TriggerOnAlarm = Config.TriggerOnAlarm;
    TriggerOnCollision = Config.TriggerOnCollision;
    TriggerOnOversize = Config.TriggerOnOversize;
    XlateAddresses = Config.XlateAddresses;
    SpeakerOn = Config.SpeakerOn;
    StopOnBufferFull = Config.StopOnBufferFull;
    strcpy (CaptureDirectory, Config.CaptureDirectory);
    strcpy (NameDirectory, Config.NameDirectory);
    strcpy (Printer, Config.Printer);
    strcpy (LogFileName, Config.LogFileName);
    strcpy (LicensedTo, Config.LicensedTo);

    DisplayOwner ();
} // LoadConfiguration

VOID DisplayOptions ()
{
    PSCREEN Screen;
    PFIELD f_hlo, f_dat, f_dhd, f_dpd, f_dsl, f_so, f_mps, f_sbf, f_cdir;
    PFIELD f_ndir, f_prn, f_trd, f_trt, f_tra, f_lfn, f_xla, f_trc, f_tro;
    USHORT OldPacketSize, lineno, nlines;
    ACTION a;

#ifdef TRIGGERS
    nlines = 15;
#else
    nlines = 14;
#endif

    Screen = ScreenCreate ("Current Options  [Help=F2]", 40, 12, 80, nlines);
    if (Screen == NULL) {
	return;
    }

    f_dsl = FieldCreate ("Display Summary Level:       ", FIELD_TYPE_BOOLEAN,
			 1, 0, 32, FIELD_FLAGS_EDITABLE);
    f_dsl->Data.Boolean = DisplaySummaryLevel;
    ScreenAddField (Screen, f_dsl);

    f_dhd = FieldCreate ("Display Hex Data:            ", FIELD_TYPE_BOOLEAN,
			 1, 1, 32, FIELD_FLAGS_EDITABLE);
    f_dhd->Data.Boolean = DisplayHexData;
    ScreenAddField (Screen, f_dhd);

    f_dpd = FieldCreate ("Display Protocol Detail:     ", FIELD_TYPE_BOOLEAN,
			 1, 2, 32, FIELD_FLAGS_EDITABLE);
    f_dpd->Data.Boolean = DisplayProtocolDetail;
    ScreenAddField (Screen, f_dpd);

    f_hlo = FieldCreate ("Display Highest Level Only:  ", FIELD_TYPE_BOOLEAN,
			 1, 3, 32, FIELD_FLAGS_EDITABLE);
    f_hlo->Data.Boolean = HighestLevelOnly;
    ScreenAddField (Screen, f_hlo);

    f_dat = FieldCreate ("Display Absolute Times:      ", FIELD_TYPE_BOOLEAN,
			 1, 4, 32, FIELD_FLAGS_EDITABLE);
    f_dat->Data.Boolean = DisplayAbsoluteTimes;
    ScreenAddField (Screen, f_dat);

    f_so =  FieldCreate ("Speaker Clicks Enabled:      ", FIELD_TYPE_BOOLEAN,
			 1, 5, 32, FIELD_FLAGS_EDITABLE);
    f_so->Data.Boolean = SpeakerOn;
    ScreenAddField (Screen, f_so);

    f_mps = FieldCreate ("Maximum Packet Size:         ", FIELD_TYPE_ULONG,
			 40, 0, 33, FIELD_FLAGS_EDITABLE);
    f_mps->Data.UnsignedLong = (ULONG)MaxPacketSize;
    ScreenAddField (Screen, f_mps);
    OldPacketSize = MaxPacketSize;

    f_sbf = FieldCreate ("Stop on Capture Buffer Full: ", FIELD_TYPE_BOOLEAN,
			 40, 1, 32, FIELD_FLAGS_EDITABLE);
    f_sbf->Data.Boolean = StopOnBufferFull;
    ScreenAddField (Screen, f_sbf);

#ifdef TRIGGERS
    f_trd = FieldCreate ("Trigger on Data Match:       ", FIELD_TYPE_BOOLEAN,
			 40, 2, 32, FIELD_FLAGS_EDITABLE);
    f_trd->Data.Boolean = TriggerOnData;
    ScreenAddField (Screen, f_trd);

    f_trt = FieldCreate ("Trigger on Time of Day:      ", FIELD_TYPE_BOOLEAN,
			 40, 3, 32, FIELD_FLAGS_EDITABLE);
    f_trt->Data.Boolean = TriggerOnTime;
    ScreenAddField (Screen, f_trt);

    f_tra = FieldCreate ("Trigger on Alarms:           ", FIELD_TYPE_BOOLEAN,
			 40, 4, 32, FIELD_FLAGS_EDITABLE);
    f_tra->Data.Boolean = TriggerOnAlarm;
    ScreenAddField (Screen, f_tra);

    f_trc = FieldCreate ("Trigger on Collisions:       ", FIELD_TYPE_BOOLEAN,
			 40, 5, 32, FIELD_FLAGS_EDITABLE);
    f_trc->Data.Boolean = TriggerOnCollision;
    ScreenAddField (Screen, f_trc);

    f_tro = FieldCreate ("Trigger on Oversize Frame:   ", FIELD_TYPE_BOOLEAN,
			 40, 6, 32, FIELD_FLAGS_EDITABLE);
    f_tro->Data.Boolean = TriggerOnOversize;
    ScreenAddField (Screen, f_tro);
#endif

#ifdef TRIGGERS
    lineno = 7;
#else
    lineno = 2;
#endif
    f_xla = FieldCreate ("Translate Ethernet Addresses:", FIELD_TYPE_BOOLEAN,
			 40, lineno, 32, FIELD_FLAGS_EDITABLE);
    f_xla->Data.Boolean = XlateAddresses;
    ScreenAddField (Screen, f_xla);

    f_prn = FieldCreate ("Default Printer Name:        ", FIELD_TYPE_STRING,
			  1, 7, 38, FIELD_FLAGS_EDITABLE);
    strcpy (f_prn->Data.String, Printer);
    ScreenAddField (Screen, f_prn);

    f_cdir = FieldCreate ("Capture File Directory:     ", FIELD_TYPE_STRING,
			  1, 8, 77, FIELD_FLAGS_EDITABLE);
    strcpy (f_cdir->Data.String, CaptureDirectory);
    ScreenAddField (Screen, f_cdir);

    f_ndir = FieldCreate ("Name Management Directory:  ", FIELD_TYPE_STRING,
			  1, 9, 77, FIELD_FLAGS_EDITABLE);
    strcpy (f_ndir->Data.String, NameDirectory);
    ScreenAddField (Screen, f_ndir);

#ifdef ALARMS
    f_lfn = FieldCreate ("Event Log File Name:        ", FIELD_TYPE_STRING,
			  1, 10, 77, FIELD_FLAGS_EDITABLE);
    strcpy (f_lfn->Data.String, LogFileName);
    ScreenAddField (Screen, f_lfn);
#endif

    a = ScreenEdit (Screen);

    if (a == ACTION_SAVE) {
	HighestLevelOnly = f_hlo->Data.Boolean;
	DisplayAbsoluteTimes = f_dat->Data.Boolean;
	DisplayHexData = f_dhd->Data.Boolean;
	DisplayProtocolDetail = f_dpd->Data.Boolean;
	DisplaySummaryLevel = f_dsl->Data.Boolean;
	XlateAddresses = f_xla->Data.Boolean;
	SpeakerOn = f_so->Data.Boolean;
	if (f_mps->Data.UnsignedLong < 16384L) {
	    MaxPacketSize = (USHORT)f_mps->Data.UnsignedLong;
	}
	StopOnBufferFull = f_sbf->Data.Boolean;
	strcpy (CaptureDirectory, f_cdir->Data.String);
	strcpy (NameDirectory, f_ndir->Data.String);
	strcpy (Printer, f_prn->Data.String);

#ifdef TRIGGERS
	TriggerOnData = f_trd->Data.Boolean;
	TriggerOnTime = f_trt->Data.Boolean;
	TriggerOnAlarm = f_tra->Data.Boolean;
#endif

#ifdef ALARMS
	strcpy (LogFileName, f_lfn->Data.String);
#endif

	if (PopupYesOrNo ("Configuration Updated.", "Make Configuration Permanent?")) {
	    SaveConfiguration ();
	}
    }
    ScreenDestroy (Screen);
} // DisplayOptions
