//
// MODULE NAME:  ARP.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for ARP,
//	the Address Resolution Protocol used with the TCP/IP suite.
//
//	There are two main entrypoints to this module.	PacketSummaryArp
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailArp returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	93/01/03.	#2.0, original.
//
// NOTICE:  Copyright (C) 1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

//
// ARP request packet (Ethertype 0x0803).
//

typedef struct _ARP {
    USHORT Hardware;			// 1 = Ethernet.
    USHORT Protocol;
    UCHAR Hlen;
    UCHAR Plen;
    USHORT Operation;
    UCHAR SenderHa [6];                 // host address (physical).
    UCHAR SenderIa [4];                 // internet address.
    UCHAR TargetHa [6];                 // host address (physical).
    UCHAR TargetIa [4];                 // internet address.
} ARP, *PARP;

//
// Routines in other modules.
//

extern PUCHAR IpAddress (UCHAR *InternetAddress);	// in IP.C.
extern UCHAR *MakeHexString (UCHAR *Address);		// in ANALYZER.C.
extern PUCHAR ResolveDlcAddress (PUCHAR DestBuf, PUCHAR SrcBuf); // in FORMAT.C.

//
// Routines in this module.
//

BOOLEAN PacketReqTypeArp (PARP Buffer, USHORT BufferLength)
{
    return TRUE;
} // PacketReqTypeArp

BOOLEAN PacketFilterArp (PARP Buffer, USHORT BufferLength)
{
    if (CaptureFilter & FILTER_ARP) {
	return TRUE;			// we're accepting ARP.
    }
    return FALSE;			// don't accept this.
} // PacketFilterArp

UCHAR *ArpCommandName (USHORT CommandCode)
{
    switch (CommandCode) {
	case 1: return "ARP Request";
	case 2: return "ARP Response";
	case 3: return "RARP Request";
	case 4: return "RARP Response";
	default: return "Undecoded ARP Packet";
    }
} // ArpCommandName

PUCHAR PacketSummaryArp (PARP Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;

    if (!(DisplayFilter & FILTER_ARP)) {
	return NULL;			// we aren't decoding ARP.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    sprintf (p, "%s", ArpCommandName (Swap (Buffer->Operation)));
    return p;
} // PacketSummaryArp

PDETREC PacketDetailArp (PARP Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;
    USHORT i, cmd, tmp;
    UCHAR *z;

    cmd = Swap (Buffer->Operation);
    z = "ARP";
    if ((cmd == 3) || (cmd == 4)) {
	z = "RARP";
    }

    sprintf (ScratchBuf, "%s:  ----- Address Resolution Protocol header -----", z);
    AppendHeader (&r, ScratchBuf);

    tmp = Swap (Buffer->Hardware);
    sprintf (ScratchBuf, "%s:  Hardware = %u (%s)",
	     z, tmp, (tmp == 1) ? "Ethernet" : "Undecoded");
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "%s:  Operation code = %u (%s)", z,
	     cmd, ArpCommandName (cmd));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "%s:  Protocol = %u", z, Swap (Buffer->Protocol));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "%s:  Hardware address length = %u", z, (USHORT)Buffer->Hlen);
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "%s:  Protocol address length = %u", z, (USHORT)Buffer->Plen);
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "%s:  Sender Hardware Address = ", z);
    for (i=0; i<Buffer->Hlen; i++) {
	sprintf (ScratchBuf1, "%02x", Buffer->SenderHa [i]);
	strcat (ScratchBuf, ScratchBuf1);
    }
    strcat (ScratchBuf, " (");
    ResolveDlcAddress (ScratchBuf1, Buffer->SenderHa);
    strcat (ScratchBuf, ScratchBuf1);
    strcat (ScratchBuf, ")");
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "%s:  Sender Internet Address = %s",
	     z, IpAddress (Buffer->SenderIa));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "%s:  Target Hardware Address = [", z);
    for (i=0; i<Buffer->Hlen; i++) {
	sprintf (ScratchBuf1, "%02x.", Buffer->TargetHa [i]);
	strcat (ScratchBuf, ScratchBuf1);
    }
    strcat (ScratchBuf, " (");
    ResolveDlcAddress (ScratchBuf1, Buffer->TargetHa);
    strcat (ScratchBuf, ScratchBuf1);
    strcat (ScratchBuf, ")");
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "%s:  Target Internet Address = %s",
	     z, IpAddress (Buffer->TargetIa));
    AppendDetRec (&r, ScratchBuf);
    return r;
} // PacketDetailArp
