//***	ANALYZER.H -- Constants and Structures for Network Analyzers.
//
//1.	Functional Description.
//	This include file contains constants and structure definitions
//	for General Software's Network Analyzers.
//
//2.	Modification History.
//	S. E. Jones	92/03/18.	Original.
//	S. E. Jones	92/11/24.	#1.8, added LLCTDI.
//	S. E. Jones	92/12/29.	#2.0, next major revision.
//	S. E. Jones	93/02/05.	#2.2, added GSFSP.
//	S. E. Jones	93/03/16.	#2.5, added GSIPC.
//
//3.	NOTICE: Copyright (C) 1992-1993 General Software, Inc.

#ifndef _ANALYZER
#define _ANALYZER

//
// Product definitions.
//

#ifdef ETHERPROBE
#define PRODUCT_NAME	"EtherProbe"
#define TRIGGERS			// support triggers in Etherprobe.
#define ALARMS				// support alarms in Etherprobe.
#define GENERATE_TRAFFIC		// support traffic generation.
// #define NODE_PAIR_CAPTURE		   // support node pair capture.
// #define SKYLINE_CAPTURE		   // support skyline capture.
// #define HISTOGRAM_CAPTURE		   // support histogram capture.
// #define FILESHARINGERROR_CAPTURE	   // support file sharing error capture.
#endif

#ifdef SNOOPER
#define PRODUCT_NAME	"The Snooper"
#endif

//
// Data structures.
//

typedef struct _SNIFHDR {
    USHORT Type;			// type of this record.
#define SNIFHDR_TYPE_VERSION	1	// version record.
#define SNIFHDR_TYPE_DATA	4	// frame data.
#define SNIFHDR_TYPE_EOF	3	// end of file (no data in this record).

    USHORT Length;			// length of remainder of this record.
    USHORT Reserved;			// reserved, MBZ.
} SNIFHDR, *PSNIFHDR;

typedef struct _SNIFVER {
    USHORT MajVer;			// major version of "sniffer".
    USHORT MinVer;			// minor version of "sniffer".
    USHORT Date;			// date and time of this file.
    USHORT Time;
    UCHAR Type;                         // type of records that follow.
    UCHAR Network;			// network type.
    UCHAR Timeunit;			// frame timestamp units.
    USHORT Reserved [3];		// 3 reserved, MBZ, words.
    UCHAR ReservedByte;                 // another reserved byte.
} SNIFVER, *PSNIFVER;

typedef struct _SNIFDATA {
    USHORT LowTime;			// low time in 0.838096 microsecond units.
    USHORT MidTime;			// mid time in 54.9255 millisecond units.
    USHORT HighTime;			// high time in hours.
    USHORT Size;			// bytes actually recorded in this record.
    UCHAR FrameStatusBits;		// status of this frame (bitflags).
    UCHAR Flags;			// flags, for internal use.
    USHORT TrueSize;			// frame's true size (if truncated).
    USHORT Reserved;			// reserved, MBZ.
} SNIFDATA, *PSNIFDATA;

typedef struct _ADDRESS_FILTER {
    struct _ADDRESS_FILTER *Fwdlink;	// next filter in chain.
    USHORT Flags;			// filter qualifications, as follows.
#define ADDRESS_FILTER_SOURCE	0x0001	// match source address.
#define ADDRESS_FILTER_DEST	0x0002	// match dest address.
#define ADDRESS_FILTER_LINK	0x0004	// match both addr's (even reversed).

    UCHAR SrcAddress [6];		// source address to match.
    UCHAR DestAddress [6];		// dest address to match.
} ADDRESS_FILTER, *PADDRESS_FILTER;

typedef struct _CAPREC {
    struct _CAPREC *Fwdlink;		// next capture record in chain.
    struct _CAPREC *Baklink;		// previous capture record in chain.

    USHORT SerialNo;			// packet serial number.
    ULONG  Timestamp;			// capture time in 100us since start.
    USHORT Flags;			// capture flags.
#define CAPREC_FLAGS_TRUNCATED	0x0001	// packet was truncated to fit in buffer.
#define CAPREC_FLAGS_CAPTURED	0x0002	// this CAPREC contains a packet.
#define CAPREC_FLAGS_MALFORMED	0x0004	// this packet was malformed.
#define CAPREC_FLAGS_DISPLAY	0x0008	// packet selected by filter.

    USHORT Length;			// length of captured data.
    UCHAR Buffer [1];			// buffer (extends beyond struct end).
} CAPREC, *PCAPREC;

typedef struct _BIGBUF {
    struct _BIGBUF *Fwdlink;		// ptr to next bigbuf in chain.
    USHORT BytesRemaining;		// free bytes left/this buffer.
    USHORT BigBufSize;			// total size, this bigbuf.
    UCHAR *NextFill;			// ptr to 1st free byte/this buffer.
    UCHAR CaptureRecords [1];		// just a place holder.
} BIGBUF, *PBIGBUF;

typedef struct _STATISTICS_BUFFER {
    ULONG DroppedPackets;		// # packets dropped by software.
} STATISTICS_BUFFER, *PSTATISTICS_BUFFER;

typedef struct _DETREC {
    struct _DETREC *Fwdlink;		// ptr to next DETREC in chain.
    UCHAR DescBuf [1];			// variable length string.
} DETREC, *PDETREC;

typedef struct _DISPLAY_WINDOW {
    PWINDOW Win;			// pointer to display window.
    USHORT WindowLines;                 // number of display lines in window.
    USHORT CurrLine;			// highlighted line (0=top).
    USHORT Type;			// type of window display.
#define DISPLAY_TYPE_SUMMARY	0
#define DISPLAY_TYPE_DETAIL	1
#define DISPLAY_TYPE_HEX	2

    PDETREC Head;			// ptr to head of ASCII text records.
    PDETREC Top;			// ptr to 1st record displayed on 1st line.
} DISPLAY_WINDOW, *PDISPLAY_WINDOW;

//
// Configuration file record format.
//

typedef struct _CONFIG_RECORD {
    ULONG DisplayFilter;
    ULONG CaptureFilter;
    BOOLEAN HighestLevelOnly;
    BOOLEAN DisplayAbsoluteTimes;
    BOOLEAN DisplayHexData;
    BOOLEAN DisplayProtocolDetail;
    BOOLEAN DisplaySummaryLevel;
    BOOLEAN SpeakerOn;
    BOOLEAN StopOnBufferFull;
    BOOLEAN TriggerOnData;
    BOOLEAN TriggerOnTime;
    BOOLEAN TriggerOnAlarm;
    BOOLEAN TriggerOnCollision;
    BOOLEAN TriggerOnOversize;
    BOOLEAN XlateAddresses;
    UCHAR CaptureDirectory [64];
    UCHAR NameDirectory [64];
    UCHAR Printer [64];
    UCHAR LogFileName [64];
    UCHAR LicensedTo [64];
} CONFIG_RECORD, *PCONFIG_RECORD;

//
// Filter bitmasks for various protocol types.
//

#define FILTER_LLC		0x00000001L
#define FILTER_IPX		0x00000002L
#define FILTER_RIP		0x00000004L
#define FILTER_ECHO		0x00000008L
#define FILTER_IPX_ERROR	0x00000010L
#define FILTER_PEP		0x00000020L
#define FILTER_SPX		0x00000040L
#define FILTER_NCP		0x00000080L
#define FILTER_NETBEUI		0x00000100L
#define FILTER_SMB		0x00000200L
#define FILTER_IP		0x00000400L
#define FILTER_TCP		0x00000800L
#define FILTER_UDP		0x00001000L
#define FILTER_SAP_SQP		0x00002000L
#define FILTER_DECNET		0x00004000L
#define FILTER_SNA		0x00008000L
#define FILTER_ISO		0x00010000L
#define FILTER_LLCTDI		0x00020000L
#define FILTER_ARP		0x00040000L
#define FILTER_GSFSP		0x00080000L
#define FILTER_GSIPC		0x00100000L
#define FILTER_OTHER		0x80000000L

#define MAX_FILTERS		22

extern PDETREC PacketDetailIpx (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailNcp (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailSpx (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailLlc (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailIp (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailSmb (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailDecnet (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailNetbeui (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailLlcTdi (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailGsFsp (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailGsIpc (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailEthernet (PVOID Buffer, USHORT BufferLength);
extern PDETREC PacketDetailArp (PVOID Buffer, USHORT BufferLength);

extern PUCHAR PacketSummaryEthernet (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryNcp (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryIpx (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummarySpx (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryLlc (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummarySmb (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryDecnet (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryNetbeui (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryLlcTdi (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryGsFsp (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryGsIpc (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryIp (PVOID Buffer, USHORT BufferLength);
extern PUCHAR PacketSummaryArp (PVOID Buffer, USHORT BufferLength);

extern BOOLEAN PacketFilterEthernet (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterIpx (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterSpx (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterSmb (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterIp (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterDecnet (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterLlc (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterNetbeui (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterLlcTdi (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterGsFsp (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterGsIpc (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketFilterArp (PVOID Buffer, USHORT BufferLength);

extern BOOLEAN PacketCountEthernet (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketCountIpx (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketCountIp (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketCountLlc (PVOID Buffer, USHORT BufferLength);

extern BOOLEAN PacketReqTypeEthernet (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeIpx (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeIp (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeLlc (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeNetbeui (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeSmb (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeNcp (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeArp (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeLlcTdi (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeGsFsp (PVOID Buffer, USHORT BufferLength);
extern BOOLEAN PacketReqTypeGsIpc (PVOID Buffer, USHORT BufferLength);

#define INDEX_LLC	0
#define INDEX_IPX	1
#define INDEX_RIP	2
#define INDEX_ECHO	3
#define INDEX_ERROR	4
#define INDEX_PEP	5
#define INDEX_SPX	6
#define INDEX_NCP	7
#define INDEX_SAP_SQP	8
#define INDEX_NETBEUI	9
#define INDEX_SMB	10
#define INDEX_IP	11
#define INDEX_TCP	12
#define INDEX_UDP	13
#define INDEX_DECNET	14
#define INDEX_SNA	15
#define INDEX_ISO	16
#define INDEX_GSLLCTDI	17
#define INDEX_ARP	18
#define INDEX_GSFSP	19
#define INDEX_GSIPC	20
#define INDEX_OTHER	21

#define FILEIO_MKDIR		0
#define FILEIO_RMDIR		1
#define FILEIO_CREATE		2
#define FILEIO_DELETE		3
#define FILEIO_CREATE_TEMP	4
#define FILEIO_RENAME		5
#define FILEIO_SEARCH		6
#define FILEIO_OPEN		7
#define FILEIO_CLOSE		8
#define FILEIO_READ		9
#define FILEIO_WRITE		10
#define FILEIO_LOCK		11
#define FILEIO_UNLOCK		12
#define FILEIO_LOGIN		13
#define FILEIO_LOGOUT		14
#define FILEIO_CREATE_PROCESS	15
#define FILEIO_DESTROY_PROCESS	16
#define MAX_FILEIO_STATISTICS	17

#define BUFLEN 256
#define BLOCKLEN 80

#define DISPLAYTYPE_GENERAL	0	// Snooper-style bargraph display.
#define DISPLAYTYPE_BY_PROTOCOL 1	// kb/s, pkts/s by protocol.
#define DISPLAYTYPE_BY_ADDRTYPE 2	// kb/s, pkts/s by unicast,mcast,bcst.
#define DISPLAYTYPE_BY_REQTYPE	3	// requests/s by request type.
#define DISPLAYTYPE_MAX_VALUE	3	// last valid display type.

#ifdef MAIN_PROGRAM

UCHAR far TmpStr [BUFLEN];		// formatting string.

UCHAR far SolidBlockBuffer [BLOCKLEN];
UCHAR far LightBlockBuffer [BLOCKLEN];
UCHAR far DarkBlockBuffer [BLOCKLEN];

BOOLEAN CaptureSuspended=FALSE;         // set to TRUE by ALARM.C.

BOOLEAN LoggingEvents=FALSE;		// used in CAPTURE.C, ALARM.C.
FILE *LogFile;				// used in CAPTURE.C, ALARM.C.

BOOLEAN TriggerOnData=FALSE;
BOOLEAN TriggerOnTime=FALSE;
BOOLEAN TriggerOnAlarm=FALSE;
BOOLEAN TriggerOnCollision=FALSE;
BOOLEAN TriggerOnOversize=FALSE;

USHORT CaptureDisplayType=0;		// see DISPLAYTYPEs (above).
PCAPREC CaptureBuffer=NULL;		// queue of captured packet buffers.
PBIGBUF BufferChain=NULL;		// bigbuf buffer chain.
PBIGBUF CurrBuf=NULL;			// ptr to current BIGBUF being used.
PCAPREC LastPacket=NULL;		// last CAPREC saved in capture buffer.
USHORT MaxPacketSize=1536;		// default packet size.
USHORT LargestDlcPacket=1536;		// maximum size of ANY packet/this LAN.
PCAPREC CurrentCapRec;			// current capture record.
PDETREC TopLevelDetRec;                 // ptr to highest level protocol detail.
PDETREC LastHeaderDetRec;		// for DISPLAY.C/FORMAT.C.

BOOLEAN XlateAddresses=TRUE;		// TRUE -> translate 802.3 addresses.
BOOLEAN HighestLevelOnly=TRUE;		// TRUE -> don't display lower layers.
BOOLEAN DisplayAbsoluteTimes=FALSE;	// TRUE -> don't display deltas.
BOOLEAN DisplayHexData=TRUE;		// TRUE -> display the hex dump window.
BOOLEAN DisplayProtocolDetail=TRUE;	// TRUE -> detailed protocol display.
BOOLEAN DisplaySummaryLevel=TRUE;	// TRUE -> summary display.
BOOLEAN SpeakerOn=TRUE;                 // TRUE -> mark incoming pkt w/audio.
BOOLEAN StopOnBufferFull=FALSE;         // TRUE -> stop capture when buf/full.
PADDRESS_FILTER AddressFilterList=NULL; // no filters to begin with.

BOOLEAN CaptureBufferFull=FALSE;	// set to TRUE to stop capture.
USHORT CurrentSerialNo;                 // master serial number.
BOOLEAN ReusedBuffer;			// set to TRUE when buffer is reused.

ULONG TempFilter;			// used by filter management stuff.
ULONG ReceivedPackets;
ULONG UnmatchedPackets;
ULONG ReceivedPacketsPerSecond;
ULONG UndersizedPackets;
ULONG OversizedPackets;
ULONG UnicastPackets;
ULONG MulticastPackets;
ULONG BroadcastPackets;
ULONG UnicastKb;
ULONG MulticastKb;
ULONG BroadcastKb;
ULONG KbPerSecond;
USHORT ByteCount;
STATISTICS_BUFFER StatBuffer;
ULONG PeakReceivedPacketsPerSecond;
ULONG PeakKbPerSecond;
ULONG PeakUndersizedPackets;
ULONG PeakOversizedPackets;
ULONG PeakDroppedPackets;
ULONG PeakUnicastPackets;
ULONG PeakMulticastPackets;
ULONG PeakBroadcastPackets;
ULONG PeakReceivedPackets;
ULONG PeakUnicastKb;
ULONG PeakMulticastKb;
ULONG PeakBroadcastKb;

UCHAR BcstAddr [6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

SNIFDATA far SnifDataRec;		// buffer used to hold data record headers.
SNIFVER far SnifVersion;		// buffer used to hold version record.
SNIFHDR far SnifHeader;                 // buffer used to hold record header.

UCHAR far CaptureDirectory [64];	// directory where capture files are stored.
UCHAR far NameDirectory [64];		// directory where name files are stored.
UCHAR far Printer [64];                 // name of printer.
UCHAR far LogFileName [64];		// name of log file.
UCHAR far LicensedTo [64];		// who this thing is licensed to.

UCHAR far ScratchBuf [128];		// scratch buffer for formatting.
UCHAR far ScratchBuf1 [128];		// secondary formatting buffer.

ULONG ProtocolTypes [MAX_FILTERS];	// packet count by protocol.
ULONG ProtocolTypesPerSec [MAX_FILTERS]; // packets per second by protocol.
ULONG FileIoTypes [MAX_FILEIO_STATISTICS];
ULONG FileIoTypesPerSec [MAX_FILEIO_STATISTICS];
ULONG PeakProtocolTypesPerSec [MAX_FILTERS];
ULONG PeakFileIoTypesPerSec [MAX_FILEIO_STATISTICS];

UCHAR * far FileIoName [MAX_FILEIO_STATISTICS] = {
    "MKDIR", "RMDIR", "CREATE", "DELETE",
    "CRETMP", "RENAME", "SEARCH", "OPEN",
    "CLOSE", "READ", "WRITE", "LOCK",
    "UNLOCK", "LOGIN", "LOGOUT", "CREPROC",
    "DELPROC",
}; // FileIoName

ULONG DisplayFilter=(0xffffffffL & ~FILTER_OTHER); // all filters enabled.
ULONG CaptureFilter=(0xffffffffL & ~FILTER_OTHER); // all filters enabled.

UCHAR * far FilterName [MAX_FILTERS] = {
    "LLC (802.2)",	"NetWare IPX",	"IPX RIP",	"IPX ECHO",
    "IPX ERROR",	"NetWare PEP",	"NetWare SPX",	"NetWare NCP",
    "NetWare SAP/SQP",	"IBM NETBEUI",	"SMB LanMan",	"IP Datagrams",
    "TCP",		"UDP",		"DECNET",	"IBM SNA",
    "ISO Family",	"GS LLCTDI",	"ARP",		"GS FSP",
    "GS IPC",		"All Others"
}; // FilterName

ULONG FilterMask [MAX_FILTERS] = {
    FILTER_LLC,         FILTER_IPX,	FILTER_RIP,	FILTER_ECHO,
    FILTER_IPX_ERROR,	FILTER_PEP,	FILTER_SPX,	FILTER_NCP,
    FILTER_SAP_SQP,	FILTER_NETBEUI, FILTER_SMB,	FILTER_IP,
    FILTER_TCP,         FILTER_UDP,	FILTER_DECNET,	FILTER_SNA,
    FILTER_ISO,         FILTER_LLCTDI,	FILTER_ARP,	FILTER_GSFSP,
    FILTER_GSIPC,	FILTER_OTHER
}; // FilterMask

#else

extern UCHAR TmpStr [BUFLEN];		// formatting string.
extern UCHAR SolidBlockBuffer [BLOCKLEN];
extern UCHAR LightBlockBuffer [BLOCKLEN];
extern UCHAR DarkBlockBuffer [BLOCKLEN];

extern BOOLEAN CaptureSuspended;	// set to TRUE by ALARM.C.
extern BOOLEAN LoggingEvents;		// used in CAPTURE.C, ALARM.C.
extern FILE *LogFile;			// used in CAPTURE.C, ALARM.C.

extern BOOLEAN TriggerOnData;
extern BOOLEAN TriggerOnTime;
extern BOOLEAN TriggerOnAlarm;
extern BOOLEAN TriggerOnCollision;
extern BOOLEAN TriggerOnOversize;

extern USHORT CaptureDisplayType;	// see DISPLAYTYPEs (earlier).
extern PCAPREC CaptureBuffer;		// queue of captured packet buffers.
extern PBIGBUF BufferChain;		// bigbuf buffer chain.
extern PBIGBUF CurrBuf;                 // ptr to current BIGBUF being used.
extern PCAPREC LastPacket;		// last CAPREC saved in capture buffer.
extern USHORT MaxPacketSize;		// default packet size.
extern USHORT LargestDlcPacket;         // maximum size of ANY packet/this LAN.
extern PCAPREC CurrentCapRec;		// current capture record.
extern PDETREC TopLevelDetRec;		// ptr to highest level protocol detail.
extern PDETREC LastHeaderDetRec;	// used by DISPLAY.C only.

extern BOOLEAN XlateAddresses;		// TRUE -> translate 802.3 addresses.
extern BOOLEAN HighestLevelOnly;	// TRUE -> don't display lower layers.
extern BOOLEAN DisplayAbsoluteTimes;	// TRUE -> don't display deltas.
extern BOOLEAN DisplayHexData;		// TRUE -> display the hex dump window.
extern BOOLEAN DisplayProtocolDetail;	// TRUE -> detailed protocol display.
extern BOOLEAN DisplaySummaryLevel;	// TRUE -> summary display.
extern BOOLEAN SpeakerOn;		// TRUE -> mark incoming pkt w/audio.
extern BOOLEAN StopOnBufferFull;	// TRUE -> stop capture when buf/full.

extern PADDRESS_FILTER AddressFilterList; // no filters to begin with.

extern BOOLEAN CaptureBufferFull;	// set to TRUE to stop capture.
extern USHORT CurrentSerialNo;		// master serial number.
extern BOOLEAN ReusedBuffer;		// set to TRUE when buffer is reused.

extern ULONG TempFilter;		// used by filter management stuff.
extern ULONG ReceivedPackets;
extern ULONG UnmatchedPackets;
extern ULONG ReceivedPacketsPerSecond;
extern ULONG UndersizedPackets;
extern ULONG OversizedPackets;
extern ULONG UnicastPackets;
extern ULONG MulticastPackets;
extern ULONG BroadcastPackets;
extern ULONG UnicastKb;
extern ULONG MulticastKb;
extern ULONG BroadcastKb;
extern ULONG KbPerSecond;
extern USHORT ByteCount;
extern STATISTICS_BUFFER StatBuffer;
extern ULONG PeakReceivedPacketsPerSecond;
extern ULONG PeakKbPerSecond;
extern ULONG PeakUndersizedPackets;
extern ULONG PeakOversizedPackets;
extern ULONG PeakDroppedPackets;
extern ULONG PeakUnicastPackets;
extern ULONG PeakMulticastPackets;
extern ULONG PeakBroadcastPackets;
extern ULONG PeakReceivedPackets;
extern ULONG PeakUnicastKb;
extern ULONG PeakMulticastKb;
extern ULONG PeakBroadcastKb;

extern UCHAR BcstAddr [6];

extern SNIFDATA SnifDataRec;		// buffer used to hold data record headers.
extern SNIFVER SnifVersion;		// buffer used to hold version record.
extern SNIFHDR SnifHeader;		// buffer used to hold record header.

extern UCHAR CaptureDirectory [64];	// directory where capture files are stored.
extern UCHAR NameDirectory [64];	// directory where name files are stored.
extern UCHAR Printer [64];		// name of printer.
extern UCHAR LogFileName [64];		// name of log file.
extern UCHAR LicensedTo [64];		// who this thing is licensed to.

extern UCHAR ScratchBuf [128];		// scratch buffer for formatting.
extern UCHAR ScratchBuf1 [128];         // secondary formatting buffer.

extern ULONG ProtocolTypes [MAX_FILTERS];      // packet count by protocol.
extern ULONG ProtocolTypesPerSec [MAX_FILTERS]; // packets per second by protocol.
extern ULONG FileIoTypes [MAX_FILEIO_STATISTICS];
extern ULONG FileIoTypesPerSec [MAX_FILEIO_STATISTICS];
extern ULONG PeakProtocolTypesPerSec [MAX_FILTERS];
extern ULONG PeakFileIoTypesPerSec [MAX_FILEIO_STATISTICS];
extern UCHAR *FileIoName [MAX_FILEIO_STATISTICS];

extern UCHAR *FilterName [MAX_FILTERS];
extern ULONG FilterMask [MAX_FILTERS];

extern ULONG DisplayFilter;		// display filtering (DISPLAY ONLY).
extern ULONG CaptureFilter;		// capture filtering (CAPTURE ONLY).

extern VOID CountProtocolType (USHORT ProtocolIndex); // in MAIN.
extern VOID CountFileIoType (USHORT FileIoIndex); // in MAIN.
#endif

#ifndef FORMAT_MODULE
extern VOID MakeAsciiz (PUCHAR Dest, PUCHAR Src, USHORT Len);
extern ULONG SwapDword (ULONG Src);
extern PDETREC AllocateDetRec (PUCHAR Buffer);
extern BOOLEAN AppendDetRec (PDETREC *Ptr, PUCHAR Buffer);
extern VOID AppendBlankDetRec (PDETREC *Ptr);
extern VOID AppendHeader (PDETREC *Ptr, PUCHAR Buffer);
extern VOID JoinDetRec (PDETREC *Ptr, PDETREC Tail);
extern PUCHAR ResolveDlcAddress (PUCHAR DestBuf, PUCHAR SrcBuf);
extern VOID DeallocateDetRecChain (PDISPLAY_WINDOW Win);
extern USHORT Swap (USHORT Value);
extern USHORT NibbleToBinary (UCHAR Value);

extern VOID Append8BitMask (PDETREC *Ptr, UCHAR *Prefix, USHORT Value,
			    UCHAR *FieldName, UCHAR *B0Name, UCHAR *B1Name,
			    UCHAR *B2Name, UCHAR *B3Name, UCHAR *B4Name,
			    UCHAR *B5Name, UCHAR *B6Name, UCHAR *B7Name);

extern VOID Append16BitMask (PDETREC *Ptr, UCHAR *Prefix, USHORT Value,
			     UCHAR *FieldName, UCHAR *B0Name, UCHAR *B1Name,
			     UCHAR *B2Name, UCHAR *B3Name, UCHAR *B4Name,
			     UCHAR *B5Name, UCHAR *B6Name, UCHAR *B7Name,
			     UCHAR *B8Name, UCHAR *B9Name, UCHAR *BaName,
			     UCHAR *BbName, UCHAR *BcName, UCHAR *BdName,
			     UCHAR *BeName, UCHAR *BfName);
#endif

//
// Assembly language routines.
//

typedef (*EVENT_ROUTINE)(VOID);

extern VOID tick ();			// in SNIFHELP.ASM.
extern ULONG  readtime ();		// in SNIFHELP.ASM.
extern USHORT PdSetEventHandler (USHORT EventType, EVENT_ROUTINE ServiceRtn);
extern USHORT PdResetStatistics ();
extern USHORT PdGetStatistics (PSTATISTICS_BUFFER Buffer, USHORT BufferLength);
extern USHORT PdOpen (USHORT MacId, USHORT *Handle);
extern USHORT PdClose (USHORT Handle);
extern USHORT PdSend (USHORT Handle, PVOID Buffer, USHORT BufferLength);
extern USHORT PdGetAddress (USHORT Handle, PVOID Buffer, USHORT BufferLength);
extern USHORT PdSetRcvMode (USHORT Handle, USHORT Mode);
#define SETRCVMODE_RCVR_OFF	1
#define SETRCVMODE_DIRECTED	2
#define SETRCVMODE_BROADCAST	3	// directed & bcst.
#define SETRCVMODE_LIM_MCST	4	// directed & limited multicast.
#define SETRCVMODE_MULTICAST	5	// directed & all multicast.
#define SETRCVMODE_PROMISCUOUS	6	// all packets on LAN.

#endif // not defined, _ANALYZER
