/*
// PROGRAM NAME:  WINDOW.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module implements the WINDOW object in COW.
//
// MODIFICATION HISTORY.
//	S. E. Jones	91/05/10.	Original.
//	S. E. Jones	92/03/18.	Added WinChangeBorder.
//	S. E. Jones	92/04/03.	Added "simple" windows.
//	S. E. Jones	92/07/13.	Added better screen type detection.
//	S. E. Jones	92/12/20.	Fixed screen detection.
//
// NOTICE:  Copyright (C) 1991-1993 General Software, Inc.
*/

#define SCREEN_COLS		80
#define SCREEN_ROWS		25

#define ATTR_BLACK		0
#define ATTR_BLUE		1
#define ATTR_GREEN		2
#define ATTR_CYAN		3
#define ATTR_RED		4
#define ATTR_MAGENTA		5
#define ATTR_BROWN		6
#define ATTR_GREY		7
#define ATTR_SHADOW		8
#define ATTR_LIGHT_BLUE         9
#define ATTR_LIGHT_GREEN	10
#define ATTR_LIGHT_CYAN         11
#define ATTR_LIGHT_RED		12
#define ATTR_LIGHT_MAGENTA	13
#define ATTR_YELLOW		14
#define ATTR_WHITE		15

#define ATTR(bg,fg) ((bg<<4) | (fg))    // builds IBM PC attribute byte.

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "cow.h"			// COW system include file.

#define DEFAULT_DESKTOP_CHARACTER 177	// medium grey block.

//
// PRIVATE data structures managed by this module.
//

static PWINDOW WinList=NULL;		// doubly-linked list of windows.
static USHORT *ScreenPtr;		// configured pointer to screen memory.
static UCHAR PaletteColors [MAX_PALETTE]; // attributes indexed by palette #.

#define BORDER_UL	0
#define BORDER_UR	1
#define BORDER_LL	2
#define BORDER_LR	3
#define BORDER_HORIZ	4
#define BORDER_VERT	5
#define BORDER_LEFT	6
#define BORDER_RIGHT	7
#define BORDER_MAX	8

static UCHAR SingleBorders [BORDER_MAX] = {
    218, 191, 192, 217, 196, 179, 195, 180
}; // SingleBorders

static UCHAR DoubleBorders [BORDER_MAX] = {
    201, 187, 200, 188, 205, 186, 204, 185
}; // DoubleBorders

BOOLEAN IsMono ()
{
    USHORT *ptr, devflags;
    ULONG addr;

    addr = 0x00400000L;                 // point to BIOS data area.
    ptr = (USHORT *)addr;		// ptr = ptr to BIOS data area.
    devflags = *(ptr+8);		// get 8th word (16th byte).
    if ((devflags & 0x0030) == 0x0030) {
	return TRUE;			// this is a monochrome system.
    }
    return FALSE;			// the current screen is not mono.
} // IsMono

VOID ReadChar (x, y, pch)
    USHORT x;
    USHORT y;
    UCHAR *pch;
{
    USHORT Word;

    Word = *(ScreenPtr + x + y*SCREEN_COLS);
    *pch = (UCHAR)(Word & 0x00ff);
} // ReadChar

VOID WriteCharAttr (x, y, ch, attr)
    USHORT x;
    USHORT y;
    UCHAR ch;
    UCHAR attr;
{
    USHORT Word;

    Word = (USHORT)ch | ((USHORT)attr << 8);
    *(ScreenPtr + x + y*SCREEN_COLS) = Word;
} // WriteCharAttr

VOID WinChangeBorderType (Window, BorderType)
    PWINDOW Window;
    UCHAR BorderType;
{
    USHORT *p, width, height, i, len, x, y, xstart, ystart;
    UCHAR *b, attr, *q;

    width = Window->Width;
    height = Window->Height;
    xstart = Window->CenterX - (width/2);
    ystart = Window->CenterY - (height/2);
    attr = Window->Attr;

    switch (BorderType & WINDOW_BORDER_KIND) {
	case WINDOW_BORDER_NONE: Window->BorderTable = NULL; break;
	case WINDOW_BORDER_SINGLE: Window->BorderTable = SingleBorders; break;
	case WINDOW_BORDER_DOUBLE: Window->BorderTable = DoubleBorders; break;
	case WINDOW_BORDER_SIMPLE_S: Window->BorderTable = SingleBorders; break;
	case WINDOW_BORDER_SIMPLE_D: Window->BorderTable = DoubleBorders; break;
	case WINDOW_BORDER_NOCHANGE: break;	// does nothing.
    }

    //
    // Now draw an optional border.
    //

    b = Window->BorderTable;
    if (b != NULL) {
	WriteCharAttr (xstart,	       ystart, *(b+BORDER_UL), attr);
	WriteCharAttr (xstart+width-1, ystart, *(b+BORDER_UR), attr);
	WriteCharAttr (xstart,	       ystart+height-1, *(b+BORDER_LL), attr);
	WriteCharAttr (xstart+width-1, ystart+height-1, *(b+BORDER_LR), attr);

	for (i=xstart+1; i<xstart+width-1; i++) {
	    WriteCharAttr (i, ystart, *(b+BORDER_HORIZ), attr);
	}
	for (i=xstart+1; i<xstart+width-1; i++) {
	    WriteCharAttr (i, ystart+height-1, *(b+BORDER_HORIZ), attr);
	}
	for (i=ystart+1; i<ystart+height-1; i++) {
	    WriteCharAttr (xstart,	   i, *(b+BORDER_VERT), attr);
	    WriteCharAttr (xstart+width-1, i, *(b+BORDER_VERT), attr);
	}

	if (Window->HeaderRows == 2) {
	    WriteCharAttr (xstart,	   ystart+2, *(b+BORDER_LEFT), attr);
	    WriteCharAttr (xstart+width-1, ystart+2, *(b+BORDER_RIGHT), attr);
	    for (i=xstart+1; i<xstart+width-1; i++) {
		WriteCharAttr (i, ystart+2, *(b+BORDER_HORIZ), attr);
	    }
	}

	//
	// Display the title string, centered in the top of the box.
	//

	if (Window->HeaderRows == 0) {
	    y = ystart;                 // handle special SIMPLE windows.
	} else {
	    y = ystart + 1;		// typical case.
	}

	len = strlen (Window->Title);
	x = xstart + (width/2 - len/2);
	for (q=Window->Title; *q; q++) {
	    WriteCharAttr (x++, y, *q, attr);
	}
    }
} // WinChangeBorderType

VOID WinHiliteBorder (Window)
    PWINDOW Window;
{
    UCHAR attr;

    attr = Window->Attr;
    if ((!IsMono ()) && !(Window->BorderType & WINDOW_BORDER_NOHILITE)) {
	Window->Attr = ATTR (ATTR_BLUE, ATTR_YELLOW);
    }
    WinChangeBorderType (Window, WINDOW_BORDER_NOCHANGE);
    Window->Attr = attr;
} // WinHiliteBorder

VOID WinUnHiliteBorder (Window)
    PWINDOW Window;
{
    WinChangeBorderType (Window, WINDOW_BORDER_NOCHANGE);
} // WinUnHiliteBorder

VOID WinDisplay (Window)
    PWINDOW Window;
{
    USHORT *p, width, height, i, len, x, y, xstart, ystart, maxx;
    UCHAR *b, attr, *q, ch;

    width = Window->Width;
    height = Window->Height;
    xstart = Window->CenterX - (width/2);
    ystart = Window->CenterY - (height/2);
    attr = Window->Attr;

    //
    // Clear the entire box.
    //

    for (x=xstart; x<xstart+width; x++) {
	for (y=ystart; y<ystart+height; y++) {
	    WriteCharAttr (x, y, 0x20, attr);
	}
    }

    //
    // Add a dropshadow if required.
    //

    if (!(Window->BorderType & WINDOW_BORDER_NOSHADOW)) {
	x = xstart + width;
	if (x < SCREEN_COLS) {
	    for (y=ystart+1; y<ystart+height; y++) {
		ReadChar (x, y, &ch);
		WriteCharAttr (x, y, ch, ATTR (ATTR_BLACK, ATTR_SHADOW));
	    }
	}

	y = ystart + height;
	if (y < SCREEN_ROWS) {
	    maxx = xstart + width + 1;
	    if (maxx > SCREEN_COLS) {
		maxx--;
	    }
	    for (x=xstart+1; x<maxx; x++) {
		ReadChar (x, y, &ch);
		WriteCharAttr (x, y, ch, ATTR (ATTR_BLACK, ATTR_SHADOW));
	    }
	}
    }

    //
    // Display the window's border & title with a hilite.
    //

    WinHiliteBorder (Window);
} // WinDisplay

VOID WinInit ()
{
    USHORT w1, w2;
    USHORT i, atr;
    ULONG l;
    union REGS regs;

    //
    // Call the video BIOS to disable blinking and allow high-intensity
    // background attributes.  This lets us use black-on-yellow, etc.
    //

    regs.h.ah = 0x10;			// (AH) = set palette function.
    regs.h.al = 0x03;			// (AL) = set blink/intensity mode.
    regs.h.bl = 0x00;			// (BX) = use intensity.
    int86 (0x10, &regs, &regs);         // set intensity, NOT blink, mode.

    //
    // Determine screen memory address.
    //

    l = 0xb8000000L;
    if (IsMono ()) {
	l = 0xb0000000L;
    }
    ScreenPtr = (USHORT *)l;

    //
    // Initialize the palette table.
    //

    if (IsMono ()) {
	atr = ATTR(ATTR_BLACK, ATTR_GREY);
	for (i=0; i<MAX_PALETTE; i++) {
	    PaletteColors [i] = ATTR (ATTR_GREY, ATTR_BLACK);
	}
	PaletteColors [ERROR_PALETTE] = ATTR (ATTR_BLACK, ATTR_GREY);
    } else {
	atr = ATTR(ATTR_BLUE, ATTR_GREY);
	PaletteColors [NORMAL_PALETTE] =  ATTR (ATTR_BLUE, ATTR_WHITE);
	PaletteColors [INIT_PALETTE] = ATTR (ATTR_CYAN, ATTR_BLUE);
	PaletteColors [INIT_INV_PALETTE] = ATTR (ATTR_BLUE, ATTR_CYAN);
	PaletteColors [HELP_PALETTE] = ATTR (ATTR_YELLOW, ATTR_BLACK);
	PaletteColors [ERROR_PALETTE] = ATTR (ATTR_RED, ATTR_YELLOW);
	PaletteColors [WARNING_PALETTE] = ATTR (ATTR_BLUE, ATTR_YELLOW);
    }
} // WinInit

VOID WinPalette (Palette, ForeGround, BackGround)
    USHORT Palette;
    UCHAR ForeGround;
    UCHAR BackGround;
{
    PaletteColors [Palette] =  ATTR (BackGround, ForeGround);
} // WinPalette

PWINDOW WinCreate (Title, CenterX, CenterY, Width, Height, BorderType, Palette)
    UCHAR *Title;
    USHORT CenterX;
    USHORT CenterY;
    USHORT Width;
    USHORT Height;
    USHORT BorderType;
    UCHAR Palette;
{
    PWINDOW w;
    USHORT i, x, y, xbase, ybase, LocalWidth, LocalHeight;

    w = (PWINDOW)malloc (sizeof (WINDOW));
    if (w == NULL) {
	return NULL;
    }

    //
    // Allocate a saved screen, even accounting for borders and
    // a dropshadow to be saved for the screen underneath.
    //

    w->SavedScreen = (USHORT *)malloc ((Width+1)*(Height+1)*2);
    if (w->SavedScreen == NULL) {
	free (w);
	return NULL;
    }

    w->BorderType = BorderType;         // saved border flags.
    w->Attr = PaletteColors [Palette];
    w->CenterX = CenterX;
    w->CenterY = CenterY;
    w->Width = Width;	// (strlen (Title) > Width) ? strlen (Title) : Width;
    w->Height = Height;
    w->HeaderRows = 2;			// normally, we have a title & a bar.
    w->BorderTable = NULL;
    switch (w->BorderType & WINDOW_BORDER_KIND) {
	case WINDOW_BORDER_NONE: break;
	case WINDOW_BORDER_SINGLE:
	    w->BorderTable = SingleBorders;
	    break;
	case WINDOW_BORDER_DOUBLE:
	    w->BorderTable = DoubleBorders;
	    break;
	case WINDOW_BORDER_SIMPLE_S:
	    w->BorderTable = SingleBorders;
	    w->HeaderRows = 0;
	    break;
	case WINDOW_BORDER_SIMPLE_D:
	    w->BorderTable = DoubleBorders;
	    w->HeaderRows = 0;
	    break;
	default: free (w); return NULL;
    }
    w->Title = malloc (strlen (Title)+2);
    if (w->Title == NULL) {		// if we couldn't make the title string.
	free ((UCHAR *)w->SavedScreen);
	free (w);
	return NULL;
    }
    strcpy (w->Title, Title);

    //
    // Save the screen underneath this window, even the dropshadow area.
    //

    LocalWidth = w->Width;
    LocalHeight = w->Height;
    if (!(w->BorderType & WINDOW_BORDER_NOSHADOW)) {
	LocalWidth++;
	LocalHeight++;
    }

    xbase = w->CenterX - (w->Width/2);
    ybase = (w->CenterY - (w->Height/2)) * SCREEN_COLS;
    for (x=0; x<LocalWidth; x++) {
	for (y=0; y<LocalHeight; y++) {
	    *(w->SavedScreen + x + y*LocalWidth) = *(ScreenPtr + xbase + x + ybase + y*SCREEN_COLS);
	}
    }

    //
    // Push this window on the front of the list.
    //

    w->Fwdlink = WinList;
    WinList = w;

    //
    // Update the screen to show the window to the user.
    //

    WinDisplay (w);

    //
    // Return a pointer to the newly-allocated window.
    //

    return w;
} // WinCreate

VOID WinDestroy (Window)
    PWINDOW Window;
{
    USHORT xbase, ybase, x, y, LocalWidth, LocalHeight;

    //
    // If this window is on the top of the list, remove him.
    //

    if (WinList == Window) {
	WinList = Window->Fwdlink;
    }

    //
    // Restore the screen underneath it, even the dropshadow area.
    //

    LocalWidth = Window->Width;
    LocalHeight = Window->Height;
    if (!(Window->BorderType & WINDOW_BORDER_NOSHADOW)) {
	LocalWidth++;
	LocalHeight++;
    }

    xbase = Window->CenterX - (Window->Width/2);
    ybase = (Window->CenterY - (Window->Height/2)) * SCREEN_COLS;
    for (x=0; x<LocalWidth; x++) {
	for (y=0; y<LocalHeight; y++) {
	    *(ScreenPtr + xbase + x + ybase + y*SCREEN_COLS) =
		*(Window->SavedScreen + x + y*LocalWidth);
	}
    }

    //
    // Now release all of the memory associated with this window.
    //

    free (Window->Title);
    free ((UCHAR *)Window->SavedScreen);
    free ((UCHAR *)Window);
} // WinDestroy

VOID WinWrite (Window, Text, x, y, Width)
    PWINDOW Window;
    UCHAR *Text;
    USHORT x;
    USHORT y;
    USHORT Width;
{
    USHORT i, j, len, xstart, ystart, localx, maxwidth;
    UCHAR *q;

    if (x > (Window->Width-2)) {
	return;                         // clip the entire string.
    }

    //
    // Convert virtual coordinates to logical ones.
    //

    xstart = Window->CenterX - (Window->Width/2) + x;
    ystart = Window->CenterY - (Window->Height/2) + y;

    if (Window->BorderTable != NULL) {
	xstart++;
	ystart += (Window->HeaderRows + 1);
    }

    //
    // Clip the string if we need to.
    //

    len = Width;			// start with his requested width.

    maxwidth = Window->Width;
    if (Window->BorderTable != NULL) {
	maxwidth -= 2;
    }

    if (len > maxwidth-x) {
	len = maxwidth-x;
    }
    localx = xstart;			// start at the edge, not in the center.

    //
    // Now write-out the string into the window.
    //

    for (i=0; i<len; i++) {
	if (*Text == 0) {		// found zero byte terminator.
	    break;
	}
	WriteCharAttr (localx++, ystart, *(Text++), Window->Attr);
    }

    //
    // We need to pad the remainder of the field with blanks.
    //

    for (j=i; j<len; j++) {
	WriteCharAttr (localx++, ystart, 0x20, Window->Attr);
    }
} // WinWrite

VOID WinWriteAttr (Window, Text, x, y, Width, Attr)
    PWINDOW Window;
    UCHAR *Text;
    USHORT x;
    USHORT y;
    USHORT Width;
    UCHAR Attr;
{
    USHORT i, j, len, xstart, ystart, localx, maxwidth;
    UCHAR *q;

    if (x > (Window->Width-2)) {
	return;                         // clip the entire string.
    }

    //
    // Convert virtual coordinates to logical ones.
    //

    xstart = Window->CenterX - (Window->Width/2) + x;
    ystart = Window->CenterY - (Window->Height/2) + y;

    if (Window->BorderTable != NULL) {
	xstart++;
	ystart += (Window->HeaderRows + 1);
    }

    //
    // Clip the string if we need to.
    //

    len = Width;			// start with his requested width.

    maxwidth = Window->Width;
    if (Window->BorderTable != NULL) {
	maxwidth -= 2;
    }

    if (len > maxwidth-x) {
	len = maxwidth-x;
    }
    localx = xstart;			// start at the edge, not in the center.

    //
    // Now write-out the string into the window.
    //

    for (i=0; i<len; i++) {
	if (*Text == 0) {		// found zero byte terminator.
	    break;
	}
	WriteCharAttr (localx++, ystart, *(Text++), Attr);
    }

    //
    // We need to pad the remainder of the field with blanks.
    //

    for (j=i; j<len; j++) {
	WriteCharAttr (localx++, ystart, 0x20, Attr);
    }
} // WinWriteAttr

VOID WinHilite (Window, x, y, Width)
    PWINDOW Window;
    USHORT x;
    USHORT y;
    USHORT Width;
{
    USHORT i, len, xstart, ystart, localx;
    UCHAR ch, attr;

    if (x > (Window->Width-2)) {
	return;                         // clip the entire hilite.
    }

    //
    // Convert virtual coordinates to logical ones.
    //

    xstart = Window->CenterX - (Window->Width/2) + 1 + x;
    ystart = Window->CenterY - (Window->Height/2) + 1 + Window->HeaderRows + y;

    //
    // Clip the hilite if we need to.
    //

    len = Width;			// start with his requested width.
    if (len > (Window->Width-2)-x) {
	len = (Window->Width-2)-x;
    }
    localx = xstart;			// start at the edge, not in the center.

    //
    // Now hilight the area in the window.
    //

    attr = (Window->Attr >> 4) | ((Window->Attr << 4) & 0x70);
    for (i=0; i<len; i++) {
	ReadChar (localx, ystart, &ch);
	WriteCharAttr (localx++, ystart, ch, attr);
    }
} // WinHilite

VOID WinUnHilite (Window, x, y, Width)
    PWINDOW Window;
    USHORT x;
    USHORT y;
    USHORT Width;
{
    USHORT i, len, xstart, ystart, localx;
    UCHAR ch, attr;

    if (x > (Window->Width-2)) {
	return;                         // clip the entire hilite.
    }

    //
    // Convert virtual coordinates to logical ones.
    //

    xstart = Window->CenterX - (Window->Width/2) + 1 + x;
    ystart = Window->CenterY - (Window->Height/2) + 1 + Window->HeaderRows + y;

    //
    // Clip the hilite if we need to.
    //

    len = Width;			// start with his requested width.
    if (len > (Window->Width-2)-x) {
	len = (Window->Width-2)-x;
    }
    localx = xstart;			// start at the edge, not in the center.

    //
    // Now turn off the hilight in the window.
    //

    for (i=0; i<len; i++) {
	ReadChar (localx, ystart, &ch);
	WriteCharAttr (localx++, ystart, ch, Window->Attr);
    }
} // WinUnHilite
