/*
// PROGRAM NAME:  STREAM.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module implements the STREAM object in COW.
//
// MODIFICATION HISTORY.
//	S. E. Jones	91/05/10.	Original.
//	S. E. Jones	92/09/28.	Cloned from LIST object.
//	S. E. Jones	92/10/18.	Added WordStar-compatible keys.
//
// NOTICE:  Copyright (C) 1991-1993 General Software, Inc.
*/

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "cow.h"			// COW system include file.

#define CTRL_A	1
#define CTRL_B	2
#define CTRL_C	3
#define CTRL_D	4
#define CTRL_E	5
#define CTRL_F	6
#define CTRL_G	7
#define CTRL_H	8
#define CTRL_I	9
#define CTRL_J	10
#define CTRL_K	11
#define CTRL_L	12
#define CTRL_M	13
#define CTRL_N	14
#define CTRL_O	15
#define CTRL_P	16
#define CTRL_Q	17
#define CTRL_R	18
#define CTRL_S	19
#define CTRL_T	20
#define CTRL_U	21
#define CTRL_V	22
#define CTRL_W	23
#define CTRL_X	24
#define CTRL_Y	25
#define CTRL_Z	26

static USHORT StartLine (Stream, Pos)
    PSTREAM Stream;
    USHORT Pos;
{
    UCHAR ch;
    USHORT ptr;

    if (Pos == 0) {
	return 0;
    }
    ptr = Pos - 1;

    while (ptr > 0) {
	ch = *(Stream->Buffer + ptr);
	if ((ch == 0x0d) || (ch == 0x0a)) {
	    ptr++;
	    break;
	}
	ptr--;
    }
    return ptr;
} // StartLine

static USHORT EndLine (Stream, Pos)
    PSTREAM Stream;
    USHORT Pos;
{
    UCHAR ch;
    USHORT ptr;

    ptr = Pos;
    while (ptr < Stream->BytesUsed) {
	ch = *(Stream->Buffer + ptr);
	if ((ch == 0) || (ch == 0x0d) || (ch == 0x0a)) {
	    break;
	}
	ptr++;
    }
    return ptr;
} // EndLine

static USHORT CursorCol (Stream)
    PSTREAM Stream;
{
    return (Stream->Cursor - StartLine (Stream, Stream->Cursor));
} // CursorCol

static USHORT NextLine (Stream, Pos)
    PSTREAM Stream;
    USHORT Pos;
{
    USHORT ptr;
    UCHAR ch;

    ptr = Pos;
    while ((ch=Stream->Buffer [ptr]) != 0) {
	if (ch == 0x0d) {
	    break;
	} else {
	    ptr++;
	}
    }

    if (ch == 0x0d) {
	ptr++;
	if (Stream->Buffer [ptr] == 0x0a) {
	    ptr++;
	}
    }
    return ptr;
} // NextLine

static UCHAR CurCh (Stream)
    PSTREAM Stream;
{
    return *(Stream->Buffer + Stream->Cursor);
} // CurrChar

static UCHAR PrevCh (Stream)
    PSTREAM Stream;
{
    UCHAR ch;

    if (Stream->Cursor == 0) {
	return (UCHAR)0;
    }

    ch = *(Stream->Buffer + Stream->Cursor - 1);
    if (ch == 0x0a) {
	ch = 0x0d;
    }
    return ch;
} // CurrChar

static BOOLEAN AtEol (Stream)
    PSTREAM Stream;
{
    UCHAR ch;

    ch = *(Stream->Buffer + Stream->Cursor);
    if ((ch == 0) || (ch == 0x0d) || (ch == 0x0a)) {
	return TRUE;
    }
    return FALSE;
} // AtEol

static BOOLEAN AtEof (Stream)
    PSTREAM Stream;
{
    UCHAR ch;

    ch = *(Stream->Buffer + Stream->Cursor);
    if (ch == 0) {
	return TRUE;
    }
    return FALSE;
} // AtEof

static BOOLEAN AtBof (Stream)
    PSTREAM Stream;
{
    if (Stream->Cursor == 0) {
	return TRUE;
    } else {
	return FALSE;
    }
} // AtBof

static BOOLEAN Fwd (Stream)
    PSTREAM Stream;
{
    UCHAR ch;

    ch = *(Stream->Buffer + Stream->Cursor);
    if (ch == 0) {
	return FALSE;
    }

    if (ch == 0x0d) {
	Stream->Cursor += 2;		// skip over CR/LF pair.
	Stream->LeftEdge = 0;
	if (Stream->CurRow >= Stream->ViewRows-1) {
	    Stream->Line1 = NextLine (Stream, Stream->Line1);
	} else {
	    Stream->CurRow++;
	}
    } else {
	Stream->Cursor++;
    }
    return TRUE;
} // Fwd

static BOOLEAN Back (Stream)
    PSTREAM Stream;
{
    UCHAR ch;

    if (Stream->Cursor == 0) {		// if at beginning of stream.
	return FALSE;
    }

    ch = *(Stream->Buffer + Stream->Cursor - 1);
    if (ch == 0x0a) {			// is it a LF from a CR/LF pair?
	Stream->Cursor -= 2;		// if so, skip over CR/LF pair.
	Stream->LeftEdge = 0;		// recompute this.
	if (Stream->CurRow == 0) {
	    Stream->Line1 = StartLine (Stream, Stream->Cursor);
	} else {
	    Stream->CurRow--;
	}
    } else {
	Stream->Cursor--;
    }
    return TRUE;
} // Back

static BOOLEAN Del (Stream)
    PSTREAM Stream;
{
    UCHAR ch;
    USHORT nchar;

    ch = *(Stream->Buffer + Stream->Cursor);
    if (ch == 0) {			// if nothing to delete.
	return FALSE;
    }

    nchar = 1;
    if (ch == 0x0d) {
	nchar++;
	Stream->LeftEdge = 0;
    } else if (Stream->LeftEdge > 0) {
	Stream->LeftEdge--;
    }
    memmove (Stream->Buffer + Stream->Cursor,
	     Stream->Buffer + Stream->Cursor + nchar,
	     Stream->BytesUsed - (Stream->Cursor + nchar));
    Stream->BytesUsed -= nchar;
    return TRUE;
} // Del

BOOLEAN Insert (Stream, Buffer, BufLen)
    PSTREAM Stream;
    UCHAR *Buffer;
    USHORT BufLen;
{
    if (Stream->BytesUsed + BufLen >= Stream->BufLen) {
	return FALSE;			// if it would overflow the buffer.
    }

    //
    // Perform an overlapping move that shifts everything at the cursor
    // right by BufLen number of bytes.  This makes room for the new stuff.
    //

    memmove (Stream->Buffer + Stream->Cursor + BufLen,
	     Stream->Buffer + Stream->Cursor,
	     Stream->BytesUsed - Stream->Cursor);

    //
    // Now copy-in our new data at the cursor location.
    //

    memmove (Stream->Buffer + Stream->Cursor, Buffer, BufLen);

    //
    // Don't forget that we now have more text in our buffer.
    //

    Stream->BytesUsed += BufLen;

    //
    // Advance the cursor position to point to the first byte
    // FOLLOWING the text we just inserted.
    //

    Stream->Cursor += BufLen;

    return TRUE;
} // Insert

static VOID DisplayStream (Stream)
    PSTREAM Stream;
{
    USHORT i, cc;
    UCHAR ch, *ptr, *eol;

    //
    // Position LeftEdge so that the cursor is always visible.
    //

    if (cc < Stream->LeftEdge) {
	cc = Stream->LeftEdge;
    }

    cc = CursorCol (Stream);
    if (cc+1 >= Stream->ViewCols + Stream->LeftEdge) {
	Stream->LeftEdge = (cc+1) - Stream->ViewCols;
    }

    //
    // Now display each row of text in the window.
    //

    ptr = Stream->Buffer + Stream->Line1;
    for (i=0; i<Stream->ViewRows; i++) {
	if (*ptr == 0) {
	    WinWrite (Stream->Window, " ", 0, i, 999);
	} else {
	    eol = ptr;
	    while (TRUE) {
		ch = *eol;
		if ((ch == 0x0d) || (ch == 0)) {
		    break;
		} else {
		    eol++;
		}
	    }

	    //
	    // Build ASCIIZ string for this line, and display it in the window.
	    //

	    *eol = 0;			// temporary end of line.
	    if (strlen (ptr) < Stream->LeftEdge) {
		WinWrite (Stream->Window, " ", 0, i, 999);
	    } else {
		WinWrite (Stream->Window, ptr+Stream->LeftEdge, 0, i, 999);
	    }
	    *eol = ch;			// restore end of line.

	    //
	    // Skip the following LF if we found a CR.
	    //

	    ptr = eol;
	    if (ch == 0) {		// if end of stream.
		continue;
	    }
	    ptr++;			// position after EOL character.
	    if (ch == 0x0d) {
		if (*ptr == 0x0a) {
		    ptr++;
		}
	    }
	}
    }
} // DisplayStream

PSTREAM StreamCreate (Title, BufLen, CenterX, CenterY, Width, Height)
    UCHAR *Title;
    USHORT BufLen;
    USHORT CenterX;
    USHORT CenterY;
    USHORT Width;
    USHORT Height;
{
    PSTREAM s;

    if (Height <= 4) {                  // we have 4 lines of overhead.
	return NULL;
    }

    s = (PSTREAM)malloc (sizeof (STREAM));
    if (s == NULL) {			// if we couldn't get the memory.
	return NULL;
    }

    s->Buffer = malloc (BufLen);
    if (s->Buffer == NULL) {
	free (s);
	return NULL;
    }
    s->Buffer [0] = 0;			// zap the buffer right away.

    s->Window = WinCreate (Title, CenterX, CenterY, Width, Height,
			   WINDOW_BORDER_DOUBLE, NORMAL_PALETTE);
    if (s->Window == NULL) {		// if we couldn't allocate a window.
	free (s->Buffer);
	free (s);
	return NULL;
    }

    s->ViewCols = Width - 2;
    s->ViewRows = Height - 4;
    s->LeftEdge = 0;			// # chars hidden by left margin.
    s->BufLen = BufLen;                 // establish size of buffer.
    s->Line1 = 0;			// reset starting display pointer.
    s->Cursor = 0;			// reset cursor text pointer.
    s->BytesUsed = 1;			// # bytes in buffer (count zero byte).
    s->CurRow = 0;			// start cursor at top row.
    return s;				// return stream object to the caller.
} // StreamCreate

VOID StreamDestroy (Stream)
    PSTREAM Stream;
{
    WinDestroy (Stream->Window);	// destroy the window for this stream.
    free (Stream->Buffer);		// free the text buffer.
    free (Stream);			// free this block of storage.
} // StreamDestroy

USHORT StreamCopy (Stream, Buffer, BufLen)
    PSTREAM Stream;
    UCHAR *Buffer;
    USHORT BufLen;
{
    USHORT len;
    len = Stream->BytesUsed;
    if (len > BufLen) {
	len = BufLen;
    }

    memmove (Buffer, Stream->Buffer, len);
    return len;
} // StreamCopy

UCHAR *StreamBuffer (Stream)
    PSTREAM Stream;
{
    return Stream->Buffer;
} // StreamBuffer

USHORT StreamLength (Stream)
    PSTREAM Stream;
{
    return Stream->BytesUsed - 1;	// don't count zero byte terminator.
} // StreamLength

BOOLEAN StreamInsert (Stream, Buffer, BufLen)
    PSTREAM Stream;
    UCHAR *Buffer;
    USHORT BufLen;
{
    BOOLEAN result;

    result = Insert (Stream, Buffer, BufLen);
    DisplayStream (Stream);
    return result;
} // StreamInsert

ACTION StreamEdit (Stream)
    PSTREAM Stream;
{
    BOOLEAN InsertMode = TRUE;
    UCHAR c, ch, NewLine [2], *p;
    USHORT i, nchar, x, col, savecur, saverow, repeat=0;
    UCHAR *t = "F1=Save,ESC=Quit";

    NewLine [0] = 0x0d;
    NewLine [1] = 0x0a;

    Stream->LeftEdge = 0;
    Stream->Cursor = 0;
    Stream->Line1 = 0;
    Stream->CurRow = 0;

    WinWrite (Stream->Window, t, 0, Stream->ViewRows, strlen (t));
    WinWrite (Stream->Window, "<Ins>", Stream->ViewCols-5, Stream->ViewRows, 5);

    while (TRUE) {
	if (repeat != 0) {		// handle repeated commands.
	    repeat--;			// count down to the end.
	} else {
	    DisplayStream (Stream);	// paint the text.

	    //
	    // Enable the cursor before accepting a keystroke.
	    //

	    x = CursorCol (Stream) - Stream->LeftEdge;
	    WinHilite (Stream->Window, x, Stream->CurRow, 1);

	    ch = KeyRead ();		// ch = user's action key.

	    //
	    // Disable the cursor before performing the action.
	    //

	    WinUnHilite (Stream->Window, x, Stream->CurRow, 1);
	}

	//
	// Now handle the keystroke.
	//

	switch (ch) {
	    case I_INSERT:		// toggle insert mode.
		InsertMode = !InsertMode;
		if (InsertMode) {
		    WinWrite (Stream->Window, "<Ins>",
			      Stream->ViewCols - 5, Stream->ViewRows, 5);
		} else {
		    WinWrite (Stream->Window, "<Ovr>",
			      Stream->ViewCols - 5, Stream->ViewRows, 5);
		}
		break;

	    case I_BACKSPACE:		// delete previous char.
		if (Back (Stream)) {
		    Del (Stream);
		}
		break;

	    case I_DELETE:		// delete char under cursor.
	    case CTRL_G:
		Del (Stream);
		break;

	    case CTRL_Y:		// delete line under cursor.
		Stream->LeftEdge = 0;
		Stream->Cursor = StartLine (Stream, Stream->Cursor);
		while (!AtEol (Stream, Stream->Cursor)) {
		    Del (Stream);
		}
		if (CurCh (Stream) == 0x0d) {
		    Del (Stream);	// delete CR.
		}
		if (CurCh (Stream) == 0x0a) {
		    Del (Stream);	// delete LF.
		}
		break;

	    case I_F1:			// return with SAVE status.
		return ACTION_SAVE;

	    case I_ESC:                 // return with ABORT status.
		return ACTION_ABORT;

	    case I_ENTER:		// position to col 0, line n+1.
		Insert (Stream, NewLine, 2);
		Stream->LeftEdge = 0;
		if (Stream->CurRow >= Stream->ViewRows-1) {
		    Stream->Line1 = NextLine (Stream, Stream->Line1);
		} else {
		    Stream->CurRow++;
		}
		break;

	    case I_LEFT:		// move cursor left one character.
	    case CTRL_A:
	    case CTRL_S:
		Back (Stream);
		break;

	    case I_RIGHT:		// move cursor right one character.
	    case CTRL_D:
	    case CTRL_F:
		Fwd (Stream);
		break;

	    case I_UP:			// move select bar up one line.
	    case CTRL_E:
		col = CursorCol (Stream);
		while (!AtBof (Stream, Stream->Cursor)) {
		    Back (Stream);
		    if (CurCh (Stream) == 0x0d) {
			break;		// found end of previous line.
		    }
		}

		Stream->LeftEdge = 0;
		Stream->Cursor = StartLine (Stream, Stream->Cursor);
		for (i=0; (i<col) && (!AtEol (Stream, Stream->Cursor)); i++) {
		    Fwd (Stream);
		}
		break;

	    case I_DOWN:		// move select bar down one line.
	    case CTRL_X:
		savecur = Stream->Cursor;
		saverow = Stream->CurRow;
		col = CursorCol (Stream);
		while (!AtEol (Stream, Stream->Cursor)) {
		    Fwd (Stream);
		}

		if (AtEof (Stream, Stream->Cursor)) {
		    Stream->Cursor = savecur;
		    Stream->CurRow = saverow;
		    break;
		}

		Fwd (Stream);		// skip over EOL.
		for (i=0; (i<col) && (!AtEol (Stream, Stream->Cursor)); i++) {
		    Fwd (Stream);
		}
		break;

	    case I_PGUP:		// page up a screenful.
	    case CTRL_R:
		repeat = Stream->ViewRows;
		ch = I_UP;		// repeat I_UP (ViewRows) times.
		break;

	    case I_PGDN:		// page down a screenful.
	    case CTRL_C:
		repeat = Stream->ViewRows;
		ch = I_DOWN;		// repeat I_DOWN (ViewRows) times.
		break;

	    case CTRL_J:		// go to opposite end of line.
		if (Stream->Cursor == StartLine (Stream, Stream->Cursor)) {
		    Stream->Cursor = EndLine (Stream, Stream->Cursor);
		} else {
		    Stream->Cursor = StartLine (Stream, Stream->Cursor);
		}
		break;

	    case I_HOME:		// position to top of file.
		Stream->Cursor = StartLine (Stream, Stream->Cursor);
		break;

	    case I_END:                 // position to bottom of file.
		Stream->Cursor = EndLine (Stream, Stream->Cursor);
		break;

	    case CTRL_B:		// "Bullet" character.
		ch = 254;		// special IBM bullet character.

	    default:			// ordinary character to insert.
		if (InsertMode) {
		    Insert (Stream, &ch, 1);
		} else {
		    c = Stream->Buffer [Stream->Cursor];
		    if ((c == 0x0d) || (c == 0)) {
			Insert (Stream, &ch, 1);
		    } else {
			Stream->Buffer [Stream->Cursor++] = ch;
		    }
		}
	}
    }
} // StreamEdit
