/*
// PROGRAM NAME:  LIST.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module implements the LIST object in COW.
//
// MODIFICATION HISTORY.
//	S. E. Jones	91/05/10.	Original.
//	S. E. Jones	91/05/31.	Added 1st letter selection to ListEdit.
//	S. E. Jones	92/11/12.	Added XlateRtn callout for EMAIL system.
//	S. E. Jones	92/12/28.	Added border highlighting.
//
// NOTICE:  Copyright (C) 1991-1993 General Software, Inc.
*/

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "cow.h"			// COW system include file.

//
// Default help for list objects.
//

UCHAR * far DefaultListHelpText [] = {
    "                 ON-LINE HELP FOR USING LIST BOXES",
    " ",
    "The highlighted window on the screen containing the cursor is",
    "called a list box.  This type of window is used to make a choice,",
    "or to build a list of items that go together.  Sometimes, it is",
    "used for both purposes.",
    " ",
    "To select an item from the list, simply position the scroll bar",
    "over the item using your UP and DOWN arrow keys, or use PGUP and",
    "PGDN keys on the keypad.  In order for these keys to work properly,",
    "your NUM-LOCK light on your keyboard must not be lit.  Another way",
    "to position the scroll bar is to press a key that matches the first",
    "letter or number listed in the desired entry.  For example, if the",
    "choices are MEAT, POTATO, and VEGETABLE, you could position the",
    "scroll bar over VEGETABLE by pressing 'V', or by using PGDN.  After",
    "the scroll bar is positioned over your selection, press ENTER to",
    "actually select the item.",
    " ",
    "Some list boxes allow you to delete items in the list.  For example,",
    "our list of food items (in the example above) could changed to not",
    "include POTATO by positioning the scroll bar over POTATO, and then",
    "pressing the DEL key on the keypad.  Note that deleting objects in",
    "a list, such as a command menu, may not be supported.  Also, the",
    "NUM-LOCK light must not be lit when using the DEL key.",
    " ",
    "Some list boxes allow you to insert items into the list.  In our",
    "example above, we could add other items such as DESSERT to the",
    "list by pressing the INS key on the keypad.  As with the delete",
    "operation, some list boxes may not have an insert operation.  Also",
    "remember that the NUM-LOCK light must not be lit when using INS.",
    " ",
    "If the list box supports INS and DEL, then the convention is normally",
    "to save your changes to the list by pressing F1, or to abandon the",
    "changes by pressing ESC.  In any case, ESC exits a list box.",
    NULL				// terminates the list.
}; // DefaultListHelpText

static UCHAR UpCase (ch)
    UCHAR ch;
{
    if ((ch >= 'a') && (ch <= 'z')) {
	return (ch + 'A' - 'a');
    }
    return ch;
} // UpCase

VOID DisplayList (ListHead)
    PLIST ListHead;
{
    PLISTE e;
    USHORT i, j;
    UCHAR LittleStr [2];

    //
    // First, draw a vertical selection line at the beginning of each line.
    //

    for (i=0; i<ListHead->ViewSize; i++) {
	WinWrite (ListHead->Window, "", 1, i, 999);
    }

    //
    // If the top element of the list is not NULL, then we can print an
    // "up-arrow" to the left of the first bar in the window, to tell the
    // user that he can scroll up.  Otherwise, print a space there.
    //

    LittleStr [0] = 0x20;		// by default, a blank.
    LittleStr [1] = 0;			// the zero-byte terminator.
    if ((ListHead->TopElement != NULL) && (ListHead->TopElement->Baklink != NULL)) {
	LittleStr [0] = 0x1e;		// up-arrow.
    }
    WinWrite (ListHead->Window, LittleStr, 0, 0, 1);

    //
    // If there are more elements of the list than can be viewed, starting
    // at TopElement, then we can print a "down-arrow" to the left of the
    // last bar in the window, to tell the user that he can scroll down.
    // Otherwise, print a space there.
    //

    LittleStr [0] = 0x20;		// by default, a blank.
    LittleStr [1] = 0;			// the zero-byte terminator.
    e = ListHead->TopElement;
    for (i=0; i<ListHead->ViewSize+1; i++) {
	if (e == NULL) {
	    break;
	}
	e = e->Fwdlink;
    }
    if (i == ListHead->ViewSize+1) {
	LittleStr [0] = 0x1f;		// down-arrow instead of a blank.
    }
    WinWrite (ListHead->Window, LittleStr, 0, ListHead->ViewSize-1, 1);

    //
    // Draw as many elements as we can, starting with the top one.  Do this
    // until we run out of viewspace, or until we run out of list elements.
    //

    e = ListHead->TopElement;
    for (i=0; i<ListHead->ViewSize; i++) {
	if (e == NULL) {
	    break;
	}
	WinWrite (ListHead->Window, e->Title, 3, i, 999);
	e = e->Fwdlink;
    }

    //
    // If we didn't display all of the view slots, then fill the rest
    // of them with an empty string.
    //

    for (j=i; j<ListHead->ViewSize; j++) {
	WinWrite (ListHead->Window, " ", 3, j, 999);
    }
} // DisplayList

static VOID DefaultListHelp (ListElement)
    PLISTE ListElement;
{
    PopupHelp (DefaultListHelpText);
} // DefaultListHelp

PLIST ListCreate (Title, CenterX, CenterY, Width, Height, Flags)
    UCHAR *Title;
    USHORT CenterX;
    USHORT CenterY;
    USHORT Width;
    USHORT Height;
    USHORT Flags;
{
    PLIST l;

    if (Height <= 4) {                  // we have 4 lines of overhead.
	return NULL;
    }

    l = (PLIST)malloc (sizeof (LIST));
    if (l == NULL) {			// if we couldn't get the memory.
	return NULL;
    }

    l->Window = WinCreate (Title, CenterX, CenterY, Width, Height,
			   WINDOW_BORDER_DOUBLE, NORMAL_PALETTE);
    if (l->Window == NULL) {		// if we couldn't allocate a window.
	free (l);
	return NULL;
    }
    WinHiliteBorder (l->Window);

    l->Flags = Flags;			// copy permission flags & attributes.
    l->ViewSize = Height - 4;		// set the view size.
    l->List = NULL;			// nothing on the list yet.
    l->TopElement = NULL;		// there is no top element yet.
    l->HelpRtn = DefaultListHelp;	// routine to execute for help.
    return l;				// return it to the caller.
} // ListCreate

VOID ListDestroy (ListHead)
    PLIST ListHead;
{
    PLISTE e;

    WinDestroy (ListHead->Window);	// destroy the window for this list.

    while (ListHead->List != NULL) {	// remove the elements from the list.
	e = ListHead->List;
	ListHead->List = e->Fwdlink;
	free (e->Title);
	free (e);
    }
    free (ListHead);			// free this block of storage.
} // ListDestroy

PLISTE ListElementCreate (Title, Value)
    UCHAR *Title;
    ULONG Value;
{
    PLISTE e, f;

    e = (PLISTE)malloc (sizeof (LISTE));
    if (e == NULL) {
	return NULL;
    }
    e->Title = malloc (strlen (Title)+1);
    if (e->Title == NULL) {
	free (e);
	return NULL;
    }
    strcpy (e->Title, Title);
    e->Value = Value;

    e->Fwdlink = NULL;
    e->Baklink = NULL;
    return e;
} // ListElementCreate

VOID ListElementDestroy (Element)
    PLISTE Element;
{
    free (Element->Title);
    free (Element);
} // ListElementDestroy

PLISTE ListInsert (ListHead, Title, Value)
    PLIST ListHead;
    UCHAR *Title;
    ULONG Value;
{
    PLISTE e, f;

    e = ListElementCreate (Title, Value);
    if (e == NULL) {
	return NULL;
    }

    //
    // Now insert this element on the list at the end.
    //

    if (ListHead->List == NULL) {
	e->Fwdlink = NULL;
	e->Baklink = NULL;
	ListHead->List = e;
	ListHead->TopElement = e;
    } else {
	for (f=ListHead->List; f->Fwdlink!=NULL; f=f->Fwdlink) ;
	f->Fwdlink = e;
	e->Baklink = f;
	e->Fwdlink = NULL;
    }
    return e;
} // ListInsert

VOID ListDelete (ListHead, Element)
    PLIST ListHead;
    PLISTE Element;
{
    if (ListHead->List == Element) {
	if (Element->Fwdlink == NULL) {
	    ListHead->List = Element->Baklink;
	} else {
	    ListHead->List = Element->Fwdlink;
	}
    }
    if (ListHead->TopElement == Element) {
	if (Element->Fwdlink == NULL) {
	    ListHead->TopElement = Element->Baklink;
	} else {
	    ListHead->TopElement = Element->Fwdlink;
	}
    }

    if (Element->Fwdlink != NULL) {
	Element->Fwdlink->Baklink = Element->Baklink;
    }
    if (Element->Baklink != NULL) {
	Element->Baklink->Fwdlink = Element->Fwdlink;
    }

    ListElementDestroy (Element);
} // ListDelete

VOID ListSort (ListHead)
    PLIST ListHead;
{
    PLISTE p, q, r, src;
    BOOLEAN Reverse;
    int test;

    if (ListHead->List == NULL) {
	return;                         // done sorting the empty list.
    }
    src = ListHead->List->Fwdlink;	// src = list of items to sort.
    ListHead->List->Fwdlink = NULL;	// initialize 1st element on list.
    Reverse = ((ListHead->Flags & LIST_FLAGS_REVERSE_SORT) != 0);

    while (src != NULL) {		// while there are items to sort...
	p = src;			// p = our item to insert.
	src = src->Fwdlink;		// src = FWA, rest of source list.
	r = NULL;			// r = trailer pointer.
	for (q=ListHead->List; q!=NULL; q=q->Fwdlink) {
	    test = strcmp (p->Title, q->Title);
	    if ((Reverse && (test > 0)) || (!Reverse && (test < 0))) {
		p->Fwdlink = q;
		p->Baklink = q->Baklink;
		if (q->Baklink != NULL) {
		    q->Baklink->Fwdlink = p;
		} else {
		    ListHead->List = p;
		}
		q->Baklink = p;
		break;
	    } else {
		r = q;
	    }
	}
	if (q == NULL) {		// we must append to end of the list.
	    p->Baklink = r;
	    p->Fwdlink = NULL;
	    r->Fwdlink = p;
	}
    }
    ListHead->TopElement = ListHead->List; // reset top of list.
} // ListSort

ACTION ListEdit (ListHead)
    PLIST ListHead;
{
    USHORT CurLine=0;			// display offset from top of list box.
    PLISTE p=NULL;			// pointer to LISTE being highlighted.
    PLISTE q, r;
    USHORT i, count;
    UCHAR ch, c;

    ListHead->TopElement = ListHead->List; // reset list display to beginning.
    DisplayList (ListHead);		// redraw this list.

    p = ListHead->List;                 // p = current line being highlighted.
    count = 0;				// number of times to exec 'ch'.

    while (TRUE) {
	if (p != NULL) {
	    WinHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
	}
	if (count) {
	    count--;			// use previous 'ch' value.
	} else {
	    ch = KeyRead ();		// ch = user's action key.
	    if (ListHead->Flags & LIST_FLAGS_CALLOUT) {
		ch = (*ListHead->XlateRtn)(ch, p); // translate as required.
	    }
	}
	switch (ch) {
	    case I_ENTER:		// return selected item.
		if (!(ListHead->Flags & LIST_FLAGS_SELECT)) {
		    break;		// if no permission.
		}
		if (p == NULL) {	// if we can't select item.
		    break;
		}
		WinUnHiliteBorder (ListHead->Window);
		WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		(*ListHead->SelectRtn)(p);
		WinHiliteBorder (ListHead->Window);
		DisplayList (ListHead);
		if (ListHead->Flags & LIST_FLAGS_MENU) {
		    return ACTION_SAVE; // menu routine returns.
		}
		break;

	    case I_F1:			// return with SAVE status.
		if (p != NULL) {
		    WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		}
		return ACTION_SAVE;

	    case I_F2:			// on-line help.
		WinUnHiliteBorder (ListHead->Window);
		WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		(*ListHead->HelpRtn)(ListHead);
		WinHiliteBorder (ListHead->Window);
		DisplayList (ListHead);
		break;

	    case I_ESC:                 // return with ABORT status.
		if (p != NULL) {
		    WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		}
		return ACTION_ABORT;

	    case I_UP:			// move select bar up one line.
		if ((p == NULL) || (p->Baklink == NULL)) {
		    break;		// if empty list, do nothing.
		}
		WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		if (CurLine > 0) {	// if not on top line.
		    CurLine--;
		} else {
		    ListHead->TopElement = ListHead->TopElement->Baklink;
		    DisplayList (ListHead);
		}
		p = p->Baklink;         // p = FWA, previous LISTE.
		break;

	    case I_DOWN:		// move select bar down one line.
		if ((p == NULL) || (p->Fwdlink == NULL)) {
		    break;		// if empty list, do nothing.
		}
		WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		if (CurLine < ListHead->ViewSize-1) { // if not on last line.
		    CurLine++;
		} else {
		    ListHead->TopElement = ListHead->TopElement->Fwdlink;
		    DisplayList (ListHead);
		}
		p = p->Fwdlink;         // p = FWA, next LISTE.
		break;

	    case I_INSERT:
		if (!(ListHead->Flags & LIST_FLAGS_INSERT)) {
		    break;		// if no permission.
		}
		WinUnHiliteBorder (ListHead->Window);
		q = (*ListHead->InsertRtn)();
		WinHiliteBorder (ListHead->Window);
		if (q == NULL) {	// if nothing to insert.
		    break;
		}
		if (p == NULL) {	// if list is currently empty.
		    ListHead->List = q;
		    ListHead->TopElement = q;
		    q->Baklink = NULL;
		    q->Fwdlink = NULL;
		    p = q;
		} else {		// insert us *after* the last line.
		    for (r=ListHead->List; r->Fwdlink!=NULL; r=r->Fwdlink) ;
		    r->Fwdlink = q;	// chain to the end of the list.
		    q->Baklink = r;
		    q->Fwdlink = NULL;	// mark q as the last one on the list.
		}
		DisplayList (ListHead);
		break;

	    case I_DELETE:
		if (!(ListHead->Flags & LIST_FLAGS_DELETE)) {
		    break;		// if no permission.
		}
		if (p == NULL) {
		    break;		// if empty list, do nothing.
		}
		WinUnHiliteBorder (ListHead->Window);
		if (!(*ListHead->DeleteRtn)(p)) {
		    WinHiliteBorder (ListHead->Window);
		    break;		// if he can't delete the object.
		}
		WinHiliteBorder (ListHead->Window);
		WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		if (p->Fwdlink != NULL) {
		    q = p->Fwdlink;
		} else {
		    q = p->Baklink;
		    if (CurLine > 0) {
		       CurLine--;
		    }
		}
		ListDelete (ListHead, p);
		DisplayList (ListHead);
		p = q;
		break;

	    case I_PGUP:
		ch = I_UP;		// repeat this.
		count = ListHead->ViewSize; // do it this many times.
		break;

	    case I_PGDN:
		ch = I_DOWN;		// repeat this.
		count = ListHead->ViewSize; // do it this many times.
		break;

	    case I_HOME:		// move select bar to top of list.
		if (p == NULL) {
		    break;		// if empty list, do nothing.
		}
		WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		ListHead->TopElement = ListHead->List;
		p = ListHead->List;
		CurLine = 0;
		DisplayList (ListHead);
		break;

	    case I_END:
		if (p == NULL) {	// if empty list, do nothing.
		    break;
		}
		WinUnHilite (ListHead->Window, 0, CurLine, ListHead->Window->Width-2);
		p = ListHead->List;
		while (p->Fwdlink != NULL) {
		    p = p->Fwdlink;
		}
		ListHead->TopElement = p;
		CurLine = 0;
		for (i=0; i<ListHead->ViewSize-1; i++) {
		    if (ListHead->TopElement->Baklink == NULL) {
			break;
		    }
		    ListHead->TopElement = ListHead->TopElement->Baklink;
		    CurLine++;
		}
		DisplayList (ListHead);
		break;

	    case I_INVALID:			// if XlateRtn eats char.
		break;

	    default:
		if (p == NULL) {
		    continue;
		}
		c = UpCase (ch);		// c = character to scan for.
		count = 0;
		ch = I_DOWN;			// duplicate this 'count' times.
		for (q=p; q!=NULL; q=q->Fwdlink) {
		    if (UpCase (q->Title [0]) == c) {
			break;
		    }
		    count++;
		}
		if (q == NULL) {		// try reverse direction.
		    count = 0;
		    ch = I_UP;			// duplicate this 'count' times.
		    for (q=p; q!=NULL; q=q->Baklink) {
			if (UpCase (q->Title [0]) == c) {
			    break;
			}
			count++;
		    }
		}
		if (q == NULL) {
		    count = 0;			// reset if not found.
		}
	}
    }
} // ListEdit
