/*
// PROGRAM NAME:  FIELD.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module implements the FIELD object in COW.
//
// MODIFICATION HISTORY.
//	S. E. Jones	91/05/10.	Original.
//	S. E. Jones	92/11/24.	Added noprint strings.
//	S. E. Jones	92/12/30.	Toggle booleans with space bar.
//
// NOTICE:  Copyright (C) 1991-1993 General Software, Inc.
*/

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "cow.h"			// COW system include file.

static UCHAR CursorStr [2] = {219, 0};	// displays block cursor.
static UCHAR far ScratchBuffer [120];	// for use by FieldDisplay.
static UCHAR far SaveBuffer [128];	// buffer for saving string data.

//
// Help texts for field types.
//

static UCHAR * far FieldHelpTextDigits [] = {
    "                   ON-LINE HELP FOR NUMERIC FIELDS",
    " ",
    "The scroll bar (reversed highlighted field on the screen) is",
    "currently positioned over a numeric input field.  This field",
    "will accept a number typed by you with digits from '0' to '9'.",
    "No dollar signs, commas, or decimal points are accepted.",
    " ",
    "If you do not wish to enter a number, simply skip to the next",
    "field in the data entry screen by pressing ENTER, or by using",
    "the UP or DOWN arrow keys, or the HOME/END keys.  You may also",
    "use the TAB or SHIFT-TAB keys to go to another field.",
    " ",
    "If you wish to change the number, you may press the BACKSPACE",
    "key above the ENTER key, or use CTRL-H, to delete the right-",
    "most digit in the number.  New digits appear on the right.",
    " ",
    "When you are finished entering data for the entire data entry",
    "screen, press the F1 key to save your data, or ESC to lose changes.",
    NULL
}; // FieldHelpTextDigits

static UCHAR * far FieldHelpTextString [] = {
    "              ON-LINE HELP FOR ALPHANUMERIC FIELDS",
    " ",
    "The scroll bar (reversed highlighted field on the screen) is",
    "currently positioned over a string input field.  This field",
    "will accept an alphabetic string that includes characters in",
    "the English alphabet, as well as digits, and special symbols",
    "such as the dollar sign, percent sign, and so on."
    " ",
    "If you do not wish to enter a string, simply skip to the next",
    "field in the data entry screen by pressing ENTER, or by using",
    "the UP or DOWN arrow keys, or the HOME/END keys.  You may also",
    "use the TAB or SHIFT-TAB keys to go to another field.",
    " ",
    "If you wish to change the string, you may press the BACKSPACE",
    "key above the ENTER key, or use CTRL-H, to delete the right-",
    "most character in the string.  New type-ins appear on the right.",
    " ",
    "When you are finished entering data for the entire data entry",
    "screen, press the F1 key to save your data, or ESC to lose changes.",
    NULL
}; // FieldHelpTextString

static UCHAR * far FieldHelpTextHex [] = {
    "              ON-LINE HELP FOR HEXADECIMAL FIELDS",
    " ",
    "The scroll bar (reversed highlighted field on the screen) is",
    "currently positioned over a hex string input field.  This field",
    "will accept an alphabetic string that includes only the characters",
    "from '0' to '9', and then 'A' through 'F'.  Hexadecimal strings",
    "form base-16 numbers that are processed by the application.",
    " ",
    "If you do not wish to enter a string, simply skip to the next",
    "field in the data entry screen by pressing ENTER, or by using",
    "the UP or DOWN arrow keys, or the HOME/END keys.  You may also",
    "use the TAB or SHIFT-TAB keys to go to another field.",
    " ",
    "If you wish to change the string, you may press the BACKSPACE",
    "key above the ENTER key, or use CTRL-H, to delete the right-",
    "most character in the string.  New type-ins appear on the right.",
    " ",
    "When you are finished entering data for the entire data entry",
    "screen, press the F1 key to save your data, or ESC to lose changes.",
    NULL
}; // FieldHelpTextHex

static UCHAR * far FieldHelpTextSecurity [] = {
    "              ON-LINE HELP FOR SECURITY FIELDS",
    " ",
    "The scroll bar (reversed highlighted field on the screen) is",
    "currently positioned over a security input field.  This field",
    "will accept an alphabetic string that includes any alphabetic,",
    "numeric, or special characters.  When you type characters, they",
    "are entered internally into a buffer, but only asterisks (*) are",
    "actually displayed on the screen.  This affords security in",
    "the data entry of passwords.",
    " ",
    "If you do not wish to enter a string, simply skip to the next",
    "field in the data entry screen by pressing ENTER, or by using",
    "the UP or DOWN arrow keys, or the HOME/END keys.  You may also",
    "use the TAB or SHIFT-TAB keys to go to another field.",
    " ",
    "If you wish to change the string, you may press the BACKSPACE",
    "key above the ENTER key, or use CTRL-H, to delete the right-",
    "most character in the string.  New type-ins appear on the right.",
    " ",
    "When you are finished entering data for the entire data entry",
    "screen, press the F1 key to save your data, or ESC to lose changes.",
    NULL
}; // FieldHelpTextSecurity

static UCHAR * far FieldHelpTextCurrency [] = {
    "                   ON-LINE HELP FOR CURRENCY FIELDS",
    " ",
    "The scroll bar (reversed highlighted field on the screen) is",
    "currently positioned over a currency input field.  This field",
    "will accept a number typed by you with digits from '0' to '9'.",
    "No dollar signs, commas, or decimal points are accepted, although",
    "the field will automatically justify your data to dollars and",
    "cents with a decimal point.",
    " ",
    "If you do not wish to enter an amount, simply skip to the next",
    "field in the data entry screen by pressing ENTER, or by using",
    "the UP or DOWN arrow keys, or the HOME/END keys.  You may also",
    "use the TAB or SHIFT-TAB keys to go to another field.",
    " ",
    "If you wish to change the amount, you may press the BACKSPACE",
    "key above the ENTER key, or use CTRL-H, to delete the right-",
    "most digit in the number.  New digits appear on the right.",
    " ",
    "When you are finished entering data for the entire data entry",
    "screen, press the F1 key to save your data, or ESC to lose changes.",
    NULL
}; // FieldHelpTextCurrency

static UCHAR * far FieldHelpTextBoolean [] = {
    "                   ON-LINE HELP FOR BOOLEAN FIELDS",
    " ",
    "The scroll bar (reversed highlighted field on the screen) is",
    "currently positioned over a Boolean input field.  This field",
    "will accept either 'Y' or 'N' inputs, allowing you to change",
    "the field from 'YES' to 'NO'.  No other inputs are allowed.",
    " ",
    "If you do not wish to enter a Boolean value, just skip to the",
    "next field in the data entry screen by pressing ENTER, or with",
    "the UP or DOWN arrow keys, or the HOME/END keys.  You may also",
    "use the TAB or SHIFT-TAB keys to go to another field.",
    " ",
    "When you are finished entering data for the entire data entry",
    "screen, press the F1 key to save your data, or ESC to lose changes.",
    NULL
}; // FieldHelpTextBoolean

//
// Routines in this module.
//

VOID FieldDisplay (Field)
    PFIELD Field;
{
    USHORT len, textlen, i;
    UCHAR *p;

    len = strlen (Field->Name);
    WinWrite (Field->Screen->Window, Field->Name,
	      Field->VirtualX, Field->VirtualY, Field->Width);
    switch (Field->Type) {
	case FIELD_TYPE_ULONG:
	    sprintf (ScratchBuffer, "%100lu", (ULONG)Field->Data.UnsignedLong);
	    p = ScratchBuffer+100-(Field->Width-len);
	    break;

	case FIELD_TYPE_LONG:
	    sprintf (ScratchBuffer, "%100ld", (LONG)Field->Data.SignedLong);
	    p = ScratchBuffer+100-(Field->Width-len);
	    break;

	case FIELD_TYPE_STRING:
	case FIELD_TYPE_HEXSTRING:
	    sprintf (ScratchBuffer, "%-100s", (UCHAR *)Field->Data.String);
	    p = ScratchBuffer;
	    break;

	case FIELD_TYPE_NOPRINT_STRING:
	    textlen = strlen (Field->Data.String);
	    for (i=0; i<textlen; i++) {
		ScratchBuffer [i] = '*';	// copy with asterisks.
	    }
	    ScratchBuffer [i] = 0;		// ending zero-byte terminator.
	    p = ScratchBuffer;
	    break;

	case FIELD_TYPE_CURRENCY:
	    sprintf (ScratchBuffer, "%97lu.%02lu",
		     (ULONG)Field->Data.Currency/100L,
		     (ULONG)Field->Data.Currency%100L);
	    p = ScratchBuffer+100-(Field->Width-len);
	    break;

	case FIELD_TYPE_BOOLEAN:
	    sprintf (ScratchBuffer, "%-100s",
		     (BOOLEAN)Field->Data.Boolean ? "YES" : "NO");
	    p = ScratchBuffer;
	    break;

	default:
	    return;			// non-displayable contents.

    }
    WinWrite (Field->Screen->Window, p,
	      Field->VirtualX+len,
	      Field->VirtualY,
	      Field->Width-len);
} // FieldDisplay

VOID DefaultFieldHelpRtn (Field)
    PFIELD Field;
{
    switch (Field->Type) {
	case FIELD_TYPE_ULONG:
	case FIELD_TYPE_LONG:
	    PopupHelp (FieldHelpTextDigits);
	    break;

	case FIELD_TYPE_STRING:
	    PopupHelp (FieldHelpTextString);
	    break;

	case FIELD_TYPE_HEXSTRING:
	    PopupHelp (FieldHelpTextHex);
	    break;

	case FIELD_TYPE_NOPRINT_STRING:
	    PopupHelp (FieldHelpTextSecurity);
	    break;

	case FIELD_TYPE_CURRENCY:
	    PopupHelp (FieldHelpTextCurrency);
	    break;

	case FIELD_TYPE_BOOLEAN:
	    PopupHelp (FieldHelpTextBoolean);
	    break;

	default:
	    return;
    }
} // DefaultFieldHelpRtn

PFIELD FieldCreate (Title, Type, x, y, Width, Flags)
    UCHAR *Title;
    UCHAR Type;
    USHORT x;
    USHORT y;
    USHORT Width;
    USHORT Flags;
{
    PFIELD f;

    f = (PFIELD)malloc (sizeof (FIELD));
    if (f == NULL) {			// if we couldn't get the memory.
	return NULL;
    }
    f->Name = malloc (strlen (Title)+1);
    if (f->Name == NULL) {
	free (f);
	return NULL;
    }
    strcpy (f->Name, Title);
    f->Type = Type;
    switch (Type) {
	case FIELD_TYPE_ULONG:
	    f->Data.UnsignedLong = 0L;
	    break;

	case FIELD_TYPE_LONG:
	    f->Data.SignedLong = 0L;
	    break;

	case FIELD_TYPE_STRING:
	case FIELD_TYPE_HEXSTRING:
	case FIELD_TYPE_NOPRINT_STRING:
	    f->Data.String = malloc (Width+1 > 80 ? Width+1 : 80);
	    if (f->Data.String == NULL) {
		free (f->Name);
		free (f);
		return NULL;
	    }
	    strcpy (f->Data.String, "");
	    break;

	case FIELD_TYPE_CURRENCY:
	    f->Data.Currency = 0L;
	    break;

	case FIELD_TYPE_BOOLEAN:
	    f->Data.Boolean = FALSE;
    }

    f->Flags = Flags;
    f->VirtualX = x;
    f->VirtualY = y;
    f->Width = Width;
    f->Cursor = 0;
    f->HelpRtn = DefaultFieldHelpRtn;
    return f;				// return pointer to field.
} // FieldCreate

VOID FieldDestroy (Field)
    PFIELD Field;
{
    free (Field->Name);

    if ((Field->Type == FIELD_TYPE_STRING) ||
	(Field->Type == FIELD_TYPE_NOPRINT_STRING) ||
	(Field->Type == FIELD_TYPE_HEXSTRING)) {
	free (Field->Data.String);
    }

    free (Field);
} // FieldDestroy

KEY_CODE FieldEdit (Field)
    PFIELD Field;
{
    UCHAR ch, *p, littlebuf [5];
    BOOLEAN SaveBool;
    ULONG SaveLong;
    USHORT len;

    //
    // Save the old data in this field in case he hits ESC to abort the edit.
    //

    switch (Field->Type) {
	case FIELD_TYPE_ULONG:
	case FIELD_TYPE_LONG:
	case FIELD_TYPE_CURRENCY:
	    SaveLong = Field->Data.UnsignedLong;
	    break;

	case FIELD_TYPE_STRING:
	case FIELD_TYPE_HEXSTRING:
	case FIELD_TYPE_NOPRINT_STRING:
	    strcpy (SaveBuffer, Field->Data.String);
	    break;

	case FIELD_TYPE_BOOLEAN:
	    SaveBool = Field->Data.Boolean;
    }

    while (TRUE) {
	FieldDisplay (Field);		// display the field.
	len = strlen (Field->Name);
	WinHilite (Field->Screen->Window, Field->VirtualX+len,
		   Field->VirtualY, Field->Width-len);
	if ((Field->Type == FIELD_TYPE_STRING) ||
	    (Field->Type == FIELD_TYPE_NOPRINT_STRING) ||
	    (Field->Type == FIELD_TYPE_HEXSTRING)) { // display block cursor.
	    len = strlen (Field->Data.String);
	    WinWrite (Field->Screen->Window, CursorStr,
		      Field->VirtualX+strlen (Field->Name)+len,
		      Field->VirtualY, 1);
	}

	ch = KeyRead ();		// ch = user's action key.
	switch (ch) {
	    case I_F2:			// on-line help.
		WinUnHiliteBorder (Field->Screen->Window);
		(*Field->HelpRtn)(Field);
		WinHiliteBorder (Field->Screen->Window);
		break;

	    case I_ENTER:
	    case I_DOWN:
	    case I_UP:
	    case I_LEFT:
	    case I_RIGHT:
	    case I_TAB:
	    case I_BACKTAB:
	    case I_INSERT:
	    case I_HOME:
	    case I_END:
	    case I_F1:
	    case I_F3:
	    case I_F4:
	    case I_F5:
	    case I_F6:
	    case I_F7:
	    case I_F9:
	    case I_F10:
		FieldDisplay (Field);	// update display.
		return ch;		// edit complete.

	    case I_ESC:                 // restore old data value.
		switch (Field->Type) {
		    case FIELD_TYPE_ULONG:
		    case FIELD_TYPE_LONG:
		    case FIELD_TYPE_CURRENCY:
			Field->Data.UnsignedLong = SaveLong; break;

		    case FIELD_TYPE_STRING:
		    case FIELD_TYPE_HEXSTRING:
		    case FIELD_TYPE_NOPRINT_STRING:
			strcpy (Field->Data.String, SaveBuffer); break;

		    case FIELD_TYPE_BOOLEAN:
			Field->Data.Boolean = SaveBool;
		}
		FieldDisplay (Field);	// update display.
		return I_ESC;		// return with field unmodified.

	    case I_BACKSPACE:
	    case I_DELETE:
		switch (Field->Type) {
		    case FIELD_TYPE_ULONG:
		    case FIELD_TYPE_LONG:
		    case FIELD_TYPE_CURRENCY:
			Field->Data.UnsignedLong /= 10L;
			break;

		    case FIELD_TYPE_STRING:
		    case FIELD_TYPE_HEXSTRING:
		    case FIELD_TYPE_NOPRINT_STRING:
			p = Field->Data.String;
			if (strlen (p) == 0) {
			    break;
			}
			*(p+strlen (p)-1) = 0;	// truncate the string.
		}
		break;

	    default:
		switch (Field->Type) {
		    case FIELD_TYPE_ULONG:
		    case FIELD_TYPE_LONG:
		    case FIELD_TYPE_CURRENCY:
			if (isdigit (ch)) {
			    Field->Data.UnsignedLong =
				(Field->Data.UnsignedLong*10L) + (ULONG)(ch-'0');
			}
			break;

		    case FIELD_TYPE_STRING:
		    case FIELD_TYPE_NOPRINT_STRING:
			len = strlen (Field->Data.String);
			if (len + strlen (Field->Name) >= Field->Width-1) {
			    break;		// can't make string bigger.
			}
			littlebuf [0] = ch;	// make little string.
			littlebuf [1] = 0;
			strcat (Field->Data.String, littlebuf);
			break;

		    case FIELD_TYPE_HEXSTRING:
			ch = toupper (ch);

			//
			// Filter-out all non-hex characters.
			//

			if ((ch >= '0') && (ch <= '9')) {
			} else if ((ch >= 'A') && (ch <= 'F')) {
			} else {
			    break;
			}
			len = strlen (Field->Data.String);
			if (len + strlen (Field->Name) >= Field->Width-1) {
			    break;		// can't make string bigger.
			}
			littlebuf [0] = ch;	// make little string.
			littlebuf [1] = 0;
			strcat (Field->Data.String, littlebuf);
			break;

		    case FIELD_TYPE_BOOLEAN:
			if (ch == 0x20) {
			    Field->Data.Boolean = !Field->Data.Boolean;
			} else if (toupper (ch) == 'Y') {
			    Field->Data.Boolean = TRUE;
			} else if (toupper (ch) == 'N') {
			    Field->Data.Boolean = FALSE;
			}
		}
	}
    }
} // FieldEdit
