------------------------------------------------------------------
--            Run-time Kernel Test Support Subprograms          --
------------------------------------------------------------------

package test_support is

  procedure gp_fault;
  procedure control_c;
  procedure timer;
  procedure video;

end test_support;

with program_control;
with rtkernel;
with system;
with text_io;   use text_io;
with unchecked_conversion;

package body test_support is

  count : integer;              -- These must be global variables,
  stop  : boolean;              --  defined in a package body, because
                                --  they are referenced by interrupt
                                --  handler code.

  saved_context : rtkernel.stack_context;
  int_stack     : array(1 .. 512) of character;
                                -- Interrupt stack for control-c
                                --  handler and stack save area.

                                -- Need to suppress stack checking
                                --  in interrupt handler subprograms
  pragma suppress(storage_check);

  procedure terminate_program is
  begin
    program_control.quit(1);
  end;

  ----------------------------------------------------------------
  --     General Protection Fault Exception Handler Example     --
  ----------------------------------------------------------------

  procedure gp_handler is
  begin
    rtkernel.rtk_restore_context;

    put_line("GP fault caught, execution terminated");
    terminate_program;
  end;

  procedure gp_fault is
    type chp is access character;

    vch  : character;
    vchp : chp;

    function cnv_addr_chp is new unchecked_conversion(system.address, chp);
  begin
    rtkernel.rtk_set_exception_vector(rtkernel.exc_general_protection_fault,
                                      gp_handler'address);
    vch   := 'A';
    put("1 vch = """);  put(vch);  put_line(""" = A ?");

    vchp     := vch'addr;
    vchp.all := 'Z';
    put("2 vch = """);  put(vch);  put_line(""" = Z ?");

    put_line("Should get a GP fault now");

    vchp  := cnv_addr_chp(system."-"(1));
    vchp.all := 'X';
    put("3 vch = """);  put(vch);  put_line(""" = X ?");
  end gp_fault;

  ----------------------------------------------------------------
  --              Control-C Interrupt Handler Example           --
  ----------------------------------------------------------------

  procedure control_c_handler is
  begin
    rtkernel.rtk_save_regs_and_restore_context;
    rtkernel.rtk_switch_stack(int_stack'address, int_stack'length,
                              saved_context);
                                -- Switch to local stack which is
                                --  addressable within the 32-bit
                                --  "flat" memory model because
                                --  program_control.quit (called by
                                --  terminate_program) does stack
                                --  overflow checking and passes
                                --  local variables by reference.
                                --  This call and the call to
                                --  rtk_restore_stack below are not
                                --  necessary for 16-bit programs.
    count := count + 1;

    if count >= 3 then
      put_line("Control-C caught, execution terminated");
      terminate_program;
    end if;

    put_line("Control-C caught, execution continued");

    rtkernel.rtk_restore_stack(saved_context);
    rtkernel.rtk_restore_regs_and_iret;
  end control_c_handler;

  procedure control_c is
  begin
    count := 0;
    rtkernel.rtk_set_interrupt_vector(rtkernel.int_control_c,
                                      control_c_handler'address);
    for i in 1 .. 500 loop
      put_line("Type Control-C now  " & integer'image(count) &
               " - " & integer'image(i));
    end loop;
  end control_c;

  ----------------------------------------------------------------
  --               Timer Interrupt Handler Example              --
  ----------------------------------------------------------------

  procedure stop_handler is
  begin
    rtkernel.rtk_save_regs_and_restore_context;

    stop := true;

    rtkernel.rtk_restore_regs_and_iret;
  end stop_handler;

  procedure timer_handler is
  begin
    rtkernel.rtk_save_regs_and_restore_context;

    count := count + 1;

    rtkernel.rtk_restore_regs_and_iret;
  end timer_handler;

  procedure timer is
    old_handler : system.address;
  begin
    count := 0;
    rtkernel.rtk_set_interrupt_vector(rtkernel.int_control_c,
                                      stop_handler'address);
    old_handler := rtkernel.rtk_get_interrupt_vector(rtkernel.int_timer);
    rtkernel.rtk_set_interrupt_vector(rtkernel.int_timer,
                                      timer_handler'address);

    stop := false;
    while not stop and then count < 182 loop
      put_line("Count = " & integer'image(count));
    end loop;

    rtkernel.rtk_set_interrupt_vector(rtkernel.int_timer, old_handler);
  end timer;

  ----------------------------------------------------------------
  --              Video Memory Buffer Access Example            --
  ----------------------------------------------------------------

  procedure video is
    max_cells : constant := 25 * 80;

    -- type vid_cell is
    --        record
    --          ch   : byte_integer;
    --          attr : byte_integer;
    --        end record;

    subtype vid_cell is rtkernel.word;

    type screen  is array(1 .. max_cells) of vid_cell;
    type screenp is access screen;

    sp_sel : rtkernel.selector32;
    sp_ptr : rtkernel.logical_pointer;
    sp     : screenp;
    res    : rtkernel.result;

    function cnv_addr_scrp is new unchecked_conversion(system.address, screenp);

    function get_cell(ptr : rtkernel.logical_pointer) return vid_cell is
    begin
      return rtkernel.rtk_peek_word(sp_sel, ptr);
    end;

    procedure set_cell(ptr : rtkernel.logical_pointer; cell : vid_cell) is
    begin
      rtkernel.rtk_poke_word(cell, sp_sel, ptr);
    end;

    procedure flip is
      pragma suppress(access_check);
                                -- Must suppress access check because
                                --  for 32-bit programs, sp may have
                                --  a value = null which in this case
                                --  is a legal value since it points
                                --  to the beginning of a different
                                --  segment.
      cell : vid_cell;
    begin
      for i in screen'first .. (screen'last / 2) loop
        -- cell                    := sp(i);
        -- sp(i)                   := sp(screen'last - i + 1);
        -- sp(screen'last - i + 1) := cell;

        cell := get_cell(sp(i)'address);
        set_cell(sp(i)'address, get_cell(sp(screen'last - i + 1)'address));
        set_cell(sp(screen'last - i + 1)'address, cell);
      end loop;
    end flip;

  begin
    rtkernel.rtk_map_physical_memory(16#B8000#, screen'size / 8,
                                     sp_sel,    sp_ptr,
                                     res);
    if res /= rtkernel.dos_normal then
      put_line("rtk_map_physical_memory failed");
      terminate_program;
    end if;
    sp := cnv_addr_scrp(sp_ptr);

    flip;
    skip_line;
    flip;
    new_line;
  end video;

end test_support;

------------------------------------------------------------------
--              Run-time Kernel Test Driver Program             --
------------------------------------------------------------------

with test_support;
with text_io;           use text_io;
procedure testrtk is
  line : string(1 .. 80);
  last : natural;
begin
  put_line("Test start");
  put("Test (g)eneral protection, (c)ontrol-c, (t)imer or (v)ideo? ");
  get_line(line, last);

  if last >= line'first then
    case line(line'first) is
      when 'g'    => test_support.gp_fault;
      when 'c'    => test_support.control_c;
      when 't'    => test_support.timer;
      when 'v'    => test_support.video;
      when others => null;
    end case;
  end if;

  put_line("Test done");
end testrtk;
