------------------------------------------------------------------
--      rtkernel -- DOS Run-Time Kernel Interface               --
------------------------------------------------------------------

------------------------------------------------------------------
--        The following software is the sole property of        --
--           Meridian Software Systems, Incorporated            --
--     containing its proprietary, confidential information.    --
--                                                              --
--      Copyright (c) 1993 Meridian Software Systems, Inc.      --
------------------------------------------------------------------

with system;

package rtkernel is

  -- This package serves to isolate the major differences between
  -- the real-mode DOS kernel and various protected-mode DOS-
  -- extender kernels that Ada application programs commonly have to
  -- deal with.  Only those functions which are used by the Meridian
  -- Ada run-time and its various libraries are implemented here.
  -- There are many additional services provided by the DOS-extenders
  -- that are not included here because they have yet to prove their
  -- utility.  It is very desirable to limit the number of services
  -- provided to the smallest number to ensure the greatest portability.
  -- In a number of cases, subprograms and parameters, which are
  -- required for 32-bit programs, are included in the 16-bit versions
  -- as well without change to improve portability.

                                -- Need to suppress package
                                --  elaboration checks in interrupt
                                --  handler support subprograms
  pragma suppress(elaboration_check);

                                -- Need to suppress stack checking
                                --  in interrupt handler support
                                --  subprograms
  pragma suppress(storage_check);

                                -- Return codes (subset of possible values).
                                --  Associated with type result.
  dos_normal                     : constant :=  0;
  dos_err_general_failure        : constant := 31;
                                -- Function not supported

                                -- Exceptions which can be intercepted
                                --  with rtk_set_exception_vector
  exc_divide_by_zero             : constant :=  0;
  exc_debug                      : constant :=  1;
  exc_breakpoint                 : constant :=  3;
  exc_overflow_detected          : constant :=  4;
  exc_bound_exceeded             : constant :=  5;
  exc_invalid_opcode             : constant :=  6;
  exc_no_processor_extension     : constant :=  7;
  exc_double_exception_detected  : constant :=  8;
  exc_invalid_task_state_segment : constant := 10;
  exc_segment_not_present        : constant := 11;
  exc_stack_segment_fault        : constant := 12;
  exc_general_protection_fault   : constant := 13;
  exc_page_fault                 : constant := 14;
  exc_floating_point_error       : constant := 16;
  exc_alignment_check            : constant := 17;

                                -- Interrupts which can be intercepted
                                --  with rtk_set_interrupt_vector
                                --  (subset of possible values)
  int_timer                      : constant := 16#1C#;
  int_control_c                  : constant := 16#23#;

                                -- Values returned by rtk_get_kernel_type
  type    run_time_kernel_types is
            ( rtk_dos,          -- Real-mode DOS kernel
              rtk_ergo_dpm16,   -- 16-bit protected-mode Ergo DPMI kernel
              rtk_ergo_dpm32    -- 32-bit protected-mode Ergo DPMI kernel
            );

  type    byte is range 0 .. 2**8 - 1;
  for     byte'size use 8;

  type    word is range 0 .. 2**16 - 1;
  for     word'size use 16;

  subtype dword is long_integer;

  subtype selector32 is word;   -- Only used by 32-bit programs to
                                --  identify the selector in a 48-bit
                                --  address.

  subtype logical_pointer is system.address;
                                -- An arbitrary pointer into memory.
                                --  For 16-bit programs it is the full
                                --  address.  For 32-bit programs it is
                                --  only the offset portion of a full
                                --  48-bit address and unless provided
                                --  with a selector32, is assumed to
                                --  point into the main program segment
                                --  (that is, it assumes the flat
                                --  memory model).

  subtype vector is system.address;
                                -- An address which points to a piece
                                --  of code.

  subtype result is integer;    -- The result code for many subprograms.

  type    real_registers is     -- Register block used to set/read registers
            record              --  for procedure rtk_issue_real_interrupt.
              es    : word;     --  Note that these are always 16-bit values.
              ds    : word;
              di    : word;
              si    : word;
              bp    : word;
              sp    : word;
              bx    : word;
              dx    : word;
              cx    : word;
              ax    : word;
              ip    : word;
              cs    : word;
              flags : word;
            end record;

  type    register_block is     -- Register block used to access saved
            record              --  registers with function rtk_get_regs.
              es    : integer;  --  Note that these are 16-bit values for
              ds    : integer;  --  16-bit programs and 32-bit values for
              di    : integer;  --  32-bit programs.
              si    : integer;
              bp    : integer;
              sp    : integer;
              bx    : integer;
              dx    : integer;
              cx    : integer;
              ax    : integer;
              ip    : integer;
              cs    : integer;
              flags : integer;
            end record;

  type    register_block_p is access register_block;

  type    stack_context is      -- Save area used to switch to a different
            record              --  interrupt stack.
              ss  : integer;    -- Saved values of SS and SP registers when
              sp  : integer;    --  the interrupt handler code was entered.
              tos : integer;    -- Saved value used to perform stack
            end record;         --  overflow checking on.

  function  rtk_get_kernel_type return run_time_kernel_types;
                                -- Get type of run-time kernel currently
                                --  running.

  function  rtk_get_interrupt_vector(number : byte) return vector;
                                -- Return current vector address associated
                                --  with interrupt <number>.  It should NOT
                                --  be used with the exceptions enumerated
                                --  above.  For protected-mode kernels this
                                --  is the protected-mode interrupt vector
                                --  and is suitable for restoring an interrupt
                                --  vector back to its original state after
                                --  changing it with rtk_set_interrupt_vector.

  procedure rtk_set_interrupt_vector(number : byte; handler : vector);
                                -- Set the vector address associated with
                                --  interrupt <number> to <handler>.  It
                                --  should NOT be used with the exceptions
                                --  enumerated above.  For protected-mode
                                --  kernels this affects both the real and
                                --  protected-mode interrupt vector.  That is,
                                --  the interrupt will be vectored to the
                                --  protected-mode <handler> no matter what
                                --  mode (real or protected) the system is
                                --  in when the interrupt occurs.

  function  rtk_get_exception_vector(number : byte) return vector;
                                -- Return current vector address associated
                                --  with exception <number>.  It should be
                                --  used ONLY with the exceptions enumerated
                                --  above.  In real-mode, this function is
                                --  identical to rtk_get_interrupt_vector.
                                --  In protected-mode, exceptions are not
                                --  the same as interrupts (whether hardware
                                --  or software generated).

  procedure rtk_set_exception_vector(number : byte; handler : vector);
                                -- Set the vector address associated with
                                --  exception <number> to <handler>.  It
                                --  should be used ONLY with the exceptions
                                --  enumerated above.  In real-mode, this
                                --  function is identical to
                                --  rtk_set_interrupt_vector.

  function  rtk_lock_memory(ptr  : logical_pointer;
                            size : dword) return result;
                                -- Locks the memory area starting at
                                --  logical address <ptr> and <size>
                                --  bytes long into RAM thereby preventing
                                --  it from being swapped out to disk when
                                --  used with a kernel that supports
                                --  virtual memory management.  This is
                                --  only a concern for interrupt handlers
                                --  since any code or data they access at
                                --  interrupt time must be in RAM.
                                --  If successful, this function returns
                                --  dos_normal;  any other value means
                                --  failure.

                                -- NOTE: interrupt/exception service routines
                                --  and the global data segment are
                                --  automatically locked when the associated
                                --  vector is set, however, any data
                                --  allocated from the heap is not locked.

  function  rtk_get_real_interrupt(number : byte) return dword;
                                -- Return current real-mode vector address
                                --  associated with interrupt <number>.  It
                                --  should NOT be used with the exceptions
                                --  enumerated above.  In real-mode, this
                                --  function is identical to
                                --  rtk_get_interrupt_vector.

  procedure rtk_issue_real_interrupt(number    :     byte;
                                     reg_block : out real_registers;
                                     res       : out result);
                                -- Switch to real-mode and issue interrupt
                                --  <number> after loading the registers
                                --  with the values in <reg_block>.  After
                                --  the interrupt handler returns, save the
                                --  current register values in <reg_block>.
                                --  If successful, <res> will be
                                --  dos_normal;  any other value means
                                --  failure.

  procedure rtk_alloc_real_memory(paragraphs   :     word;
                                  real_segment : out word;
                                  sel          : out selector32;
                                  ptr          : out logical_pointer;
                                  max_paras    : out word;
                                  res          : out result);
                                -- Allocate <paragraphs> 16-byte paragraphs
                                --  from "real" memory (the first megabyte
                                --  of RAM that is accessible to real-mode
                                --  programs and DOS itself).  Return the
                                --  logical real-mode segment number of the
                                --  allocated memory in <real_segment>.  For
                                --  16-bit programs, the 32-bit logical
                                --  address (which is meaningful to the
                                --  calling program) is returned in <ptr>
                                --  (<sel> is meaningless).  For 32-bit
                                --  programs, the full 48-bit logical address
                                --  is returned in <sel> and <ptr>, both of
                                --  which must be used to reference the area.
                                --  If the request cannot be completed due to
                                --  lack of available memory, put the number
                                --  of paragraphs of the largest free block in
                                --  <max_paras>.
                                --  If successful, <res> will be dos_normal;
                                --  any other value means failure.

  procedure rtk_map_physical_memory(physical_address :     dword;
                                    size             :     dword;
                                    sel              : out selector32;
                                    ptr              : out logical_pointer;
                                    res              : out result);
                                -- Map the RAM memory starting at the
                                --  physical byte address <physical_address>
                                --  and <size> bytes long into the logical
                                --  address space of the calling program.
                                --  Return the associated logical address in
                                --  <address>.  For 16-bit programs, return
                                --  the 32-bit logical address in <ptr>
                                --  (<sel> is meaningless).  For 32-bit
                                --  programs, return the full 48-bit logical
                                --  address in <sel> and <ptr>, both of which
                                --  must be used to reference the area.
                                --  In real-mode, <physical_address> cannot be
                                --  greater than 16#FFFFF# (the first megabyte
                                --  of RAM).  Note, that this request does not
                                --  allocate any memory, it just acquires the
                                --  ability to access it.
                                --  If successful, <res> will be dos_normal;
                                --  any other value means failure.

  procedure rtk_disable_interrupts;
                                -- Disable hardware interrupts

  procedure rtk_enable_interrupts;
                                -- Enable hardware interrupts

                                -- The following three procedures are only
                                --  intended to be used within interrupt/
                                --  exception handlers.  These handlers
                                --  cannot have any compiler-generated
                                --  prolog code in them that will be
                                --  executed before the rtk_save_regs
                                --  procedure is called.  In particular,
                                --  elaboration and stack checking must be
                                --  disabled within handlers by using
                                --  appropriate compiler options (ada -fs)
                                --  or pragmas (suppress).  Also, handlers
                                --  may not delcare any local variables.

  procedure rtk_save_regs_and_restore_context;
                                -- Save all the general-purpose registers
                                --  of the processor on top of the caller's
                                --  stack frame.  Floating point registers
                                --  are not saved.  Also restore whatever
                                --  registers are necessary to enable the
                                --  program to access its data (typically
                                --  the DS register).

  procedure rtk_restore_context;
                                -- Restore whatever registers are necessary
                                --  to enable the program to access its data
                                --  (typically the DS register).  This
                                --  must be done before any global data
                                --  structures are accessed by a handler.

  function  rtk_get_regs return register_block_p;
                                -- Return a pointer to the stack area where
                                --  the registers were saved by
                                --  rtk_save_regs_and_restore_context.
                                --  Any changes to these values (except for
                                --  sp) will be loaded into the hardware
                                --  registers when rtk_restore_regs_and_iret
                                --  is called.

  procedure rtk_restore_regs_and_iret;
                                -- Restore the general-purpose registers
                                --  that were saved by the
                                --  rtk_save_regs_and_restore_context
                                --  procedure which must have been called
                                --  at the very beginning of the caller of
                                --  this procedure.  After restoring the
                                --  registers, issue a return from interrupt
                                --  instruction, IRET.

  procedure rtk_switch_stack(stack_base    : in  logical_pointer;
                             stack_size    : in  integer;
                             saved_context : out stack_context);
                                -- Save the current run-time stack context
                                --  (SS and SP registers and the current
                                --  stack limit used to check for stack
                                --  overflow) in <saved_context> and then
                                --  switch to the new stack area whose lowest
                                --  numbered address is <stack_base> and
                                --  whose length is <stack_size> bytes.
                                --  The new stack should be at least 150 bytes
                                --  long and it must be located in either the
                                --  global data segment (DS) or allocated
                                --  from the heap (with new).  It is the
                                --  user's responsibility to ensure that the
                                --  same interrupt stack will not be used in
                                --  more than one interrupt context at a time.
                                --  Although this procedure can be used by
                                --  16-bit programs, its intended use is by
                                --  32-bit interrupt handlers.  In this case,
                                --  the stack must be switched if any of the
                                --  subprograms the interrupt handler calls
                                --  do stack overflow checking or pass local
                                --  variables by reference.  This is due to
                                --  the fact that Meridian Ada only supports
                                --  the "flat" memory model for 32-bit programs
                                --  and the interrupt stack set up by the
                                --  DOS extender kernel is not located in the
                                --  "flat" address space.  Therefore, it is
                                --  necessary to switch to a new stack which
                                --  is addressable within the "flat" model.

  procedure rtk_restore_stack(saved_context : in stack_context);
                                -- Restore the run-time stack context that
                                --  was saved by a previous call to
                                --  rtk_switch_stack.

  function  rtk_peek_byte(sel : selector32; ptr : logical_pointer) return byte;
                                -- Return the byte value at <sel>:<ptr>.
                                --  <Sel> is ignored for 16-bit programs.

  function  rtk_peek_word(sel : selector32; ptr : logical_pointer) return word;
                                -- Return the (always 16-bit) word value at
                                --  <sel>:<ptr>.
                                --  <Sel> is ignored for 16-bit programs.

  procedure rtk_poke_byte(value : byte;
                          sel   : selector32;
                          ptr   : logical_pointer);
                                -- Set the byte value located at <sel>:<ptr>
                                --  to <value>.
                                --  <Sel> is ignored for 16-bit programs.

  procedure rtk_poke_word(value : word;
                          sel   : selector32;
                          ptr   : logical_pointer);
                                -- Set the word value located at <sel>:<ptr>
                                --  to <value>.
                                --  <Sel> is ignored for 16-bit programs.

  procedure rtk_move_bytes(srcsel : selector32;
                           srcptr : logical_pointer;
                           dstsel : selector32;
                           dstptr : logical_pointer;
                           len    : dword);
                                -- Copy <len> bytes from logical address
                                --  <srcsel>:<srcptr> to <dstsel>:<dstptr>.
                                --  <Srcsel> and <dstsel> are ignored for
                                --  16-bit programs.

  function  rtk_code_selector return selector32;
                                -- Returns the current value of the CS
                                --  register from the caller's context.

  function  rtk_data_selector return selector32;
                                -- Returns the current value of the DS
                                --  register.

  procedure rtk_get_segment_limit(sel   :     selector32;
                                  limit : out dword;
                                  res   : out result);
                                -- Return in <limit> the number of bytes of
                                --  memory allocated to selector <sel>.  This
                                --  procedure is only meaningful for protected
                                --  mode programs.
                                --  If successful, <res> will be dos_normal;
                                --  any other value means failure.

private

  pragma interface( builtin, rtk_get_kernel_type               );
  pragma interface( builtin, rtk_get_interrupt_vector          );
  pragma interface( builtin, rtk_set_interrupt_vector          );
  pragma interface( builtin, rtk_get_exception_vector          );
  pragma interface( builtin, rtk_set_exception_vector          );
  pragma interface( builtin, rtk_lock_memory                   );
  pragma interface( builtin, rtk_get_real_interrupt            );
  pragma interface( builtin, rtk_issue_real_interrupt          );
  pragma interface( builtin, rtk_alloc_real_memory             );
  pragma interface( builtin, rtk_map_physical_memory           );
  pragma interface( builtin, rtk_disable_interrupts            );
  pragma interface( builtin, rtk_enable_interrupts             );
  pragma interface( builtin, rtk_save_regs_and_restore_context );
  pragma interface( builtin, rtk_restore_context               );
  pragma interface( builtin, rtk_get_regs                      );
  pragma interface( builtin, rtk_restore_regs_and_iret         );
  pragma interface( builtin, rtk_switch_stack                  );
  pragma interface( builtin, rtk_restore_stack                 );
  pragma interface( builtin, rtk_peek_byte                     );
  pragma interface( builtin, rtk_peek_word                     );
  pragma interface( builtin, rtk_poke_byte                     );
  pragma interface( builtin, rtk_poke_word                     );
  pragma interface( builtin, rtk_move_bytes                    );
  pragma interface( builtin, rtk_code_selector                 );
  pragma interface( builtin, rtk_data_selector                 );
  pragma interface( builtin, rtk_get_segment_limit             );

end rtkernel;
