--        ͻ
--        ۺ
--                                                                   ۺ
--                         Meridian Software Systems                 ۺ
--                                                                   ۺ
--                            Copyright (C)  1990                    ۺ
--                                                                   ۺ
--                            ALL RIGHTS RESERVED                    ۺ
--                                                                   ۺ
--        ۺ
--        ͼ

------------------------------------------------------------------------------
--
--  Unit Name :  graph_Init - package body
--
--  Purpose of unit:  This package initializes and closes the Ada Graphics
--                    Utility Library.  It must be called before any other
--                    graphics procedure or function, but may only be called
--                    once within an application program.  The Initiate_Graph
--                    procedure selects the video page 1 as the active screen
--                    and erases it.  All windows and worlds are initialized
--                    to the graphic screen coordinates.  This procedure will
--                    also turn the graphics mode off and return the system 
--                    to the text mode which was active before Initiate_Graph
--                    was called.
--
------------------------------------------------------------------------------


with COMMON_GRAPHIC_TYPES, COMMON_DISPLAY_TYPES;
with DISK, DIRECTORY, INTERRUPT, TTY;
with PROGRAM_CONTROL, BIT_OPS;
use  PROGRAM_CONTROL, BIT_OPS;

package body GRAPH_INIT  is

  VIDEO_DESIRED  : COMMON_GRAPHIC_TYPES.VIDEO_SYSTEM;
  VIDEO_NUMBER   : integer;
  VIDEO_MEMORY   : integer;
  SYSTEM_COLOR   : boolean := false;
  BAD_CARD       : boolean := false;
  REG_FILE       : INTERRUPT.REGISTERS;
  HIGH_REG_CLEAR : constant integer := 16#00FF#;    -- format of variable 0000_0000_1111_1111

  --ͻ

  procedure CLEAR_SCREEN is
    -- Load function 06h into the AH register and clear the video screen.
  begin
    if COMMON_GRAPHIC_TYPES.GRAPH_SCREEN <= 6 then
      REG_FILE.BX := 16#0007#;
    else
      REG_FILE.BX := 16#0000#;
    end if;

    REG_FILE.AX := 16#0600#;
    REG_FILE.DX := 16#184F#;

    -- Call interrupt 10h
    INTERRUPT.VECTOR (ON             => 16#10# ,
		      REGISTER_BLOCK => REG_FILE );
  end CLEAR_SCREEN;

  --ͼ

  --ͻ

  procedure CLOSE_GRAPH is
    -- This procedure will set the computer system back to video mode prior
    -- to initializing the graphic system.
  begin
    -- Load old video mode into the AX register
    REG_FILE.AX := COMMON_GRAPHIC_TYPES.VIDEO_SYSTEM'pos(COMMON_GRAPHIC_TYPES.TEXT_SCREEN);
    COMMON_GRAPHIC_TYPES.GRAPH_SCREEN := REG_FILE.AX;

    -- Call interrupt 10h
    INTERRUPT.VECTOR (ON             => 16#10#,
		      REGISTER_BLOCK => REG_FILE );
  end CLOSE_GRAPH;

  --ͼ

  --ͻ

  procedure DETECT_GRAPHIC_CARD is
    TEMP_NUMBER : integer;
  begin
    -- Load 0Fh into the AH register to determine video mode prior to
    -- intialization.
    REG_FILE.AX := 16#0F00#;

    -- Call interrupt 10h
    INTERRUPT.VECTOR (ON             => 16#10#,
		      REGISTER_BLOCK => REG_FILE );

    -- Clear AH and assign temp_number to AL (video mode number).
    TEMP_NUMBER :=  REG_FILE.AX and HIGH_REG_CLEAR;

    -- Assign text_screen to old video mode name.
    COMMON_GRAPHIC_TYPES.TEXT_SCREEN  := COMMON_GRAPHIC_TYPES.VIDEO_SYSTEM'val(TEMP_NUMBER);
    COMMON_GRAPHIC_TYPES.GRAPH_SCREEN := TEMP_NUMBER;
  end DETECT_GRAPHIC_CARD;

  --ͼ

  --ͻ

  procedure SET_GRAPHIC_CARD is
  begin
    if VIDEO_NUMBER >= 13 then

      -- Load 1200h into the AX register to determine the video memory for
      -- the video board installed.  Register BX, 10h, requests function
      -- 12h to return the video configuration information.  The return of 
      -- reg BH indicates whether color or monochrome is the default video
      -- mode and BL indicates the amount of memory either on the video board
      -- or allocated to the video board.
      REG_FILE.AX := 16#1200#;
      REG_FILE.BX := 16#0010#;

      -- Call interrupt 10h
      INTERRUPT.VECTOR(ON             => 16#10#,
		       REGISTER_BLOCK => REG_FILE );

      case REG_FILE.BX is
	when   0 =>  SYSTEM_COLOR := true;   VIDEO_MEMORY :=  64;
	when   1 =>  SYSTEM_COLOR := true;   VIDEO_MEMORY := 128;
	when   2 =>  SYSTEM_COLOR := true;   VIDEO_MEMORY := 192;
	when   3 =>  SYSTEM_COLOR := true;   VIDEO_MEMORY := 256;
	when 256 =>  SYSTEM_COLOR := false;  VIDEO_MEMORY :=  64;
	when 257 =>  SYSTEM_COLOR := false;  VIDEO_MEMORY := 128;
	when 258 =>  SYSTEM_COLOR := false;  VIDEO_MEMORY := 192;
	when 259 =>  SYSTEM_COLOR := false;  VIDEO_MEMORY := 256;

        when others => null;
      end case;
    end if;  

    if (VIDEO_MEMORY = 64) and (VIDEO_NUMBER = 16) then
      VIDEO_NUMBER := 14;
    end if;

    -- Load function 00h into the AH register and the video mode number into
    -- the AL register to set either AGUL default or the application 
    -- programmer's desired video mode.

    COMMON_GRAPHIC_TYPES.GRAPH_SCREEN := VIDEO_NUMBER;
    REG_FILE.AX                       := VIDEO_NUMBER;

    -- Call interrupt 10h
    INTERRUPT.VECTOR (ON             => 16#10# ,
		      REGISTER_BLOCK => REG_FILE );

    CLEAR_SCREEN;
  end SET_GRAPHIC_CARD;

  --ͼ

  --ͻ

  procedure SET_GLOBAL_VARIABLES is
    BREAK_ON              : PROGRAM_CONTROL.STATE_TYPE;
    PAGE1_START_MEMORY,
    PAGE1_END_MEMORY,
    PAGE2_START_MEMORY,
    PAGE2_END_MEMORY      : COMMON_GRAPHIC_TYPES.AGUL_MEMORY_LOCATION;
  begin

    case VIDEO_NUMBER is

      when 4 | 5 => 
        --  Color Graphic Adapter Board
        --  4 = 320 x 200 4-color graphics
        --  5 = 320 x 200 4-grey graphics

        COMMON_GRAPHIC_TYPES.ASPECT_RATIO  := 1.20;
        COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := false;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X := 319;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y := 199;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_START := 16#B800_0000# - 16#1_0000_0000#;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_END   := 16#3FFF_0000#;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_START := 16#B800_0000# - 16#1_0000_0000#;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_END   := 16#3FFF_0000#;

      when 6 => 
        --  Color Graphic Adapter Board
        --  6 = 640 x 200 2-color graphics

        COMMON_GRAPHIC_TYPES.ASPECT_RATIO  := 2.4;
        COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := false;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X := 639;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y := 199;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_START := 16#B800_0000# - 16#1_0000_0000#;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_END   := 16#3FFF_0000#;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_START := 16#B800_0000# - 16#1_0000_0000#;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_END   := 16#3FFF_0000#;

      when 13 => 
        --  Enhanced Graphics Adapter & Video Graphics Array Board
        --  13 = 320 x 200 16-color graphics

        COMMON_GRAPHIC_TYPES.ASPECT_RATIO  := 1.2;
        COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := true;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X := 319;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y := 199;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_START := 16#A000_0000# - 16#1_0000_0000#;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_END   := 16#7FFF_0000#;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_START := 16#A800_0000# - 16#1_0000_0000#;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_END   := 16#FFFF_0000# - 16#1_0000_0000#;

      when 14 => 
        --  Enhanced Graphics Adapter & Video Graphics Array Board
        --  14 = 640 x 200 16-color graphics

        if VIDEO_MEMORY = 64 then
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := false;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        else
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := true;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#B000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        end if;
        COMMON_GRAPHIC_TYPES.ASPECT_RATIO  := 2.40;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X := 639;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y := 199;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_START := PAGE1_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_END   := PAGE1_END_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_START := PAGE2_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_END   := PAGE2_END_MEMORY;

      when 15 => 
        --  Enhanced Graphics Adapter & Video Graphics Array Board
        --  15 = 640 x 350 monochrome graphics

        if VIDEO_MEMORY = 64 then
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := false;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        else
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := true;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#B000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        end if;
        COMMON_GRAPHIC_TYPES.ASPECT_RATIO  := 1.26;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X := 639;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y := 349;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_START := PAGE1_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_END   := PAGE1_END_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_START := PAGE2_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_END   := PAGE2_END_MEMORY;

      when 16 =>
        --  Enhanced Graphics Adapter & Video Graphics Array Board
        --  16 = 640 x 350 16-color graphics (128kb)
        --  16 = 640 x 350 4-color graphics (64kb EGA)

        if VIDEO_MEMORY = 64 then
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := false;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        else
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := true;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#7FFF_0000#;
	  PAGE2_START_MEMORY := 16#A800_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#7FFF_0000#;
        end if;
        COMMON_GRAPHIC_TYPES.ASPECT_RATIO  := 1.37;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X := 639;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y := 349;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_START := PAGE1_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_END   := PAGE1_END_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_START := PAGE2_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_END   := PAGE2_END_MEMORY;

      when 17 =>
        --  MultiColor and Video Graphics Array (MCGA and VGA)
        --  17 = 640 x 480 2-color graphics

        if VIDEO_MEMORY = 64 then
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := false;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        else
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := true;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#B000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        end if;
        COMMON_GRAPHIC_TYPES.ASPECT_RATIO  := 1.00;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X := 639;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y := 479;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_START := PAGE1_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_END   := PAGE1_END_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_START := PAGE2_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_END   := PAGE2_END_MEMORY;

      when 18 => 
        --  Video Graphics Array  -  640 x 480 16-color graphics

        if VIDEO_MEMORY = 64 then
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := false;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        else
          COMMON_GRAPHIC_TYPES.SCREEN_FLAG   := true;
	  PAGE1_START_MEMORY := 16#A000_0000# - 16#1_0000_0000#;
	  PAGE1_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
	  PAGE2_START_MEMORY := 16#B000_0000# - 16#1_0000_0000#;
	  PAGE2_END_MEMORY   := 16#FFFF_0000# - 16#1_0000_0000#;
        end if;
        COMMON_GRAPHIC_TYPES.ASPECT_RATIO  := 1.00;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X := 639;
        COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y := 479;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_START := PAGE1_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE1_MEMORY_END   := PAGE1_END_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_START := PAGE2_START_MEMORY;
	COMMON_GRAPHIC_TYPES.PAGE2_MEMORY_END   := PAGE2_END_MEMORY;

      when others => null;
    end case;

    COMMON_GRAPHIC_TYPES.ACTIVE_PAGE            := 0;
    COMMON_GRAPHIC_TYPES.CURRENT_VIDEO_PAGE     := 0;
    COMMON_GRAPHIC_TYPES.CHARACTER_PIXEL_WIDTH  := 8;
    COMMON_GRAPHIC_TYPES.CHARACTER_PIXEL_HEIGHT := 14;

    COMMON_GRAPHIC_TYPES.CURRENT_DISK_DRIVE := DISK.GET_DEFAULT;
    DIRECTORY.CURRENT_NAME (
	 FOR_DRIVE => COMMON_GRAPHIC_TYPES.CURRENT_DISK_DRIVE,
	 NAME      => COMMON_GRAPHIC_TYPES.CURRENT_DISK_DIRECTORY,
	 LAST      => COMMON_GRAPHIC_TYPES.CURRENT_DISK_DIRECTORY_LAST_CHAR,
	 ERROR     => COMMON_GRAPHIC_TYPES.ERROR_CODE );

    BREAK_ON := ON;
    PROGRAM_CONTROL.SET_BREAK_STATUS (TO_STATE => BREAK_ON);

    COMMON_GRAPHIC_TYPES.CURRENT_WORLD_LOWER_RIGHT_X := COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X;
    COMMON_GRAPHIC_TYPES.CURRENT_WORLD_LOWER_RIGHT_Y := COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y;
    COMMON_GRAPHIC_TYPES.CURRENT_WORLD_INDEX_NO := 0;
    COMMON_GRAPHIC_TYPES.SCREEN_WORLD_RATIO_X   := 1.0;
    COMMON_GRAPHIC_TYPES.SCREEN_WORLD_RATIO_Y   := 1.0;

    COMMON_GRAPHIC_TYPES.CURRENT_WINDOW_UPPER_LEFT_X  := COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_UPPER_LEFT_X;
    COMMON_GRAPHIC_TYPES.CURRENT_WINDOW_UPPER_LEFT_Y  := COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_UPPER_LEFT_Y;
    COMMON_GRAPHIC_TYPES.CURRENT_WINDOW_LOWER_RIGHT_X := COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_X;
    COMMON_GRAPHIC_TYPES.CURRENT_WINDOW_LOWER_RIGHT_Y := COMMON_GRAPHIC_TYPES.SCREEN_DIMENSION_LOWER_RIGHT_Y;
    COMMON_GRAPHIC_TYPES.CURRENT_WINDOW_INDEX_NO := 0;
    COMMON_GRAPHIC_TYPES.CLIP_ENABLE := true;
    COMMON_GRAPHIC_TYPES.WINDOW_FORE_COLOR := COMMON_DISPLAY_TYPES.WHITE;
    COMMON_GRAPHIC_TYPES.WINDOW_BACK_COLOR := COMMON_DISPLAY_TYPES.BLACK;

    COMMON_GRAPHIC_TYPES.FORE_COLOR := COMMON_DISPLAY_TYPES.WHITE;
    COMMON_GRAPHIC_TYPES.BACK_COLOR := COMMON_DISPLAY_TYPES.BLACK;

  end SET_GLOBAL_VARIABLES;

  --ͼ

  --ͻ

  procedure INITIATE_GRAPH (DESIRED_VIDEO : COMMON_GRAPHIC_TYPES.VIDEO_SYSTEM) is
  begin
    VIDEO_NUMBER := COMMON_GRAPHIC_TYPES.VIDEO_SYSTEM'pos(DESIRED_VIDEO);
    DETECT_GRAPHIC_CARD;

    if not (((VIDEO_NUMBER >= 0) and (VIDEO_NUMBER <= 3)) or
       ((VIDEO_NUMBER >= 7) and (VIDEO_NUMBER <= 12))) then
      VIDEO_DESIRED := DESIRED_VIDEO;

      if BAD_CARD then
	TTY.Put( ROW           => 12,
		 COLUMN        =>  5,
		 ITEM          => "Error",
		 UNDERLINE     => false,
		 REVERSE_VIDEO => false,
		 BLINK         => true,
		 INTENSITY     => false );

	TTY.Put( ROW           => 12,
		 COLUMN        => 11,
		 ITEM          => "- you either selected the wrong video graphics card",
		 UNDERLINE     => false,
		 REVERSE_VIDEO => false,
		 BLINK         => false,
		 INTENSITY     => false );

        TTY.Put( ROW           => 13,
		 COLUMN        => 13,
		 ITEM          => "or a graphics card is not installed in the system",
		 UNDERLINE     => false,
		 REVERSE_VIDEO => false,
		 BLINK         => false,
		 INTENSITY     => false );

        delay 3.0;
        CLOSE_GRAPH;
      else
        SET_GRAPHIC_CARD;
        SET_GLOBAL_VARIABLES;
      end if;
    else
      TTY.Put( ROW           => 12,
	       COLUMN        =>  5,
	       ITEM          => "Error",
	       UNDERLINE     => false,
	       REVERSE_VIDEO => false,
	       BLINK         => true,
	       INTENSITY     => false );

      TTY.Put( ROW           => 12,
	       COLUMN        => 11,
	       ITEM          => "- you selected TEXT MODE instead of GRAPHICS MODE.",
	       UNDERLINE     => false,
	       REVERSE_VIDEO => false,
	       BLINK         => false,
	       INTENSITY     => false );

      delay(3.0);
    end if;

  end INITIATE_GRAPH;

  --ͼ

end GRAPH_INIT;
