/*
==============================================================================

    Program:    AWMCISAM

    Purpose:    Sample program that utilizies the Gold Disk MCI Animation
                Driver.

    Note:
    Gold Disk does not warrant, guarantee or make any representations
    regarding the use of, or the results of the use of the Sample MCI
    application in terms of correctness, accuracy, reliability, currentness,
    or otherwise; and you rely on the Sample MCI application and results
    solely at your own risk.

==============================================================================
*/

#include "awmcisam.h"
#include "mciawi_x.h"

VOID    msDisplayMCIError (DWORD);
VOID    msSetControlsState (HWND, BOOL);
BOOL    msIsMoviePlaying (VOID);
VOID    msFillWindowColor (HWND, int);
VOID    msSetDlgItemCaps (HWND,DWORD, WORD);


// If flag is TRUE we play into the driver's default screen otherwise
// we play into the static window within the main dialog
static BOOL bFullScreen;

// This is the main instance handle for the app
static HANDLE   hMainInstance;

// This holds the device Id of a open MCI element
static WORD wDeviceID = -1;

//  main dialog
static HWND hdlgMain;

/*****************************************************************************

    Function: WinMain ()

*****************************************************************************/

int PASCAL WinMain (
HANDLE hInstance,				 // current instance
HANDLE hPrevInstance,			 // previous instance
LPSTR lpCmdLine,				 // command line
int nCmdShow)					 // show-window type (open/icon)
{
    MSG         msg;
    FARPROC     lpfnMainDlgProc; // Pointer for the Main Dialog Box
    
    hMainInstance = hInstance;
    lpfnMainDlgProc = MakeProcInstance (msMainDlgProc, hMainInstance);
    return DialogBox (hMainInstance, IDDLG_MAIN, NULL, lpfnMainDlgProc);
}


/*****************************************************************************

    Function:   MainWndProc ()          - windows callable

    Desc:       Main hidden window; parent of main dialog box.

*****************************************************************************/

LONG FAR PASCAL MainWndProc (
HWND hwnd,
unsigned message,
WORD wParam,
LONG lParam)
{
    switch (message)
    {

    default:
		    return (DefWindowProc(hwnd, message, wParam, lParam));
    }
    return (NULL);
}


/******************************************************************************

	Function:	msMainDlgProc ()        - windows callable

	Desc:		Dialog procedure for the 'Main' dialog box.

	Returns:    BOOL    TRUE            
                        FALSE           

******************************************************************************/

BOOL FAR PASCAL
msMainDlgProc (
    HWND        hdlg,
	unsigned	message,
	WORD		wParam,
	LONG		lParam)
{
    // MCI structure that is needed when playing a movie    
    MCI_ANIM_PLAY_PARMS      mciAnimPlayParms;

    // MCI structure that is needed when stepping in a movie    
    MCI_ANIM_STEP_PARMS      mciAnimStepParms;

    // MCI structure that is needed when changing the window of a movie    
    MCI_ANIM_WINDOW_PARMS    mciAnimWindowParms;

    // MCI structure that is needed when seeking within a movie    
    MCI_SEEK_PARMS           mciSeekParms;

    FARPROC     lpfnOpenDlgProc;      // Pointer for the Open Dialog Box
    FARPROC     lpfnSeekDlgProc;      // Pointer for the Seek Dialog Box
    FARPROC     lpfnDevCapsDlgProc;   // Pointer for the Device Caps Dialog Box
    FARPROC     lpfnInfoDlgProc;      // Pointer for the Info Dialog Box
    FARPROC     lpfnStatusDlgProc;    // Pointer for the Status Dialog Box
    FARPROC     lpfnAboutDlgProc;     // Pointer for the About Dialog Box

    DWORD       dwError;
    RECT        rc;
    HMENU       hSysMenu;             // Handle to the system menu
    char        szAbout[msMAX];

    PAINTSTRUCT ps;
    POINT       ptTopLeft;
    POINT       ptBottomRight;


	switch (message)
	{
    case WM_INITDIALOG:
        hdlgMain = hdlg;

    	// adds a About option to the system menu
	    hSysMenu = GetSystemMenu (hdlg,0);
    	AppendMenu (hSysMenu, MF_SEPARATOR, 0, NULL);

        LoadString (hMainInstance,IDS_ABOUT, szAbout, sizeof(szAbout)-1);               
    	AppendMenu (hSysMenu, MF_STRING, IDM_ABOUT, szAbout);

        // sets up the icon for the application   
        SetClassWord (hdlg, GCW_HICON, LoadIcon (hMainInstance, IDI_APPICON));

        // Init values to defaults
        //
        bFullScreen = FALSE;
        CheckDlgButton (hdlg, IDCB_MAIN_FULL_SCRN, bFullScreen);

        // disable all the controls 
        msSetControlsState (hdlg, FALSE);

        return(TRUE);
		break;

    case WM_CTLCOLOR:
        if (bFullScreen)
        {
            if (GetDlgItem (hdlg, IDST_MAIN_PLAYWND)
                    == LOWORD (lParam) )
            {
                msFillWindowColor (GetDlgItem (hdlg, IDST_MAIN_PLAYWND),
                                GRAY_BRUSH);
            }
        }
        break;

    case WM_SYSCOMMAND:
		switch (wParam)
	    {
	    case IDM_ABOUT:
            lpfnAboutDlgProc = MakeProcInstance (msAboutDlgProc, hMainInstance);
	        DialogBox (hMainInstance, IDDLG_ABOUT, hdlg, lpfnAboutDlgProc);
            break;

        default:
	        break;
        }
        break;

    case MM_MCINOTIFY:
        if (wParam == MCI_NOTIFY_SUCCESSFUL)
        {
            // set focus back to the main dialog
            SetFocus (hdlgMain);

            msSetControlsState (hdlg, TRUE);  // ungrey all the controls

            // set up the window that the movie is to be played in
            if (bFullScreen)
            {
                mciAnimWindowParms.hWnd = MCI_ANIM_WINDOW_DEFAULT;
                dwError = mciSendCommand (wDeviceID, MCI_WINDOW,
                                MCI_ANIM_WINDOW_HWND,
                                (DWORD) (LPVOID) &mciAnimWindowParms);

            }
            else
            {
                MCI_ANIM_RECT_PARMS mciAnimRectParms;

                mciAnimWindowParms.hWnd = GetDlgItem (hdlg, IDST_MAIN_PLAYWND);
                dwError = mciSendCommand (wDeviceID, MCI_WINDOW,
                                MCI_ANIM_WINDOW_HWND,
                                (DWORD) (LPVOID) &mciAnimWindowParms);

                //  open up some area around the child window ... note that
                //  setting the animation client area to a smaller size may
                //  slow the animation due to clipping calculations.

                GetClientRect (mciAnimWindowParms.hWnd, &mciAnimRectParms.rc);
                mciAnimRectParms.rc.left += 12;
                mciAnimRectParms.rc.top += 10;

                //  note right & bottom fields are actually extents!!

                mciAnimRectParms.rc.right -= 24;
                mciAnimRectParms.rc.bottom -= 20;

                dwError = mciSendCommand (wDeviceID, MCI_PUT,
                                MCI_ANIM_PUT_DESTINATION|MCI_ANIM_RECT,
                                (DWORD) (LPVOID) &mciAnimRectParms);
            }


            // show the movie
            dwError = mciSendCommand (wDeviceID, MCI_SEEK,
                        MCI_WAIT | MCI_SEEK_TO_START, (DWORD) (LPVOID) &mciAnimPlayParms);

            if (!dwError)
            {
                dwError = mciSendCommand (wDeviceID, MCI_STOP,
                           MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);
            }

            if (dwError)
            {
                msDisplayMCIError (dwError);
            }

            // set focus back to the main dialog
            SetFocus (hdlgMain);
        }
        else
        {
            dwError = mciSendCommand (wDeviceID, MCI_CLOSE,
                    MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);
            msSetControlsState (hdlg, FALSE);  // grey all the controls
        }
        break;

	case WM_COMMAND:

		switch (wParam)
        {
        case IDPB_MAIN_TOSTART:
            dwError = mciSendCommand (wDeviceID, MCI_SEEK,
                        MCI_WAIT | MCI_SEEK_TO_START,
                        (DWORD) (LPVOID) &mciSeekParms);

            if (dwError)
            {
               msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_TOEND:
            dwError = mciSendCommand (wDeviceID, MCI_SEEK,
                        MCI_WAIT | MCI_SEEK_TO_END,
                        (DWORD) (LPVOID) &mciSeekParms);

            if (dwError)
            {
               msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_DEV_CAP:
            if (wDeviceID != -1)
            {
                // if the movie is playing then stop it
                if (msIsMoviePlaying)
                {

                   dwError = mciSendCommand (wDeviceID, MCI_STOP,
                            MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);
                   if (dwError)
                   {
                        msDisplayMCIError (dwError);
                        break;
                   }
 
                }
                lpfnDevCapsDlgProc = MakeProcInstance (msDevCapsDlgProc, hMainInstance);
	            DialogBox (hMainInstance, IDDLG_DEVCAPS, hdlg, lpfnDevCapsDlgProc);
            }
            break;

        case IDPB_MAIN_STATUS:
            if (wDeviceID != -1)
            {
                // if the movie is playing then stop it
                if (msIsMoviePlaying)
                {

                   dwError = mciSendCommand (wDeviceID, MCI_STOP,
                            MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);
                   if (dwError)
                   {
                        msDisplayMCIError (dwError);
                        break;
                   }
 
                }
                lpfnStatusDlgProc = MakeProcInstance (msStatusDlgProc, hMainInstance);
	            DialogBox (hMainInstance, IDDLG_STATUS, hdlg, lpfnStatusDlgProc);
            }
            break;

        case IDPB_MAIN_INFO:
            if (wDeviceID != -1)
            {
                // if the movie is playing then stop it
                if (msIsMoviePlaying)
                {

                   dwError = mciSendCommand (wDeviceID, MCI_STOP,
                            MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);
                   if (dwError)
                   {
                        msDisplayMCIError (dwError);
                        break;
                   }
 
                }
                lpfnInfoDlgProc = MakeProcInstance (msInfoDlgProc, hMainInstance);
	            DialogBox (hMainInstance, IDDLG_INFO, hdlg, lpfnInfoDlgProc);
            }
            break;


        case IDPB_MAIN_SEEK:
            if (wDeviceID != -1)
            {
                // if the movie is playing then stop it
                if (msIsMoviePlaying)
                {

                   dwError = mciSendCommand (wDeviceID, MCI_STOP,
                            MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);
                   if (dwError)
                   {
                        msDisplayMCIError (dwError);
                        break;
                   }
 
                }
                lpfnSeekDlgProc = MakeProcInstance (msSeekDlgProc, hMainInstance);
	            DialogBox (hMainInstance, IDDLG_SEEK, hdlg, lpfnSeekDlgProc);
            }
            break;


        case IDPB_MAIN_STOP:
            dwError = mciSendCommand (wDeviceID, MCI_STOP,
                            MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);
            if (dwError)
            {
                msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_PAUSE:
            dwError = mciSendCommand (wDeviceID, MCI_PAUSE,
                        MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);

            if (dwError)
            {
                msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_CLOSE:
            dwError = mciSendCommand (wDeviceID, MCI_CLOSE,
                    MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);

            if (!dwError)
            {
                if (wDeviceID != -1)
                    wDeviceID = -1;

                // fill the play window with gray
                msFillWindowColor (GetDlgItem (hdlg, IDST_MAIN_PLAYWND),
                                GRAY_BRUSH);
 
                msSetControlsState (hdlg, FALSE);  // grey all the controls
            }
            else
            {
                msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_PLAY_FORWARD:
            dwError = mciSendCommand (wDeviceID, MCI_PLAY, 0,
                        (DWORD) (LPVOID) &mciAnimPlayParms);

            if (dwError)
            {
                msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_PLAY_REVERSE:
            dwError = mciSendCommand (wDeviceID, MCI_PLAY,
                        MCI_ANIM_PLAY_REVERSE,
                        (DWORD) (LPVOID) &mciAnimPlayParms);
            if (dwError)
            {
                msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_STEP_FORWARD:
            mciAnimStepParms.dwFrames = 1;
            dwError = mciSendCommand (wDeviceID, MCI_STEP,
                    MCI_WAIT | MCI_ANIM_STEP_FRAMES,
                    (DWORD) (LPVOID) &mciAnimStepParms);

            if (dwError)
            {
                msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_STEP_REVERSE:
            mciAnimStepParms.dwFrames = 1;
            dwError = mciSendCommand (wDeviceID, MCI_STEP,
                    MCI_WAIT | MCI_ANIM_STEP_FRAMES | MCI_ANIM_STEP_REVERSE,
                    (DWORD) (LPVOID) &mciAnimStepParms);

            if (dwError)
            {
                msDisplayMCIError (dwError);
            }
            break;

        case IDPB_MAIN_OPEN:
            // call the open dialog box
            lpfnOpenDlgProc = MakeProcInstance (msOpenDlgProc, hMainInstance);
            DialogBox (hMainInstance, IDDLG_OPEN, hdlg, lpfnOpenDlgProc);
            FreeProcInstance (lpfnOpenDlgProc);

            // set focus back to the main dialog
            SetFocus (hdlgMain);
            break;

        case IDCB_MAIN_FULL_SCRN:
            if (wDeviceID != -1)
            {
                if (!bFullScreen)
                {
                    mciAnimWindowParms.hWnd = MCI_ANIM_WINDOW_DEFAULT;
                    dwError = mciSendCommand (wDeviceID, MCI_WINDOW,
                                    MCI_ANIM_WINDOW_HWND,
                                    (DWORD) (LPVOID) &mciAnimWindowParms);

                }
                else
                {
                    mciAnimWindowParms.hWnd = GetDlgItem (hdlg, IDST_MAIN_PLAYWND);
                    dwError = mciSendCommand (wDeviceID, MCI_WINDOW,
                                    MCI_ANIM_WINDOW_HWND,
                                    (DWORD) (LPVOID) &mciAnimWindowParms);
                }


                
                if (dwError)
                {
                    msDisplayMCIError (dwError);
                    break;
                }
                else
                {
                    bFullScreen = !bFullScreen;
                    CheckDlgButton (hdlg, IDCB_MAIN_FULL_SCRN, bFullScreen);
                }

                dwError = mciSendCommand (wDeviceID, MCI_SEEK,
                            MCI_WAIT | MCI_SEEK_TO_START, (DWORD) (LPVOID) &mciAnimPlayParms);

                if (!dwError)
                {
                    dwError = mciSendCommand (wDeviceID, MCI_STOP,
                               MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);
                }


                if (dwError)
                {
                    msDisplayMCIError (dwError);
                }
                else
                {
                    // set focus back to the main dialog
                    SetFocus (hdlg);
                }
 
               
            }
            break;

		default:
			// Message not processed
			return(FALSE);
		}

        // Message processed
        return(TRUE);
        break;

    case WM_CLOSE:
        if (wDeviceID != -1)
        {
           dwError = mciSendCommand (wDeviceID, MCI_CLOSE,
                        MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);

           if (dwError)
           {
               msDisplayMCIError (dwError);
           }
 
        }
        EndDialog (hdlg, FALSE);
  		break;

    default:
        // Message not processed
        return(FALSE);

    }
	return(FALSE);
}


/******************************************************************************

	Function:	msOpenDlgProc ()		- windows callable

	Desc:		Dialog procedure for the 'Open' dialog box.

	Returns:    BOOL    TRUE            - OK was pressed
                        FALSE           - Cancel was pressed

******************************************************************************/

BOOL FAR PASCAL
msOpenDlgProc (
    HWND        hdlg,
	unsigned	message,
	WORD		wParam,
	LONG		lParam)
{
    // MCI structure that is needed when loading a movie    
    MCI_ANIM_OPEN_PARMS      mciAnimOpenParms;

    // MCI structure that is needed when playing a movie    
    MCI_ANIM_PLAY_PARMS      mciAnimPlayParms;

    char    szFile[msMAX];
    DWORD   dwError;

	switch (message)
	{
	case WM_INITDIALOG:
        SendDlgItemMessage(hdlg, IDEB_OPEN_MOVIE_FILE, EM_LIMITTEXT, msMAX, 0L);
        
        return(TRUE);
		break;

	case WM_COMMAND:

		switch (wParam)
        {

        case IDPB_OK:
            // gets the filename from the dialog
            GetDlgItemText (hdlg, IDEB_OPEN_MOVIE_FILE, szFile,
				msMAX-1);

            // if we already have a movie open then we close it
            if (wDeviceID != -1)
            {
                mciSendCommand (wDeviceID, MCI_CLOSE,
                    MCI_WAIT, (DWORD) (LPVOID) &mciAnimPlayParms);

                wDeviceID = -1;
            }

            // disable all the controls
            msSetControlsState (hdlgMain, FALSE);
 
            mciAnimOpenParms.lpstrDeviceType = NULL;
	        mciAnimOpenParms.lpstrElementName = (LPSTR) szFile;
	        mciAnimOpenParms.dwStyle = WS_OVERLAPPED | WS_THICKFRAME;
	        mciAnimOpenParms.lpstrAlias = "Scarper";
            mciAnimOpenParms.dwCallback = hdlgMain;

            dwError = mciSendCommand (NULL, MCI_OPEN,
                    MCI_OPEN_ALIAS | MCI_WAIT | MCI_NOTIFY | MCI_OPEN_ELEMENT | MCI_ANIM_OPEN_WS,
                    (DWORD) (LPVOID) &mciAnimOpenParms);

            if (dwError)
            {
                SetFocus (hdlgMain);

                msDisplayMCIError (dwError);
            }
            else
            {
                wDeviceID = mciAnimOpenParms.wDeviceID;
            }

            // if there was an error we return FALSE, else return TRUE
            EndDialog (hdlg, !dwError);
            break;

        case IDPB_CANCEL:
            EndDialog (hdlg, FALSE);
            break;

		default:
			// Message not processed
			return(FALSE);
		}

        // Message processed
        return(TRUE);
        break;

    case WM_CLOSE:
        EndDialog (hdlg, FALSE);
  		break;

    default:
        // Message not processed
        return(FALSE);

    }
	return(FALSE);
}


/******************************************************************************

	Function:	msSeekDlgProc ()		- windows callable

	Desc:		Dialog procedure for the 'Seek' dialog box.

	Returns:    BOOL    TRUE            - OK was pressed
                        FALSE           - Cancel was pressed

******************************************************************************/

BOOL FAR PASCAL
msSeekDlgProc (
    HWND        hdlg,
	unsigned	message,
	WORD		wParam,
	LONG		lParam)
{

    // MCI structure that is needed when seeking within movie    
    MCI_SEEK_PARMS           mciSeekParms;

    int     nToFrame;
    BOOL    bNoError;
    DWORD   dwError;

	switch (message)
	{

	case WM_INITDIALOG:
        
        return(TRUE);
		break;

	case WM_COMMAND:

		switch (wParam)
        {

        case IDPB_OK:
            nToFrame = GetDlgItemInt (hdlg, IDEB_SEEK_TOFRAME, &bNoError, FALSE);

            if (bNoError)
            {
                mciSeekParms.dwTo = nToFrame;
                dwError = mciSendCommand (wDeviceID, MCI_SEEK,
                        MCI_WAIT | MCI_TO,
                        (DWORD) (LPVOID) &mciSeekParms);

                if (dwError)
                {
                   msDisplayMCIError (dwError);
                }
                else
                {
                    EndDialog (hdlg, FALSE);
                }
            }
            break;

        case IDPB_CANCEL:
            EndDialog (hdlg, FALSE);
            break;

		default:
			// Message not processed
			return(FALSE);
		}

        // Message processed
        return(TRUE);
        break;

    case WM_CLOSE:
        EndDialog (hdlg, FALSE);
  		break;

    default:
        // Message not processed
        return(FALSE);

    }
	return(FALSE);
}


/******************************************************************************

	Function:	msDisplayError ()	    - local

	Desc:		Handles all the error message for the MCI commands.

	Returns:    VOID
                    

******************************************************************************/

VOID
msDisplayMCIError (DWORD dwError)
{
    char        szError[msMAX];
    char        szTitle[msMAX];

    // converts the ID of the error into a string and displays it to the user
    mciGetErrorString (dwError, szError, msMAX-1);
    LoadString (hMainInstance,IDS_ERROR_TITLE, szTitle, sizeof(szTitle)-1);               
    MessageBox (NULL, szError, szTitle,MB_ICONSTOP | MB_OK | MB_TASKMODAL);
}

static VOID msEnableDlgItem (HWND hdlg, WORD wId, BOOL bEnable)
{
    HWND    hwnd = GetDlgItem (hdlg, wId);

    if (IsWindow (hwnd))
        EnableWindow (hwnd, bEnable);
}

/******************************************************************************

	Function:	msSetControlsState ()	    - local

	Desc:		Enable or disables all the controls (expect Open).

	Returns:    VOID
                    

******************************************************************************/

VOID
msSetControlsState (
HWND hdlg,
BOOL bState)
{
    // Need to Grey all the control that can't be used
    // unless a movie has been opened
    msEnableDlgItem (hdlg, IDPB_MAIN_CLOSE, bState);
    msEnableDlgItem (hdlg, IDCB_MAIN_FULL_SCRN, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_STOP, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_PAUSE, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_PLAY_FORWARD, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_PLAY_REVERSE, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_STEP_REVERSE, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_STEP_FORWARD, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_SEEK, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_DEV_CAP, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_STATUS, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_INFO, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_TOSTART, bState);
    msEnableDlgItem (hdlg, IDPB_MAIN_TOEND, bState);
}


/******************************************************************************

	Function:	msDevCapsDlgProc ()		- windows callable

	Desc:		Dialog procedure for the 'Device Capabilities' dialog box.

	Returns:    BOOL    TRUE            - OK was pressed
                        FALSE           - Cancel was pressed

******************************************************************************/

BOOL FAR PASCAL
msDevCapsDlgProc (
    HWND        hdlg,
	unsigned	message,
	WORD		wParam,
	LONG		lParam)
{

    MCI_GETDEVCAPS_PARMS     mciGetDevCapsParms;
    DWORD       dwResult;
    char        szResult[msMAX];

	switch (message)
	{

	case WM_INITDIALOG:

        // set dialog if the device can be ejected
        msSetDlgItemCaps (hdlg,
                          MCI_GETDEVCAPS_CAN_EJECT,
                          IDST_DEVCAP_EJECT);

        // sets dialog if the device can be played
        msSetDlgItemCaps (hdlg,
                          MCI_GETDEVCAPS_CAN_PLAY,
                          IDST_DEVCAP_PLAY);

        // sets dialog if the device can be record
        msSetDlgItemCaps (hdlg,
                          MCI_GETDEVCAPS_CAN_RECORD,
                          IDST_DEVCAP_RECORD);

        // sets dialog if the device can be save
        msSetDlgItemCaps (hdlg,
                          MCI_GETDEVCAPS_CAN_SAVE,
                          IDST_DEVCAP_SAVE);

        // sets dialog if the device can use device elements
        msSetDlgItemCaps (hdlg,
                          MCI_GETDEVCAPS_COMPOUND_DEVICE,
                          IDST_DEVCAP_DEVELE);

        // finds out the type of device we are working with
        mciGetDevCapsParms.dwItem = MCI_GETDEVCAPS_DEVICE_TYPE;
        mciSendCommand (wDeviceID, MCI_GETDEVCAPS,
                    MCI_WAIT | MCI_GETDEVCAPS_ITEM,
                    (DWORD) (LPVOID) &mciGetDevCapsParms);

        dwResult = mciGetDevCapsParms.dwReturn;
        switch (dwResult)
        {
        case MCI_DEVTYPE_ANIMATION:
            LoadString (hMainInstance,IDS_DEVTYPE_ANIMATION,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        case MCI_DEVTYPE_CD_AUDIO:
            LoadString (hMainInstance,IDS_DEVTYPE_CD_AUDIO,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        case MCI_DEVTYPE_DAT:
            LoadString (hMainInstance,IDS_DEVTYPE_DAT,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        case MCI_DEVTYPE_DIGITAL_VIDEO:
            LoadString (hMainInstance,IDS_DEVTYPE_DIGITAL_VIDEO,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        case MCI_DEVTYPE_OTHER:
            LoadString (hMainInstance,IDS_DEVTYPE_OTHER,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        case MCI_DEVTYPE_OVERLAY:
            LoadString (hMainInstance,IDS_DEVTYPE_OVERLAY,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        case MCI_DEVTYPE_SCANNER:
            LoadString (hMainInstance,IDS_DEVTYPE_SCANNER,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        case MCI_DEVTYPE_SEQUENCER:
            LoadString (hMainInstance,IDS_DEVTYPE_SEQUENCER,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        case MCI_DEVTYPE_VIDEODISC:
            LoadString (hMainInstance,IDS_DEVTYPE_VIDEODISC,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

// Video not supported in the version of MCI I am working with
//      case MCI_DEVTYPE_VIDEOTAPE:
//          LoadString (hMainInstance,IDS_DEVTYPE_VIDEOTAPE,
//                          szResult, sizeof(szResult)-1);               
//          SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
//          break;


        case MCI_DEVTYPE_WAVEFORM_AUDIO:
            LoadString (hMainInstance,IDS_DEVTYPE_WAVEFORM_AUDIO,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;


        deafault:
            LoadString (hMainInstance,IDS_UNKNOWN,
                            szResult, sizeof(szResult)-1);               
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_DEVTYPE), szResult);
            break;

        }
 

        // sets dialog if the device has audio
        msSetDlgItemCaps (hdlg,
                          MCI_GETDEVCAPS_HAS_AUDIO,
                          IDST_DEVCAP_AUDIO);

  
        // sets dialog if the device has video
        msSetDlgItemCaps (hdlg,
                          MCI_GETDEVCAPS_HAS_VIDEO,
                          IDST_DEVCAP_VIDEO);

   
        // sets dialog if the device can use user files 
        msSetDlgItemCaps (hdlg,
                          MCI_GETDEVCAPS_USES_FILES,
                          IDST_DEVCAP_USESFILES);


        // sets dialog if the device can play in reverse
        msSetDlgItemCaps (hdlg,
                          MCI_ANIM_GETDEVCAPS_CAN_REVERSE,
                          IDST_DEVCAP_PLAYREVERSE);

 
        // sets dialog if the device can stretch
        msSetDlgItemCaps (hdlg,
                          MCI_ANIM_GETDEVCAPS_CAN_STRETCH,
                          IDST_DEVCAP_STRETCH);


        // finds out the fast rate of the device 
        mciGetDevCapsParms.dwItem = MCI_ANIM_GETDEVCAPS_FAST_RATE;
        mciSendCommand (wDeviceID, MCI_GETDEVCAPS,
                    MCI_WAIT | MCI_GETDEVCAPS_ITEM,
                    (DWORD) (LPVOID) &mciGetDevCapsParms);

        dwResult = mciGetDevCapsParms.dwReturn;
        if (dwResult == MCIERR_UNSUPPORTED_FUNCTION)
        {
            LoadString (hMainInstance,IDS_UNSUPPORTED_FUNCTION,
                            szResult, sizeof(szResult)-1);
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_FASTRATE), szResult);
        }
        else
        {
            ltoa (dwResult, szResult, 10);
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_FASTRATE), szResult);
        }

 
        // finds out the normal rate of the device 
        mciGetDevCapsParms.dwItem = MCI_ANIM_GETDEVCAPS_NORMAL_RATE;
        mciSendCommand (wDeviceID, MCI_GETDEVCAPS,
                    MCI_WAIT | MCI_GETDEVCAPS_ITEM,
                    (DWORD) (LPVOID) &mciGetDevCapsParms);

        dwResult = mciGetDevCapsParms.dwReturn;
        if (dwResult == MCIERR_UNSUPPORTED_FUNCTION)
        {
            LoadString (hMainInstance,IDS_UNSUPPORTED_FUNCTION,
                            szResult, sizeof(szResult)-1);
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_NORMRATE), szResult);
        }
        else
        {
            ltoa (dwResult, szResult, 10);
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_NORMRATE), szResult);
        }


        // finds out the slow rate of the device 
        mciGetDevCapsParms.dwItem = MCI_ANIM_GETDEVCAPS_SLOW_RATE;
        mciSendCommand (wDeviceID, MCI_GETDEVCAPS,
                    MCI_WAIT | MCI_GETDEVCAPS_ITEM,
                    (DWORD) (LPVOID) &mciGetDevCapsParms);

        dwResult = mciGetDevCapsParms.dwReturn;
        if (dwResult == MCIERR_UNSUPPORTED_FUNCTION)
        {
            LoadString (hMainInstance,IDS_UNSUPPORTED_FUNCTION,
                            szResult, sizeof(szResult)-1);
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_SLOWRATE), szResult);
        }
        else
        {
            ltoa (dwResult, szResult, 10);
            SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_SLOWRATE), szResult);
        }

  
        // finds out the max windows this device can handle 
        mciGetDevCapsParms.dwItem = MCI_ANIM_GETDEVCAPS_MAX_WINDOWS;
        mciSendCommand (wDeviceID, MCI_GETDEVCAPS,
                    MCI_WAIT | MCI_GETDEVCAPS_ITEM,
                    (DWORD) (LPVOID) &mciGetDevCapsParms);

        dwResult = mciGetDevCapsParms.dwReturn;
        ltoa (dwResult, szResult, 10);
        SetWindowText (GetDlgItem (hdlg, IDST_DEVCAP_MAXWIN), szResult);

 

        // sets dialog if the device can use palettes
        msSetDlgItemCaps (hdlg,
                          MCI_ANIM_GETDEVCAPS_PALETTES,
                          IDST_DEVCAP_PALETTE);

        return(TRUE);
		break;

	case WM_COMMAND:

		switch (wParam)
        {

        case IDPB_OK:
            EndDialog (hdlg, TRUE);
            break;


		default:
			// Message not processed
			return(FALSE);
		}

        // Message processed
        return(TRUE);
        break;

    case WM_CLOSE:
        EndDialog (hdlg, FALSE);
  		break;

    default:
        // Message not processed
        return(FALSE);

    }
	return(FALSE);
}


/******************************************************************************

	Function:	msInfoDlgProc ()		- windows callable

	Desc:		Dialog procedure for the 'Info' dialog box.

	Returns:    BOOL    TRUE            - OK was pressed
                        FALSE           - Cancel was pressed

******************************************************************************/

BOOL FAR PASCAL
msInfoDlgProc (
    HWND        hdlg,
	unsigned	message,
	WORD		wParam,
	LONG		lParam)
{
    MCI_VERSION_PARMS      mciVerParms;
    MCI_GETDEVCAPS_PARMS   mciGetDevCapsParms;
    MCI_INFO_PARMS         mciInfoParms;

    char        szResult[msMAX];
    DWORD       dwError;
    char        szIn[msMAX];
    char        szOut[msMAX];
 

	switch (message)
	{

	case WM_INITDIALOG:

        // finds out if the device can use user files 
        mciGetDevCapsParms.dwItem = MCI_GETDEVCAPS_USES_FILES;
        mciSendCommand (wDeviceID, MCI_GETDEVCAPS,
                    MCI_WAIT | MCI_GETDEVCAPS_ITEM,
                    (DWORD) (LPVOID) &mciGetDevCapsParms);


        if (mciGetDevCapsParms.dwReturn)
        {
            // finds out the filename of the device 
            mciInfoParms.lpstrReturn = szResult;
            mciInfoParms.dwRetSize = sizeof(szResult);

            dwError = mciSendCommand (wDeviceID, MCI_INFO,
                        MCI_WAIT | MCI_INFO_FILE,
                        (DWORD) (LPVOID) &mciInfoParms);

            if (dwError)
            {
               msDisplayMCIError (dwError);
            }
            else
            {

                 SetWindowText (GetDlgItem (hdlg, IDST_INFO_FILENAME),
                            mciInfoParms.lpstrReturn);
            }

        }
        else
        {
            LoadString (hMainInstance,IDS_UNSUPPORTED_FUNCTION,
                            szResult, sizeof(szResult)-1);
            SetWindowText (GetDlgItem (hdlg, IDST_INFO_FILENAME), szResult);
        }

        


        // finds out the caption on the window of the device
        dwError = mciSendCommand (wDeviceID, MCI_INFO,
                    MCI_ANIM_INFO_TEXT,
                    (DWORD) (LPVOID) &mciInfoParms);

        if (dwError)
        {
            msDisplayMCIError (dwError);
        }
        else
        {
            SetWindowText (GetDlgItem (hdlg, IDST_INFO_CAPTION),
                       mciInfoParms.lpstrReturn);
        }

        // finds out the product info
        dwError = mciSendCommand (wDeviceID, MCI_INFO,
                    MCI_INFO_PRODUCT,
                    (DWORD) (LPVOID) &mciInfoParms);

        if (dwError)
        {
            msDisplayMCIError (dwError);
        }
        else
        {
            SetWindowText (GetDlgItem (hdlg, IDST_INFO_PRODUCT),
                       mciInfoParms.lpstrReturn);

        }

        // Get the MCI driver version number
        //

        dwError = mciSendCommand (wDeviceID,
            GOLDDISK_MCI_GETVER, 0, (DWORD) (LPVOID)&mciVerParms);

        // To use mciSendString we need to use an alias like "Scarper"
        // the version is returned in the dwReturn field of the mciVerParms struct
        // eg. dwError = mciSendString ("version Scarper", szOut, sizeof (szOut), &mciVerParms);

        if (dwError)
        {
            msDisplayMCIError (dwError);
        }
        else
        {
            GetDlgItemText (hdlg, IDST_INFO_DDVERSION, (LPSTR) szIn, sizeof(szIn)-1);
            wsprintf ((LPSTR) szOut, "%s %d.%02.02d.%02.02d", (LPSTR) szIn,
            GetRValue(mciVerParms.dwReturn), GetGValue(mciVerParms.dwReturn),
            GetBValue(mciVerParms.dwReturn));

            SetDlgItemText (hdlg, IDST_INFO_DDVERSION, (LPSTR) szOut);
        }
        return(TRUE);
		break;


	case WM_COMMAND:

		switch (wParam)
        {
        case IDPB_OK:
            EndDialog (hdlg, TRUE);
            break;

		default:
			// Message not processed
			return(FALSE);
		}

        // Message processed
        return(TRUE);
        break;

    case WM_CLOSE:
        EndDialog (hdlg, FALSE);
  		break;

    default:
        // Message not processed
        return(FALSE);

    }
	return(FALSE);
}


/******************************************************************************

	Function:	msStatusDlgProc ()		- windows callable

	Desc:		Dialog procedure for the 'Status' dialog box.

	Returns:    BOOL    TRUE            - OK was pressed
                        FALSE           - Cancel was pressed

******************************************************************************/

BOOL FAR PASCAL
msStatusDlgProc (
    HWND        hdlg,
	unsigned	message,
	WORD		wParam,
	LONG		lParam)
{
    MCI_SET_PARMS            mciSetParms;
    MCI_STATUS_PARMS         mciStatusParms;
    char        szResult[msMAX];
    DWORD       dwError;


	switch (message)
	{

	case WM_INITDIALOG:

        // sets the time format to frames
        mciSetParms.dwTimeFormat = MCI_FORMAT_FRAMES;
	    mciSendCommand (wDeviceID, MCI_SET, (DWORD) MCI_SET_TIME_FORMAT,
            (DWORD) (LPSTR) &mciSetParms);

        // we only have one movie
        mciStatusParms.dwTrack = (DWORD) 1;


        // finds out the current frame the movie is on
        mciStatusParms.dwItem = MCI_STATUS_POSITION;

        dwError = mciSendCommand (wDeviceID, MCI_STATUS,
                    (DWORD) MCI_STATUS_ITEM,
                    (DWORD) (LPSTR)(&mciStatusParms));

        if (dwError)
        {
            msDisplayMCIError (dwError);
        }
        else
        {
            ltoa (mciStatusParms.dwReturn, szResult, 10);
            SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRFRAME), szResult);
        }


        // finds out the number of frames in the movie
        mciStatusParms.dwItem = MCI_STATUS_LENGTH;

        dwError = mciSendCommand (wDeviceID, MCI_STATUS,
                    (DWORD) MCI_STATUS_ITEM | MCI_TRACK,
                    (DWORD) (LPSTR)(&mciStatusParms));

        if (dwError)
        {
            msDisplayMCIError (dwError);
        }
        else
        {
            ltoa (mciStatusParms.dwReturn, szResult, 10);
            SetWindowText (GetDlgItem (hdlg, IDST_STATUS_NUMFRAMES), szResult);
        }

        // finds out the mode of the movie
        mciStatusParms.dwItem = MCI_STATUS_MODE;

        dwError = mciSendCommand (wDeviceID, MCI_STATUS,
                    (DWORD) MCI_STATUS_ITEM,
                    (DWORD) (LPSTR)(&mciStatusParms));

        if (dwError)
        {
            msDisplayMCIError (dwError);
        }
        else
        {
            switch (mciStatusParms.dwReturn)
            {
            case MCI_MODE_NOT_READY:
                LoadString (hMainInstance, IDS_MODE_NOT_READY,
                            szResult, sizeof(szResult)-1);               
                SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
                break;

            case MCI_MODE_PAUSE:
                LoadString (hMainInstance,IDS_MODE_PAUSE,
                            szResult, sizeof(szResult)-1);               
                SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
                break;

            case MCI_MODE_PLAY:
                // finds out if we are playing forward
                mciStatusParms.dwItem = MCI_ANIM_STATUS_FORWARD;

                dwError = mciSendCommand (wDeviceID, MCI_STATUS,
                            (DWORD) MCI_STATUS_ITEM,
                            (DWORD) (LPSTR)(&mciStatusParms));

                if (dwError)
                {
                    msDisplayMCIError (dwError);
                }

                if (mciStatusParms.dwReturn)
                {
                    LoadString (hMainInstance,IDS_MODE_PLAY_FORWARD,
                            szResult, sizeof(szResult)-1);               
                    SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
 
                }
                else
                {
                    LoadString (hMainInstance,IDS_MODE_PLAY_REVERSE,
                            szResult, sizeof(szResult)-1);               
                    SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
 
                }
                break;

            case MCI_MODE_STOP:
                 LoadString (hMainInstance,IDS_MODE_STOP,
                         szResult, sizeof(szResult)-1);               
                 SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
                 break;

            case MCI_MODE_OPEN:
                 LoadString (hMainInstance,IDS_MODE_OPEN,
                         szResult, sizeof(szResult)-1);               
                 SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
                 break;

            case MCI_MODE_RECORD:
                 LoadString (hMainInstance,IDS_MODE_RECORD,
                         szResult, sizeof(szResult)-1);               
                 SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
                 break;

            case MCI_MODE_SEEK:
                 LoadString (hMainInstance,IDS_MODE_SEEK,
                         szResult, sizeof(szResult)-1);               
                 SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
                 break;


            default:
                LoadString (hMainInstance,IDS_UNKNOWN,
                            szResult, sizeof(szResult)-1);               
                SetWindowText (GetDlgItem (hdlg, IDST_STATUS_CURRMODE), szResult);
            break;



            }
        }


        // finds out the speed of animation 
        mciStatusParms.dwItem = MCI_ANIM_STATUS_SPEED;
        dwError = mciSendCommand (wDeviceID, MCI_STATUS,
                    (DWORD) MCI_STATUS_ITEM,
                    (DWORD) (LPSTR)(&mciStatusParms));

        if (dwError)
        {
            msDisplayMCIError (dwError);
        }
        else
        {
            ltoa (mciStatusParms.dwReturn, szResult, 10);
            SetWindowText (GetDlgItem (hdlg, IDST_STATUS_SPEED), szResult);
        }

        mciStatusParms.dwItem = MCI_ANIM_STATUS_HWND;

        dwError = mciSendCommand (wDeviceID, MCI_STATUS,
                    (DWORD) MCI_STATUS_ITEM,
                    (DWORD) (LPSTR)(&mciStatusParms));

        if (dwError)
        {
            msDisplayMCIError (dwError);
        }
        else
        {
            SetDlgItemInt (hdlg, IDST_STATUS_HWND, (WORD)mciStatusParms.dwReturn, FALSE);
        }

        return(TRUE);
		break;


	case WM_COMMAND:

		switch (wParam)
        {

        case IDPB_OK:
            EndDialog (hdlg, TRUE);
            break;

		default:
			// Message not processed
			return(FALSE);
		}

        // Message processed
        return(TRUE);
        break;

    case WM_CLOSE:
        EndDialog (hdlg, FALSE);
  		break;

    default:
        // Message not processed
        return(FALSE);

    }
	return(FALSE);
}


/******************************************************************************

	Function:	msAboutDlgProc ()		- windows callable

	Desc:		Dialog procedure for the 'About' dialog box.

	Returns:    BOOL    TRUE            - OK was pressed

******************************************************************************/

BOOL FAR PASCAL
msAboutDlgProc (
    HWND        hdlg,
	unsigned	message,
	WORD		wParam,
	LONG		lParam)
{
    char        szResult[msMAX];
    char        szIn[msMAX];
    char        szOut[msMAX];
    DWORD       dwError;

	switch (message)
	{

	case WM_INITDIALOG:
        // Set the version number
        //
        GetDlgItemText (hdlg, IDST_VERSION, (LPSTR) szIn, sizeof(szIn)-1);
        wsprintf ((LPSTR) szOut, "%s %d.%02.02d.%02.02d", (LPSTR) szIn,
        		GetRValue(msMCISAMVERSION), GetGValue(msMCISAMVERSION),
        		GetBValue(msMCISAMVERSION));
 
		SetDlgItemText (hdlg, IDST_VERSION, (LPSTR) szOut);
        return(TRUE);
		break;

	case WM_COMMAND:

		switch (wParam)
        {

        case IDPB_OK:
            EndDialog (hdlg, TRUE);
            break;


		default:
			// Message not processed
			return(FALSE);
		}

        // Message processed
        return(TRUE);
        break;

    case WM_CLOSE:
        EndDialog (hdlg, FALSE);
  		break;

    default:
        // Message not processed
        return(FALSE);

    }
	return(FALSE);
}


/****************************************************************************

	Function:	msIsMoviePlaying ()		- local

	Desc:		Checks to see if we are playing the movie.

	Returns:    BOOL    TRUE            - We are playing the movie
                        FALSE           - We are not playing the movie

****************************************************************************/

BOOL
msIsMoviePlaying (VOID)
{
    MCI_STATUS_PARMS    mciStatusParms;
    DWORD               dwError;


    // finds out the mode of the movie
    mciStatusParms.dwItem = MCI_STATUS_MODE;

    dwError = mciSendCommand (wDeviceID, MCI_STATUS,
                (DWORD) MCI_STATUS_ITEM,
                (DWORD) (LPSTR)(&mciStatusParms));

    if (dwError)
    {
        msDisplayMCIError (dwError);
        return FALSE;
    }
    else
    {
        if (mciStatusParms.dwReturn == MCI_MODE_PLAY)
        {
            return TRUE;
        }
    }
    return FALSE;
}


/****************************************************************************

	Function:	msFillWindowColor ()		- local

	Desc:		Fills the given window in the given color.

	Returns:    VOID    
                        

****************************************************************************/

VOID
msFillWindowColor (HWND hwndPlay, int nIndex)
{
    HDC         hdcPlay;
    POINT       ptTopLeft;
    RECT        rc;

    GetClientRect (hwndPlay, &rc);

    // Get DC of the window and fill it with the given brush index
    hdcPlay = GetDC (hwndPlay);
    FillRect (hdcPlay, &rc, GetStockObject (nIndex));
    ReleaseDC (hwndPlay, hdcPlay);
}


/****************************************************************************

	Function:	msSetDlgItemCaps ()		- local

	Desc:		Determines where the given Item is supported or not.
                Fills the given dialog.

	Returns:    VOID    
                        

****************************************************************************/

VOID
msSetDlgItemCaps (
HWND hdlg,      // handle to the DevCaps dialog
DWORD dwItem,   // the item to check
WORD wId)       // the string ID of dwItem
{
    MCI_GETDEVCAPS_PARMS     mciGetDevCapsParms;
    DWORD       dwResult;
    char        szResult[msMAX];

    mciGetDevCapsParms.dwItem = dwItem;
    mciSendCommand (wDeviceID, MCI_GETDEVCAPS,
                MCI_WAIT | MCI_GETDEVCAPS_ITEM,
                (DWORD) (LPVOID) &mciGetDevCapsParms);


    if (mciGetDevCapsParms.dwReturn)
    {
        LoadString (hMainInstance,IDS_SUPPORTED_FUNCTION,
                        szResult, sizeof(szResult)-1);               
    }
    else
    {
        LoadString (hMainInstance,IDS_UNSUPPORTED_FUNCTION,
                        szResult, sizeof(szResult)-1);               
 
    }
    SetWindowText (GetDlgItem (hdlg, wId), szResult);
}
