#if !defined(BRRECT_H) && !defined(__BEDRC__)
#define BRRECT_H
//========================================================================================
//
//     File:		BRRect.h
//     Release Version:	$ 1.0d1 $
//
//     Creation Date:	11/20/90
//
//     COPYRIGHT 1990-1993 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED -- RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT NOTICE IS
//     PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE Contains PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions as Set
//     forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and Computer
//     Software clause at DFARS 252.227-7013. Symantec Corporation, 10201 Torre Avenue,
//     Cupertino, CA 95014.
//
//========================================================================================

#ifndef BRSTDDEF_H
#include "BRStdDef.h"
#endif

#ifndef BROBJECT_H
#include "BRObject.h"
#endif

#if defined(BR_BUILD_MAC) && !defined(__QUICKDRAW__)
#include <QuickDraw.h>
#endif

#if defined(BR_BUILD_WIN) && !defined(__INC_WINDOWS)
#include <windows.h>
#endif

//========================================================================================
//	Type definitions
//========================================================================================

typedef long BR_Coordinate;
typedef long BR_Distance;

//========================================================================================
//	structure BR_SPoint
//========================================================================================

struct BR_SPoint
{
	BR_Coordinate fX;
	BR_Coordinate fY;
};

//========================================================================================
//	CLASS BR_CPoint
//========================================================================================
// This class represents a position in two dimensions.

class BR_CPoint : public BR_SPoint
{
public:
	typedef unsigned short int XYSelector;
	enum XYSelectorConstants
	{
		kX = 0,
		kY = 1
	};

public:
	//------------------------------------------------------------------------------------
	// Constructors/Destructor
	//------------------------------------------------------------------------------------

	BR_CPoint(BR_Coordinate X, BR_Coordinate Y);
	BR_CPoint(const BR_CPoint& point);
	BR_CPoint(const BR_SPoint& point);
	BR_CPoint();
	~BR_CPoint();

	//------------------------------------------------------------------------------------
	// Setters
	//------------------------------------------------------------------------------------
	
	void Set(BR_Coordinate X, BR_Coordinate Y);
	
	//------------------------------------------------------------------------------------
	// Selector operators, two are needed one for operating on const BR_CPoints and the other
    // on non-const BR_CPoints.
	//------------------------------------------------------------------------------------
	
	BR_Coordinate& operator[](XYSelector sel);
		// For non-const BR_CPoints
		
	const BR_Coordinate& operator[](XYSelector sel) const;
		// For const BR_CPoints
	
	//------------------------------------------------------------------------------------
	// Arithmetic operators
	//------------------------------------------------------------------------------------

	BR_CPoint& operator-=(const BR_CPoint& pt);
	BR_CPoint& operator+=(const BR_CPoint& pt);
	BR_CPoint operator-(const BR_CPoint& pt) const;
	BR_CPoint operator+(const BR_CPoint& pt) const;
	
	//------------------------------------------------------------------------------------
	// Copy
	//------------------------------------------------------------------------------------

	BR_CPoint& operator=(const BR_SPoint& pt);
	BR_CPoint& operator=(const BR_CPoint& pt);

	//------------------------------------------------------------------------------------
	// Relational operators
	//------------------------------------------------------------------------------------
	BR_Boolean operator==(const BR_CPoint& pt) const;
	BR_Boolean operator!=(const BR_CPoint& pt) const;
	BR_Boolean operator<(const BR_CPoint& pt) const;
	BR_Boolean operator<=(const BR_CPoint& pt) const;
	BR_Boolean operator>(const BR_CPoint& pt) const;
	BR_Boolean operator>=(const BR_CPoint& pt) const;

	//------------------------------------------------------------------------------------
	// Macintosh specific
	//------------------------------------------------------------------------------------
#ifdef BR_BUILD_MAC
public:
	BR_CPoint(const Point &pt);	
	BR_CPoint& operator=(const Point &pt);	
	operator Point() const;
#endif

	//------------------------------------------------------------------------------------
	// Windows specific
	//------------------------------------------------------------------------------------
#ifdef BR_BUILD_WIN
public:
	operator POINT() const;

	BR_CPoint(LONG lParam);	
	BR_CPoint(const POINT &pt);
	BR_CPoint& operator=(const POINT &pt);
	BR_CPoint& operator=(LONG lParam);
	operator LONG() const;
#endif

	//------------------------------------------------------------------------------------
	// Orthogonal
	//------------------------------------------------------------------------------------
	static XYSelector GetOrthogonal(XYSelector selector);
};

//----------------------------------------------------------------------------------------
//	BR_CPoint::Set
//----------------------------------------------------------------------------------------
inline void BR_CPoint::Set(BR_Coordinate X,
						   BR_Coordinate Y)
{
	fX = X;
	fY = Y;
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator-=
//----------------------------------------------------------------------------------------
inline BR_CPoint& BR_CPoint::operator-=(const BR_CPoint& pt)
{
	fX -= pt.fX;
	fY -= pt.fY;
	return (*this);
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator+=
//----------------------------------------------------------------------------------------
inline BR_CPoint& BR_CPoint::operator+=(const BR_CPoint& pt)
{
	fX += pt.fX;
	fY += pt.fY;
	return (*this);
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator-
//----------------------------------------------------------------------------------------
inline BR_CPoint BR_CPoint::operator-(const BR_CPoint& pt) const
{
	return (BR_CPoint(fX - pt.fX, fY - pt.fY));
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator+
//----------------------------------------------------------------------------------------
inline BR_CPoint BR_CPoint::operator+(const BR_CPoint& pt) const
{
	return (BR_CPoint(fX + pt.fX, fY + pt.fY));
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator=
//----------------------------------------------------------------------------------------
inline BR_CPoint& BR_CPoint::operator=(const BR_SPoint& pt)
{
	fX = pt.fX;
	fY = pt.fY;
	return (*this);
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator=
//----------------------------------------------------------------------------------------
inline BR_CPoint& BR_CPoint::operator=(const BR_CPoint& pt)
{
	fX = pt.fX;
	fY = pt.fY;
	return (*this);
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator==
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CPoint::operator==(const BR_CPoint& pt) const
{
	return (fX == pt.fX && fY == pt.fY);
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator!=
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CPoint::operator!=(const BR_CPoint& pt) const
{
	return (fX != pt.fX || fY != pt.fY);
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator<
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CPoint::operator<(const BR_CPoint& pt) const
{
	return ((fY < pt.fY) || (fY == pt.fY && fX < pt.fX));
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator>
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CPoint::operator>(const BR_CPoint& pt) const
{
	return ((fY > pt.fY) || (fY == pt.fY && fX > pt.fX));
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator<=
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CPoint::operator<=(const BR_CPoint& pt) const
{
	return !operator>(pt);
}

//----------------------------------------------------------------------------------------
//	BR_CPoint::operator>=
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CPoint::operator>=(const BR_CPoint& pt) const
{
	return !operator<(pt);
}

#ifdef BR_BUILD_MAC
//----------------------------------------------------------------------------------------
//	BR_CPoint::operator=
//----------------------------------------------------------------------------------------
inline BR_CPoint& BR_CPoint::operator=(const Point &pt)
{
	fX = pt.h;
	fY = pt.v;
	
	return *this;
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
//	BR_CPoint::operator=
//----------------------------------------------------------------------------------------
inline BR_CPoint& BR_CPoint::operator=(LONG lParam)
{
	fX = LONG(short(LOWORD(lParam)));
	fY = LONG(short(HIWORD(lParam)));
	return (*this);
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
//	BR_CPoint::operator=
//----------------------------------------------------------------------------------------
inline BR_CPoint& BR_CPoint::operator=(const POINT &pt)
{
	fX = pt.x;
	fY = pt.y;

	return *this;
}
#endif

//========================================================================================
//	CLASS BR_SRectangle
//========================================================================================

struct BR_SRectangle
{
	BR_Coordinate fLeft;
	BR_Coordinate fTop;
	BR_Coordinate fRight;
	BR_Coordinate fBottom;
};

//========================================================================================
//	CLASS BR_CRectangle
//	This class represents a rectangle consisting of four coordinates--
//	the fLeft, fTop, fRight, and fBottom
//========================================================================================

class BR_CRectangle : public BR_SRectangle
{
public:
	typedef unsigned short int PointSelector;
	enum PointSelectorConstants
	{
		kTopLeft = 0,
		kBottomRight = 1
	};

public:
	//------------------------------------------------------------------------------------
	// Constructors/Destructor
	//------------------------------------------------------------------------------------
	BR_CRectangle();
	BR_CRectangle(const BR_CPoint& topLeft,
				  const BR_CPoint& bottomRight);
	BR_CRectangle(BR_Coordinate l, BR_Coordinate t, 
				  BR_Coordinate r, BR_Coordinate b);
	BR_CRectangle(const BR_SRectangle& r);
	BR_CRectangle(const BR_CRectangle& r);
	virtual~ BR_CRectangle();

	//------------------------------------------------------------------------------------
	// BR_CPoint selectors for BR_CRectangle
	//------------------------------------------------------------------------------------

	BR_CPoint& operator[](PointSelector sel);
		// Selector for non-const BR_CRectangle
		
	const BR_CPoint& operator[](PointSelector sel) const;
		// Selector for const BR_CRectangle

	//------------------------------------------------------------------------------------
	// Getters
	//------------------------------------------------------------------------------------
	BR_Distance GetWidth() const;
	BR_Distance GetHeight() const;
	BR_Distance GetLength(BR_CPoint::XYSelector s) const;
	BR_CPoint GetTopLeft() const;
	BR_CPoint GetTopRight() const;
	BR_CPoint GetBottomLeft() const;
	BR_CPoint GetBottomRight() const;
	BR_CPoint GetSize() const;

	//------------------------------------------------------------------------------------
	// Setters
	//------------------------------------------------------------------------------------
	void Set(BR_Coordinate lLeft,
                     BR_Coordinate lTop,
                     BR_Coordinate lRight,
                     BR_Coordinate lBottom);
	void Set(const BR_CPoint& topLeft,
			 const BR_CPoint& bottomRight);

	//------------------------------------------------------------------------------------
	// Copy
	//------------------------------------------------------------------------------------
	void operator=(const BR_CRectangle& r);
	void operator=(const BR_SRectangle& r);

	//------------------------------------------------------------------------------------
	// Relational operators
	//------------------------------------------------------------------------------------
	BR_Boolean operator==(const BR_CRectangle& r) const;
	BR_Boolean operator!=(const BR_CRectangle& r) const;
	void Intersect(const BR_CRectangle& rect1,
				   const BR_CRectangle& rect2);
	BR_Boolean IsEmpty() const;
	
	//------------------------------------------------------------------------------------
	// Operations on rectangle
	//------------------------------------------------------------------------------------
	void Inflate(BR_Distance fX, BR_Distance fY);
	void Offset(BR_Distance fX, BR_Distance fY);
	void Union(const BR_CRectangle& rect1,
			   const BR_CRectangle& rect2);
	void Place(BR_Coordinate fX, BR_Coordinate fY);

	//------------------------------------------------------------------------------------
	// Hit testing
	//------------------------------------------------------------------------------------
	BR_Boolean IsPointIn(const BR_CPoint& pt) const;
	BR_Boolean IsOutside(const BR_CRectangle& r) const;

	void Sort();

	//------------------------------------------------------------------------------------
	// Macintosh specific
	//------------------------------------------------------------------------------------
#ifdef BR_BUILD_MAC
public:
	operator Rect() const;
	
	BR_CRectangle(const Rect& r);				// ?INTERNAL?
	void Set(const Rect& r);					// ?INTERNAL?
	void operator=(const Rect& r);				// ?INTERNAL?

#endif

	//------------------------------------------------------------------------------------
	// Windows Specific
	//------------------------------------------------------------------------------------
#ifdef BR_BUILD_WIN
public:
	operator RECT() const;
	
	BR_CRectangle(const RECT& r);				// ?INTERNAL?
	void Set(const RECT& r);					// ?INTERNAL?
	void operator=(const RECT& r);				// ?INTERNAL?

#endif
};

//----------------------------------------------------------------------------------------
//	BR_CRectangle::GetWidth
//----------------------------------------------------------------------------------------
inline BR_Distance BR_CRectangle::GetWidth() const
{
	return (fRight - fLeft);
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::GetHeight
//----------------------------------------------------------------------------------------
inline BR_Distance BR_CRectangle::GetHeight() const
{
	return (fBottom - fTop);
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::GetTopLeft
//----------------------------------------------------------------------------------------
inline BR_CPoint BR_CRectangle::GetTopLeft() const
{
	return (BR_CPoint(fLeft, fTop));
}

//----------------------------------------------------------------------------------------
// BR_CRectangle::GetTopRight
//----------------------------------------------------------------------------------------

inline BR_CPoint BR_CRectangle::GetTopRight() const
{
	return (BR_CPoint(fRight, fTop));
}

//----------------------------------------------------------------------------------------
// BR_CRectangle::GetBottomLeft
//----------------------------------------------------------------------------------------

inline BR_CPoint BR_CRectangle::GetBottomLeft() const
{
	return (BR_CPoint(fLeft, fBottom));
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::GetBottomRight
//----------------------------------------------------------------------------------------
inline BR_CPoint BR_CRectangle::GetBottomRight() const
{
	return (BR_CPoint(fRight, fBottom));
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::GetSize
//----------------------------------------------------------------------------------------
inline BR_CPoint BR_CRectangle::GetSize() const
{
	return (BR_CPoint(fRight - fLeft, fBottom - fTop));
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::GetLength
//----------------------------------------------------------------------------------------

inline BR_Distance BR_CRectangle::GetLength(BR_CPoint::XYSelector s) const
{
	return s == BR_CPoint::kY ? fBottom - fTop : fRight - fLeft;
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::operator=
//----------------------------------------------------------------------------------------
inline void BR_CRectangle::operator=(const BR_CRectangle& r)
{
	fLeft = r.fLeft;
	fTop = r.fTop;
	fRight = r.fRight;
	fBottom = r.fBottom;
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::operator=
//----------------------------------------------------------------------------------------
inline void BR_CRectangle::operator=(const BR_SRectangle& r)
{
	fLeft = r.fLeft;
	fTop = r.fTop;
	fRight = r.fRight;
	fBottom = r.fBottom;
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::operator==
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CRectangle::operator==(const BR_CRectangle& r) const
{
	return (fLeft == r.fLeft && fRight == r.fRight && fTop == r.fTop && fBottom == r.fBottom);
}

//----------------------------------------------------------------------------------------
//	BR_CRectangle::operator!=
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CRectangle::operator!=(const BR_CRectangle& r) const
{
	return (!(*this == r));
}

#ifdef BR_BUILD_MAC
//----------------------------------------------------------------------------------------
//	BR_CRectangle::operator=
//----------------------------------------------------------------------------------------
inline void BR_CRectangle::operator=(const Rect& r)
{
	Set(r);
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
//	BR_CRectangle::operator=
//----------------------------------------------------------------------------------------
inline void BR_CRectangle::operator=(const RECT& r)
{
	Set(r);
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
//	WinGetWidth
//----------------------------------------------------------------------------------------
inline short WinGetWidth(RECT& rRect)
{
	return (rRect.right - rRect.left);
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
//	WinGetHeight
//----------------------------------------------------------------------------------------
inline short WinGetHeight(RECT& rRect)
{
	return (rRect.bottom - rRect.top);
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
//	WinToBedRect
//----------------------------------------------------------------------------------------
inline void WinToBedRect(BR_CRectangle& rBed,
						 const RECT& rWin)
{
	rBed = rWin;
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
//	BedToWinRect
//----------------------------------------------------------------------------------------
inline void BedToWinRect(RECT& rWin,
						 const BR_CRectangle& rBed)
{
	rWin = RECT(rBed);
}
#endif

#endif

