#if !defined(BRCANVAS_H) && !defined(__BEDRC__)
#define BRCANVAS_H

//========================================================================================
//
//     File:			BRCanvas.h
//     Release Version:	$ 1.0d1 $
//
//     Creation Date:	12/10/90
//
//     COPYRIGHT 1990-1993 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED -- RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT NOTICE IS
//     PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE CONTAINS PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions as set
//     forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and Computer
//     Software clause at DFARS 252.227-7013. Symantec Corporation, 10201 Torre Avenue,
//     Cupertino, CA 95014.
//
//========================================================================================

#ifndef BRRECT_H
#include "BRRect.h"
#endif

#ifndef BRGRCOMM_H
#include "BRGrComm.h"		// Common graphics type and constant defines
#endif

#ifndef BRPEN_H
#include "BRPen.h"
#endif

#ifndef BRFONT_H
#include "BRFont.h"
#endif

#ifndef BRWALLPA_H
#include "BRWallpa.h"
#endif

#ifndef BRREGION_H
#include "BRRegion.h"
#endif

#ifndef BCCOLL_H
#include "BCColl.h"
#endif

#if defined(BR_BUILD_MAC) && !defined(__QUICKDRAW__)
#include <QuickDraw.h>
#endif

#if defined(BR_BUILD_WIN) && !defined(BRCOLOR_H)
#include "BRColor.h"
#endif

//========================================================================================
//	Forward class declaration
//========================================================================================

class BR_CView;
class BR_CBitmap;
class BR_CIcon;
class BR_CPicture;
class BR_CDrawStateData;

//========================================================================================
// STRUCT BR_SBedCanvasResolutions
// Specifies what coordinate xform factors are used when drawing
//========================================================================================
// Will be scoped when all compilers support it

struct BR_SBedCanvasResolutions
{
	unsigned short fXNum, fXDen;					// X and Y conversion for all graphics
	unsigned short fYNum, fYDen;
	
	unsigned short fTextSizeNum, fTextSizeDen;		// and text size
	
	unsigned short fPenSizeNum, fPenSizeDen;		// and pen size
	
	unsigned short fXTextPosNum, fXTextPosDen;		// except text position
	unsigned short fYTextPosNum, fYTextPosDen;
};

//========================================================================================
// STRUCT BR_SBedGraphicsCache
// This is what we cache when a view is opened for drawing
//========================================================================================

struct BR_SBedGraphicsCache
{
	BR_SBedCanvasResolutions fResolutions;		// Resolutions of the current canvas
	BR_CPoint fCanvasOrigin;
	BR_CPoint fViewToCanvasOffset;					// QD/GDI offset for 32bit coord system
	BR_CRectangle fDrawArea;					// The draw area of the current canvas
	
#ifdef BR_BUILD_WIN
	HRGN fWinGraphicsClipRgn;				// clip regions used by printing
	HRGN fWinTextClipRgn;
	HRGN fWinSelectedClipRgn;
#endif
};

//========================================================================================
// BR_CCanvas
//========================================================================================

class BR_CCanvas : public BR_CObject
{
public:
	enum CoordConvSelector
	{
		kGraphicsPos, kTextPos
	};											// ?INTERNAL?


	enum ScaleAndOffset
	{
		kScaleOnly, kScaleAndOffset
	};											// ?INTERNAL?

public:
	virtual~ BR_CCanvas();
	
	BR_CView 	*GetCanvasOwner() const;
	void		SetCanvasOwner(BR_CView *canvasOwner);
	
	virtual void GetDrawArea(BR_CRectangle& area) const = 0;

	virtual void OpenPane(BR_CDrawStateData& drawStateData);
		// Initializes the drawing context of the canvas for the current view which can
		//  be obtained by GetPane().

	virtual void SuspendPane(BR_CDrawStateData& drawStateData);
		// Suspends drawing on the current view and initializes drawing context to
		//  default values. Uses DrawInitiator for per-view data so that we can nest
		//  SuspendPane.

	virtual void ResumePane(const BR_CDrawStateData& drawStateData);
		// Resumes drawing on the current view using data saved by SuspendPane in the
		//  DrawInitiator.

	virtual void ClosePane(const BR_CDrawStateData& drawStateData);
		// Terminates drawing on the current view: unselects all currently selected
		//  tools and prepares the canvas for a drawing environment shutdown.

	virtual void GetOrigin(BR_Coordinate& x,
						   BR_Coordinate& y) const;
	virtual void SetOrigin(BR_Coordinate x,
						   BR_Coordinate y);
	virtual void OffsetOrigin(BR_Distance xOffset,
							  BR_Distance yOffset);

	virtual void GetResolution(BR_CPoint& ptResolution) const;
		//Retrieves canvas resolution in logical units per fInch.

	// ============ Text handling methods

	unsigned short GetFontHeight() const;
	void GetTextMetrics(BR_TextMetrics& bedTextMetrics) const;

	// For text with tabs:
	unsigned short 	GetCharacterWidth(BR_Char c);
	unsigned short GetStringWidth(const BR_TString<BR_Char>& text,
						  			long startX = 0) const;
	void GetStringSizes(const BR_TString<BR_Char>& text,
						unsigned short& width,
						unsigned short& height,
						long startX = 0) const;
	unsigned short GetStringIndexFromPoint(const BR_TString<BR_Char>& text,
								   long startX,
								   long endX,
								   IndexFromPtOption option) const;
	void DrawText(BR_Coordinate x,
				  BR_Coordinate y,
				  const BR_TString<BR_Char>& text);
	void GrayText(BR_Coordinate x,
				  BR_Coordinate y,
				  const BR_TString<BR_Char>& text);
	void HiliteText(BR_Coordinate x,
					BR_Coordinate y,
					const BR_TString<BR_Char>& text);
	void HiliteText(const BR_CRectangle& rect);
	void SetTabOrigin(BR_Coordinate x);

	// For wrapping a string in a rectangle (no tab expansion):
	unsigned short GetWrappedStringHeight(const BR_TString<BR_Char>& text,
								  			unsigned short width) const;
	void DrawText(const BR_CRectangle& r,
				  const BR_TString<BR_Char>& text,
				  BR_DrawTextFlags flags);

	// ============ Primitive drawing methods
	
	// ----- Pen Position
	void MovePen(BR_Distance x,
				 BR_Distance y);
	void MovePen(const BR_CPoint& point);
	void MovePenTo(const BR_CPoint& point);
	void GetPenPosition(BR_CPoint& point);
	virtual void MovePenTo(BR_Coordinate x,
						   BR_Coordinate y);

	// ----- Line
	void DrawLine(BR_Distance x,
				  BR_Distance y);
	void DrawLine(const BR_CPoint& point);
	void DrawLineTo(const BR_CPoint& point);
	virtual void DrawLineTo(BR_Coordinate x,
							BR_Coordinate y);

	// ----- Rectangle
	void DrawRectangle(BR_Coordinate left,
					   BR_Coordinate top,
					   BR_Coordinate right,
					   BR_Coordinate bottom);
	void DrawRectangle(const BR_CRectangle& rect,
					   const BR_CColor& color);
	void DrawRectangle(BR_Coordinate left,
					   BR_Coordinate top,
					   BR_Coordinate right,
					   BR_Coordinate bottom,
					   const BR_CColor& color);
	virtual void DrawRectangle(const BR_CRectangle& rect);
	virtual void FillRectangle(const BR_CRectangle& rect,
							   BR_CWallPaper& wallPaper);
	virtual void DrawWallPaper(const BR_CRectangle& area,
							   BR_CWallPaper& wallPaper,
							   BR_RasterOperation mode = BEDROP_PATCOPY);
	virtual void DrawFrame(const BR_CRectangle& frame,
						   BR_CWallPaper& wallPaper);
	virtual void DrawFrame(const BR_CRegion& rgnFrame,
						   BR_CWallPaper& wallPaper,
						   BR_Distance width,
						   BR_Distance height);

	// ----- Focus Rect
	virtual void DrawFocusRect(const BR_CRectangle& rect);
	
	// ----- INvert Rectangle
	virtual void InvertRectangle(const BR_CRectangle& rect);
	
	// ----- RoundRect
	virtual void DrawRoundRect(const BR_CRectangle& rect,
							   BR_Distance width,
							   BR_Distance height);
	// ----- Ellipse
	virtual void DrawEllipse(const BR_CRectangle& rect);
	
	// ----- Chord/Pie/Arc
	virtual void DrawChord(const BR_CRectangle& rect,
						   BR_Coordinate x1,
						   BR_Coordinate y1,
						   BR_Coordinate x2,
						   BR_Coordinate y2);
	virtual void DrawPie(BR_Coordinate xCenter,
						 BR_Coordinate yCenter,
						 BR_Distance radius,
						 BR_Float startAngle,
						 BR_Float sweepAngle);

	virtual void DrawArc(const BR_CRectangle& pRect,
						   BR_Coordinate x1,
						   BR_Coordinate y1,
						   BR_Coordinate x2,
						   BR_Coordinate y2);
	// ----- Polygon
	virtual void DrawPolygon(const BC_TCollection<BR_CPoint>& points);
	virtual void DrawPolyLine(const BC_TCollection<BR_CPoint>& points);
	
	// ----- Bitmap
	virtual void DrawBitmap(BR_Coordinate x,
							BR_Coordinate y,
							const BR_CRectangle& rBitRect,
							BR_CBitmap& bitmap,
							BR_RasterOperation mode = BEDROP_SRCCOPY);
	virtual void DrawBitmap(BR_Coordinate x,
							BR_Coordinate y,
							BR_CBitmap& bitmap,
							BR_RasterOperation mode = BEDROP_SRCCOPY);
	virtual void DrawIcon(BR_Coordinate x,
						  BR_Coordinate y,
						  const BR_CIcon& icon);

	// this should probably go away.  use BR_CBitmap::Stretch instead
	virtual void StretchBitmap(BR_Coordinate x,
							   BR_Coordinate y,
							   BR_Distance width,
							   BR_Distance height,
							   const BR_CRectangle& sourceRect,
							   BR_CBitmap& bitmap,
							   BR_RasterOperation mode = BEDROP_SRCCOPY);
	virtual void StretchBitmap(BR_Coordinate x,
							   BR_Coordinate y,
							   BR_Distance width,
							   BR_Distance height,
							   BR_CBitmap& bitmap,
							   BR_RasterOperation mode = BEDROP_SRCCOPY);

	// ----- Picture
	virtual void DrawPicture(BR_CPicture& picture);
	virtual void DrawPicture(BR_Coordinate x,
							 BR_Coordinate y,
							 BR_CPicture& picture);
	
	// ----- Region
	virtual void DrawRegion(const BR_CRegion& tegion);

	// ----- Pixels
	virtual void FloodFill(BR_Coordinate x,
						   BR_Coordinate y,
						   const BR_CColor& fillColor);
	virtual void SetPixel(BR_Coordinate x,
						  BR_Coordinate y,
						  const BR_CColor& clrPixel);
	virtual BR_CColor GetPixel(BR_Coordinate x,
							   BR_Coordinate y) const;

	virtual BR_DrawMode SetDrawMode(BR_DrawMode mode);
	virtual BR_DrawMode GetDrawMode() const;

	void ExcludeRect(const BR_CRectangle& rect);

	BR_Boolean Copy(const BR_CCanvas& canvas,
					BR_Coordinate x,
					BR_Coordinate y,
					const BR_CRectangle& copyRect,
					BR_RasterOperation ropCode = BEDROP_SRCCOPY);

	BR_Boolean Copy(const BR_CCanvas& canvas,
					BR_Coordinate x,
					BR_Coordinate y,
					BR_Distance width,
					BR_Distance height,
					const BR_CRectangle& copyRect,
					BR_RasterOperation ropCode = BEDROP_SRCCOPY);

	virtual void SetExtraTransform(const BR_CPoint& extraTransform);// ?INTERNAL?
		//	It is possible to specify an extra transform value which is then added to view coordinates
	  	//	when drawing a primitive.  This allows us to change where a view is drawn without
	  	//	changing view's origins.  This is useful for printing. 

	void GetExtraTransform(BR_CPoint& extraTransform) const;// ?INTERNAL?
		//	Retrieves the current value of extra transform.

	void SetPane(BR_CView* view);				// ?INTERNAL?
	BR_CView* GetPane() const;					// ?INTERNAL?

	void SetCurrentPen(BR_CPen* aPen);			// ?INTERNAL?
	BR_CPen* GetCurrentPen() const;				// ?INTERNAL?

	void SetCurrentFont(BR_CFont* aFont);		// ?INTERNAL?
	BR_CFont* GetCurrentFont() const;			// ?INTERNAL?

	void SetCurrentWallPaper(BR_CWallPaper* aWallPaper);// ?INTERNAL?
	BR_CWallPaper* GetCurrentWallPaper() const;	// ?INTERNAL?

	void SetCurrentRegion(BR_CRegion* aRegion);	// ?INTERNAL?
	BR_CRegion* GetCurrentRegion() const;		// ?INTERNAL?

	virtual void BedSelectClipRgn(BR_CRegion* region);
	virtual void BedRestoreNullClipRgn();

	virtual void BedReinitGraphicsCache(BR_CView *theView);	

	BR_Boolean BedConvertFromPane(const BR_CRectangle& viewRect,
								  RECT& rect,
								  CoordConvSelector whichRes,
								  ScaleAndOffset scaleOnly = kScaleAndOffset) const;

	BR_Boolean BedConvertFromPane(const BR_CPoint& viewPt,
								  POINT& pt,
								  CoordConvSelector whichRes,
								  ScaleAndOffset scaleOnly = kScaleAndOffset) const;

	void BedConvertToPane(const RECT& rect,
						  BR_CRectangle& viewRect,
						  CoordConvSelector whichRes,
						  ScaleAndOffset scaleOnly = kScaleAndOffset) const;

	void BedConvertToPane(const POINT& pt,
						  BR_CPoint& viewPt,
						  CoordConvSelector whichRes,
						  ScaleAndOffset scaleOnly = kScaleAndOffset) const;

	BR_CPen::PenThickness BedScalePenSize(BR_CPen::PenThickness penSize);
	BR_CFont::FontSize BedScaleFontSize(BR_CFont::FontSize fontSize);
	virtual BR_CPoint GetDeviceResolution() const;

protected:
	BR_CCanvas();
										   
	static BR_SBedGraphicsCache gGraphicsCache;// ?INTERNAL?


private:
	BR_CFont* 		fCurrentFont;				// selected font
	BR_CWallPaper* 	fCurrentWallPaper;			// selected wallpaper
	BR_CPen* 		fCurrentPen;				// selected pen
	BR_CRegion* 	fCurrentRegion;				// selected region
	BR_CView* 		fView;						// The view currently associated with this canvas
	BR_CPoint 		fExtraTransform;			// We offset view coordinates by this much before drawing
	BR_Coordinate 	fTabOrigin;					// Used by text handling methods
	BR_CView 		*fCanvasOwner;				// The owner of this canvas
	BR_CPoint		fCanvasResolution;
	
	static BR_Float Cosine(BR_Float n, BR_Float radius);
	static BR_Float Sine(BR_Float n, BR_Float radius);

	void BaseOpenPane(BR_CFont* font,
					  GraphicsToolSelectionOptions selectionOptions,
					  BR_SGraphicsToolSuspendResumeState* state);

	void BaseClosePane();

	BR_Coordinate BedGetTabOrigin() const;		// ?INTERNAL?

	void BedPrimHiliteText(RECT& rc);			// ?INTERNAL?

	BR_Boolean DoFillOp() const;				// ?INTERNAL?
	BR_Boolean DoPenOp() const;					// ?INTERNAL?
	BR_Boolean BedIsPatternPen() const;			// ?INTERNAL?
	BR_Boolean BedIsPenGapTransparent() const;	// ?INTERNAL?

	void BedSortRect(RECT& rect);				// ?INTERNAL?
	BR_Boolean BedDeflateRect(RECT& rect);		// ?INTERNAL?

	BR_DrawMode fDrawMode;						// ?INTERNAL?						// current draw mode
	BR_Coordinate fXOrigin;						// ?INTERNAL?
	BR_Coordinate fYOrigin;						// ?INTERNAL?

	// ----------------------------------------------------------------------------------------
	// Windows implementation
	// ----------------------------------------------------------------------------------------
#ifdef BR_BUILD_WIN
public:
	enum PrepareOpOptions
	{
		kPrepareClipRgn = 1, kPrepareColors = 2
	};											// ?INTERNAL?


public:
	virtual HDC BedWinCreateDC(BR_CView& view) = 0;
	virtual void BedWinDeleteDC(BR_CView& view,
						  		  HDC hDC) = 0;
	HDC BedWinGetDC() const;
	void BedWinSetDC(HDC newDC);

	// Virtual color selection stuff
	void BedWinSetFillColors(BR_CColor::RGBValue textColor,
							 BR_CColor::RGBValue backgroundColor);	// ?INTERNAL?
	void BedWinSetPenColors(BR_CColor::RGBValue textColor,
							BR_CColor::RGBValue backgroundColor);	// ?INTERNAL?

	void WinSetFontTextColor(const BR_CColor& textColor);// ?INTERNAL?
	void WinSetFontBackgroundColor(const BR_CColor& backgroundColor);// ?INTERNAL?

	void WinGetFontTextColor(BR_CColor& textColor) const;		// ?INTERNAL?
	void WinGetFontBackgroundColor(BR_CColor& backgroundColor) const;// ?INTERNAL?

	void BedWinSetPenHandle(HANDLE penHandle);	// ?INTERNAL?
	HANDLE BedWinGetPenHandle() const;			// ?INTERNAL?
	void BedWinSetFontHandle(HANDLE fontHandle);// ?INTERNAL?
	HANDLE BedWinGetFontHandle() const;			// ?INTERNAL?

	void BedWinPrepareFillOp(PrepareOpOptions options = kPrepareClipRgn | kPrepareColors);// ?INTERNAL?
	void BedWinPrepareFontOp(PrepareOpOptions options = kPrepareClipRgn | kPrepareColors);// ?INTERNAL?
	void BedWinPreparePenOp(PrepareOpOptions options = kPrepareClipRgn | kPrepareColors);// ?INTERNAL?

protected:
	static void ComputePoint(int xCenter, int yCenter,
							 BR_Float angle, int radius,
							 LPPOINT point);

private:
	HDC fDC;									// ?INTERNAL?									// This canvas's HDC

	BR_CColor::RGBValue fWinFillTextColor, fWinFillBkColor;// ?INTERNAL?
	BR_CColor::RGBValue fWinFontTextColor, fWinFontBkColor;// ?INTERNAL?
	BR_CColor::RGBValue fWinPenTextColor, fWinPenBkColor;	// ?INTERNAL?

	HANDLE fWinPenHandle;						// ?INTERNAL?
	HANDLE fWinFontHandle;						// ?INTERNAL?

	BR_SGraphicsToolSuspendResumeState fFontState;// ?INTERNAL?
	BR_SGraphicsToolSuspendResumeState fPenState;// ?INTERNAL?

	void BedWinAdjustRect(RECT& rect);			// ?INTERNAL?

	BR_Boolean BedWinPenAndFillConflict() const;// ?INTERNAL?
	HANDLE BedWinGetPenBrush() const;			// ?INTERNAL?
#endif

	//----------------------------------------------------------------------------------------
	// Macintosh implementation
	//----------------------------------------------------------------------------------------
#ifdef BR_BUILD_MAC
public:
	virtual GrafPtr 	BedMacGetGrafPort() const = 0;
	virtual void 		BedMacViewToCanvasOwner(BR_CPoint& aPoint) const = 0;
	virtual BR_Boolean 	BedMacGetBitMap(BitMap*& someMap) const;

protected:
	static void 		ComputePoint(int xCenter, int yCenter,
							 			BR_Float angle, int radius,
							 			Point* point);

private:
	void 				BedMacPrepareFillOp();
	void 				BedMacPreparePenOp();
		//	We use pen pats to do both area filling and line drawing, so we need to
	  	//	somehow resolve conflicts between what fCurrentPen and fCurrentWallPaper want.
#endif


	BR_SETCLASSNAME(BR_CCanvas);
};


//----------------------------------------------------------------------------------------
// BR_CCanvas::SetPane
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::SetPane(BR_CView* Pane)
{
	fView = Pane;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::GetPane
//----------------------------------------------------------------------------------------
inline BR_CView* BR_CCanvas::GetPane() const
{
	return fView;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::SetCurrentPen
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::SetCurrentPen(BR_CPen* aPen)
{
	fCurrentPen = aPen;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::GetCurrentPen
//----------------------------------------------------------------------------------------
inline BR_CPen* BR_CCanvas::GetCurrentPen() const
{
	return fCurrentPen;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::SetCurrentFont
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::SetCurrentFont(BR_CFont* aFont)
{
	fCurrentFont = aFont;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::GetCurrentFont
//----------------------------------------------------------------------------------------
inline BR_CFont* BR_CCanvas::GetCurrentFont() const
{
	return fCurrentFont;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::SetCurrentWallPaper
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::SetCurrentWallPaper(BR_CWallPaper* aWallPaper)
{
	fCurrentWallPaper = aWallPaper;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::GetCurrentWallPaper
//----------------------------------------------------------------------------------------
inline BR_CWallPaper* BR_CCanvas::GetCurrentWallPaper() const
{
	return fCurrentWallPaper;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::SetCurrentRegion
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::SetCurrentRegion(BR_CRegion* aRegion)
{
	fCurrentRegion = aRegion;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::GetCurrentRegion
//----------------------------------------------------------------------------------------
inline BR_CRegion* BR_CCanvas::GetCurrentRegion() const
{
	return fCurrentRegion;
}

//----------------------------------------------------------------------------------------
//	BR_CCanvas::DrawLine
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::DrawLine(const BR_CPoint& point)
{
	DrawLine(point.fX, point.fY);
}

//----------------------------------------------------------------------------------------
//	BR_CCanvas::DrawLineTo
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::DrawLineTo(const BR_CPoint& point)
{
	DrawLineTo(point.fX, point.fY);
}

//----------------------------------------------------------------------------------------
//	BR_CCanvas::MovePen
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::MovePen(const BR_CPoint& point)
{
	MovePen(point.fX, point.fY);
}

//----------------------------------------------------------------------------------------
//	BR_CCanvas::MovePenTo
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::MovePenTo(const BR_CPoint& point)
{
	MovePenTo(point.fX, point.fY);
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::GetCanvasOwner
//----------------------------------------------------------------------------------------
inline BR_CView* BR_CCanvas::GetCanvasOwner() const
{
	return fCanvasOwner;
}

//----------------------------------------------------------------------------------------
// BR_CCanvas::SetCanvasOwner
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::SetCanvasOwner(BR_CView* canvasOwner)
{
	fCanvasOwner = canvasOwner;
}

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
// BR_CCanvas::BedWinGetDC
//----------------------------------------------------------------------------------------
inline HDC BR_CCanvas::BedWinGetDC() const
{
	return (fDC);
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
// BR_CCanvas::BedWinSetDC
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::BedWinSetDC(HDC hNewDC)
{
	fDC = hNewDC;
}
#endif

#ifdef BR_BUILD_WIN

//----------------------------------------------------------------------------------------
// BR_CCanvas::BedWinSetPenHandle
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::BedWinSetPenHandle(HANDLE penHandle)
{
	fWinPenHandle = penHandle;
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
// BR_CCanvas::BedWinGetPenHandle
//----------------------------------------------------------------------------------------
inline HANDLE BR_CCanvas::BedWinGetPenHandle() const
{
	return fWinPenHandle;
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
// BR_CCanvas::BedWinSetFontHandle
//----------------------------------------------------------------------------------------
inline void BR_CCanvas::BedWinSetFontHandle(HANDLE fontHandle)
{
	fWinFontHandle = fontHandle;
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
// BR_CCanvas::BedWinGetFontHandle
//----------------------------------------------------------------------------------------
inline HANDLE BR_CCanvas::BedWinGetFontHandle() const
{
	return fWinFontHandle;
}
#endif

#endif
