/*----------------------------------------------------------------------------*\
|   aashow.c - A template for a Windows application 			       |
|	       using the Autodesk Animation Player			       |
|                                                                              |
|   Usage:                                                                     |
|	To make a quick windows application, modify the following files        |
|	    aashow.c	    - App source code				       |
|	    makefile	    - App makefile (unix make, none of this dos crap)  |
|	    aashow.h	    - App constants				       |
|	    aashow.def	    - App exported functions and module name	       |
|	    aashow.rc	    - App resources, DLG boxes, ...		       |
|           aashow.ico      - App icon					       |
|                                                                              |
|   History:                                                                   |
|	01/09/91 clevea     Added animation player			       |
|	01/01/88 toddla     Created					       |
|                                                                              |
|	Portions Copyright (C) 1990, Autodesk, Inc			       |
|                                                                              |
\*----------------------------------------------------------------------------*/

#include <windows.h>
#define _MT
#include <stdarg.h>
#undef _FAR_
#undef _MT
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include "aaplay.h"
#include "aashow.h"
#include "dialog.h"

typedef void (*GETPROC)(LPSTR, LPVOID, LPSTR);

#define BOUND(x,min,max) ((x) < (min) ? (min) : ((x) > (max) ? (max) : (x)))
#define SWAP(x,y)   ((x)^=(y)^=(x)^=(y))
#define MIN(x,y) (((x) <= (y)) : x ? y)
#define ALIGNB(x)   (((x) + 7) & ~0x07)

#define END_TEXTFILE '\032'

static char AAForever[] = "Forever";
static char AALoopSound[] = "Sound";

/*----------------------------------------------------------------------------*\
|                                                                              |
|   g l o b a l   v a r i a b l e s                                            |
|                                                                              |
\*----------------------------------------------------------------------------*/
static  char    szAppName[]="Animator Player Example";
static  char	szCaption[sizeof(szAppName) + 8];

static	HANDLE  hInstApp;
static	HWND    hwndApp;

static	HAA	hAnim = NULL;

static	WORD	fCommand = 0;

static	WORD	wMode = AA_LOOPFRAME | AA_STOPNOTIFY;

static	WORD	StopMsg, NotifyMsg;
static	BOOL	ShowStops = FALSE;

static	char	AnimFileName[128] = "";
static	char	SndFileName[128] = "";
static	char	SndDevice[32] = "";

static  WORD	fSound = FALSE;
static	WORD	fScript = FALSE;

/*----------------------------------------------------------------------------*\
|                                                                              |
|   f u n c t i o n   d e f i n i t i o n s                                    |
|                                                                              |
\*----------------------------------------------------------------------------*/

BOOL FAR PASCAL AppAbout(HWND hdlg, short msg, WORD wParam, LONG lParam);
BOOL FAR PASCAL AppAddNots(HWND hdlg, short msg, WORD wParam, LONG lParam);
BOOL FAR PASCAL AppCanNots(HWND hdlg, short msg, WORD wParam, LONG lParam);
BOOL FAR PASCAL AppChgWdw(HWND hdlg, short msg, WORD wParam, LONG lParam);
LONG FAR PASCAL AppWndProc (HWND hwnd, unsigned uiMessage, WORD wParam, LONG lParam);
int fDialog(LPSTR id,HWND hwnd,FARPROC fpfn);

LONG  NEAR PASCAL AppCommand(HWND hwnd, unsigned msg, WORD wParam, LONG lParam);
void  NEAR PASCAL AppPaint(HWND hwnd, HDC hdc);

static void copytoken(LPSTR to, LPSTR start, LPSTR end, WORD max);
static LPSTR NextToken(LPSTR FAR *b, BOOL bSkipLines);

/*----------------------------------------------------------------------------*\
|   AppInit( hInst, hPrev)						       |
|									       |
|   Description:							       |
|	This is called when the application is first loaded into	       |
|	memory.  It performs all initialization that doesn't need to be done   |
|	once per instance.						       |
|									       |
|   Arguments:								       |
|	hInstance	instance handle of current instance		       |
|	hPrev		instance handle of previous instance		       |
|									       |
|   Returns:								       |
|	TRUE if successful, FALSE if not				       |
|									       |
\*----------------------------------------------------------------------------*/
BOOL AppInit(HANDLE hInst,HANDLE hPrev,WORD sw,LPSTR szCmdLine)
{
    WNDCLASS cls;
    HMENU    hmenu;
    RECT     wrect;
    static char AAClassName[] = "Animation Player Example";

    /* Save instance handle for DialogBoxs */
    hInstApp = hInst;

    /*
     *  Get the message number used by the Player to return
     *  status and frame synchronization events
     */
    StopMsg = RegisterWindowMessage(AA_STOP);
    NotifyMsg = RegisterWindowMessage(AA_NOTIFY);

    if (!hPrev)
    {
	/*
	 *  Register a class for the main application window
	 */
        cls.hCursor        = LoadCursor(NULL,IDC_ARROW);
        cls.hIcon          = LoadIcon(hInst,MAKEINTATOM(ID_APP));
        cls.lpszMenuName   = MAKEINTATOM(ID_APP);
        cls.lpszClassName  = AAClassName;
        cls.hbrBackground  = (HBRUSH)COLOR_APPWORKSPACE + 1;
        cls.hInstance      = hInst;
        cls.style          = CS_DBLCLKS;
        cls.lpfnWndProc    = AppWndProc;
        cls.cbWndExtra     = 0;
        cls.cbClsExtra     = 0;

        if (!RegisterClass(&cls))
	    return FALSE;
    }

    wrect.top = 0;
    wrect.left = 0;
    wrect.right = 319;
    wrect.bottom = 200;
    AdjustWindowRect(&wrect, WS_OVERLAPPEDWINDOW, TRUE);

    hwndApp = CreateWindow (AAClassName,	    // Class name
                            szAppName,              // Caption
                            WS_OVERLAPPEDWINDOW,    // Style bits
                            CW_USEDEFAULT, 0,       // Position
			    wrect.right - wrect.left + 1,
			    wrect.bottom - wrect.top + 1,
                            (HWND)NULL,             // Parent window (no parent)
                            (HMENU)NULL,            // use class menu
                            (HANDLE)hInst,          // handle to window instance
                            (LPSTR)NULL             // no params to pass on
                           );
    ShowWindow(hwndApp,sw);

    /*
     *  Open the animation player library,
     *  and determine whether sound or scripting
     *  is available
     */
    aaOpen();
    fSound = aaGetCaps(AA_CAP_SOUND);
    fScript = aaGetCaps(AA_CAP_SCRIPT);

    /*
     *  If no sound, remove the menu item for sound
     */
    if (!fSound)
	DeleteMenu(GetMenu(hwndApp), MENU_SOUND, MF_BYCOMMAND);

    /*
     *  If no scripting, remove the New, Save and Save As
     *  menu items
     */
    if (!fScript) {
	DeleteMenu(hmenu = GetMenu(hwndApp), MENU_NEW, MF_BYCOMMAND);
	DeleteMenu(hmenu, MENU_SAVE, MF_BYCOMMAND);
	DeleteMenu(hmenu, MENU_SAVEAS, MF_BYCOMMAND);
    }

    /*
     *  Check the command line for arguments, the arguments
     *  on the command line can be the name of an animation,
     *  followed by a flag -R or -r, which indicates that
     *  the animation is to be started.
     */
    fCommand = 0;
    if (szCmdLine != NULL) {
	LPSTR cp;

	if ((cp = NextToken(&szCmdLine, TRUE)) != NULL) {
	    copytoken(AnimFileName, cp, szCmdLine,
		sizeof(AnimFileName));
	    if ((hAnim = aaLoad(AnimFileName, hwndApp, wMode,
		    0, 0, 0, 0, 0, 0)) != NULL) {
		if ((cp = NextToken(&szCmdLine, FALSE)) != NULL) {
		    if (*cp == '-' && (cp[1] == 'r' || cp[1] == 'R')
			    && (szCmdLine - cp) == 2) {
			fCommand = -1;
			PostMessage(hwndApp, WM_COMMAND, MENU_PLAY, 0L);
		    }
		}
	    }
	}
    }

    /*
     *  If there is not an animation, grey the
     *  menu bar items that apply to animations,
     *  Play, Stop, <<, and >>
     */
    if (hAnim == NULL) {
	EnableMenuItem(hmenu = GetMenu(hwndApp), MENU_PLAY, MF_GRAYED);
	EnableMenuItem(hmenu, MENU_STOP, MF_GRAYED);
	EnableMenuItem(hmenu, MENU_PREV, MF_GRAYED);
	EnableMenuItem(hmenu, MENU_NEXT, MF_GRAYED);
	DrawMenuBar(hwndApp);
    }

    return TRUE;
}

/*
 *  isolate the next token on the command line
 */
static LPSTR
NextToken(LPSTR FAR *b, BOOL bSkipLines)
{
    LPSTR p = *b, q;

    if (p == NULL)
	return(NULL);

    while (bSkipLines || *p != '\n') {
	if (!isspace(*p)) {
	    if (*p == '\\') {
		if (p[1] == '\n')
		    p++;
		else if (p[1] == '\r' && p[2] == '\n')
		    p += 2;
		else
		    break;
	    } else
		break;
	}
	p++;
    }

    if (*p == '\0' || *p == END_TEXTFILE || (!bSkipLines && *p == '\n')) {
	*b = p;
	return(NULL);
    }

    q = p;
    while (*p != '\0' && !isspace(*p)) {
	if (*p == '\\') {
	    if (p[1] == '\n' || (p[1] == '\r' && p[2] == '\n'))
		break;
	}
	p++;
    }

    *b = p;
    return(q);
}

/*
 *  Copy token from command line to buffer
 */
static void
copytoken(LPSTR to, LPSTR start, LPSTR end, WORD max)
{
    WORD size;

    if (to == NULL)
	return;
    size = (WORD)(end - start);
    if (size >= max)
	size = max - 1;
    _fmemmove(to, start, size);
    to[size] = '\0';
}

/*
 *  Add the frame number to the caption.
 */
static void
DisplayCaption(void)
{
    WORD frame;

    if (hAnim != NULL) {
	frame = AA_POSFRAME(aaGetParm(hAnim, AA_POSITION));
	lstrcpy(szCaption, szAppName);
	wsprintf(&szCaption[lstrlen(szCaption)], " - %u", frame);
	SetWindowText(hwndApp, szCaption);
    }
}

/*----------------------------------------------------------------------------*\
|   WinMain( hInst, hPrev, lpszCmdLine, cmdShow )			       |
|                                                                              |
|   Description:                                                               |
|       The main procedure for the App.  After initializing, it just goes      |
|       into a message-processing loop until it gets a WM_QUIT message         |
|       (meaning the app was closed).                                          |
|                                                                              |
|   Arguments:                                                                 |
|	hInst		instance handle of this instance of the app	       |
|	hPrev		instance handle of previous instance, NULL if first    |
|       szCmdLine       ->null-terminated command line                         |
|       cmdShow         specifies how the window is initially displayed        |
|                                                                              |
|   Returns:                                                                   |
|       The exit code as specified in the WM_QUIT message.                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
int PASCAL WinMain(HANDLE hInst, HANDLE hPrev, LPSTR szCmdLine, int sw)
{
    MSG     msg;

    /* Call initialization procedure */
    if (!AppInit(hInst,hPrev,sw,szCmdLine))
        return FALSE;

    /* Polling messages from event queue */
    while (GetMessage(&msg, NULL, 0, 0))
    {
        if (msg.message == WM_QUIT)
            goto exit;

        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
exit:

    aaClose();

    return msg.wParam;
}



/*----------------------------------------------------------------------------*\
|   AppPaint(hwnd, hdc) 						       |
|                                                                              |
|   Description:                                                               |
|       The paint function.  Right now this does nothing.                      |
|                                                                              |
|   Arguments:								       |
|	hwnd		 window painting into				       |
|	hdc		 display context to paint to			       |
|                                                                              |
|   Returns:                                                                   |
|       nothing                                                                |
|                                                                              |
\*----------------------------------------------------------------------------*/
#ifdef __BORLANDC__
#pragma argsused
#endif /* __BORLANDC__ */
void NEAR PASCAL AppPaint (HWND hwnd, HDC hdc)
{
}

/*----------------------------------------------------------------------------*\
|   AppWndProc( hwnd, uiMessage, wParam, lParam )			       |
|                                                                              |
|   Description:                                                               |
|       The window proc for the app's main (tiled) window.  This processes all |
|       of the parent window's messages.                                       |
|                                                                              |
|   Arguments:                                                                 |
|	hwnd		window handle for the window			       |
|       uiMessage       message number                                         |
|       wParam          message-dependent                                      |
|       lParam          message-dependent                                      |
|                                                                              |
|   Returns:                                                                   |
|       0 if processed, nonzero if ignored                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
LONG FAR PASCAL AppWndProc(hwnd, msg, wParam, lParam)
    HWND     hwnd;
    unsigned msg;
    WORD     wParam;
    long     lParam;
{
    PAINTSTRUCT ps;
    DWORD	pos;
    char	prtbuf[100];

    /*
     *  First check for Stop messages or the right mouse button
     *  We stop the animation when one of these messages happen
     */
    if (msg == StopMsg || msg == WM_RBUTTONDOWN) {
	/*
	 *  Stop the animation
	 */
	aaStop(hAnim);
	DisplayCaption();
	if (msg == StopMsg) {
	    /*
	     *  Display a message when an error occurs
	     */
	    if (lParam != 0L) {
		wsprintf(prtbuf,
		    "The error %ld was returned from the animation player.",
		    lParam);

		MessageBox(hwndApp, prtbuf,
		    "Animation Error", MB_APPLMODAL | MB_ICONEXCLAMATION);
	    } else if (ShowStops) {
		MessageBox(hwndApp, "A stop message was received.",
		    "Animation Stop", MB_APPLMODAL | MB_ICONASTERISK);
	    }
	}
    } else if (msg == NotifyMsg) {
	/*
	 *  Next check for frame synchronization
	 */
	pos = aaGetParm(hAnim, AA_POSITION);
	wsprintf(prtbuf, "Notifcation @%d:%d, parameter %lu.",
	    AA_POSLOOPS(pos), AA_POSFRAME(pos), lParam);
	MessageBox(hwndApp, prtbuf, "Notify Message",
	    MB_APPLMODAL | MB_ICONASTERISK);
    } else {
	/*
	 *  Now all of the other stuff
	 */
	switch (msg) {
	    case WM_CREATE:
		break;

	    case WM_ERASEBKGND:
		break;

	    case WM_KEYDOWN:
		break;

	    case WM_ACTIVATEAPP:
		break;

	    case WM_TIMER:
		break;

	    case WM_INITMENU:
		/*
		 *  Enable and disable menu items. Basicly
		 *  items are disable there is no animation,
		 *  or the animation is not the correct type
		 */
		EnableMenuItem(wParam, MENU_NEW,
		    fScript ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(wParam, MENU_SAVE,
		    aaGetParm(hAnim, AA_FILETYPE) == AA_SCRIPT
		    ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(wParam, MENU_SAVEAS,
		    aaGetParm(hAnim, AA_FILETYPE) == AA_SCRIPT
		    ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(wParam, MENU_SET,
		    hAnim != NULL ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(wParam, MENU_SOUND,
		    (hAnim != NULL && fSound
			&& aaGetParm(hAnim, AA_FILETYPE) != AA_SCRIPT)
		    ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(wParam, MENU_ADD_NOTIFY,
		    hAnim != NULL ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(wParam, MENU_CANCEL_NOTIFY,
		    hAnim != NULL ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(wParam, MENU_CHANGE_WINDOW,
		    hAnim != NULL ? MF_ENABLED : MF_GRAYED);
		CheckMenuItem (wParam, MENU_MEMORY,
		    (wMode & AA_MEMORYLOAD) ? MF_CHECKED : MF_UNCHECKED);
		CheckMenuItem (wParam, MENU_LOOPFRAME,
		    (wMode & AA_LOOPFRAME) ? MF_CHECKED : MF_UNCHECKED);
		CheckMenuItem (wParam, MENU_FULLSCREEN,
		    (wMode & AA_FULLSCREEN) ? MF_CHECKED : MF_UNCHECKED);
		CheckMenuItem (wParam, MENU_HIDEWINDOW,
		    (wMode & AA_HIDEWINDOW) ? MF_CHECKED : MF_UNCHECKED);
		CheckMenuItem (wParam, MENU_NOPALETTE,
		    !(wMode & AA_NOPALETTE) ? MF_CHECKED : MF_UNCHECKED);
		if (wMode & AA_NOPALETTE) {
		    CheckMenuItem (wParam, MENU_RESERVEPALETTE, MF_UNCHECKED);
		    EnableMenuItem(wParam, MENU_RESERVEPALETTE, MF_GRAYED);
		} else {
		    EnableMenuItem(wParam, MENU_RESERVEPALETTE, MF_ENABLED);
		    CheckMenuItem (wParam, MENU_RESERVEPALETTE,
			(wMode & AA_RESERVEPALETTE) ? MF_CHECKED
			: MF_UNCHECKED);
		}
		CheckMenuItem (wParam, MENU_STOPSTATUS,
		    (wMode & AA_STOPSTATUS) ? MF_CHECKED : MF_UNCHECKED);
		CheckMenuItem (wParam, MENU_STOPNOTIFY,
		    (wMode & AA_STOPNOTIFY) ? MF_CHECKED : MF_UNCHECKED);
		CheckMenuItem (wParam, MENU_SHOWSTOPS,
		    ShowStops ? MF_CHECKED : MF_UNCHECKED);
		break;

	    case WM_COMMAND:
		return AppCommand(hwnd,msg,wParam,lParam);

	    case WM_DESTROY:
		PostQuitMessage(0);
		break;

	    case WM_CLOSE:
		if (hAnim != NULL)
		    aaUnload(hAnim);
		break;

	    case WM_LBUTTONDBLCLK:
		break;

	    case WM_PALETTECHANGED:
		if (wParam == hwnd)
		    break;

		// fall through

	    case WM_QUERYNEWPALETTE:
		/* if palette realization causes a palette change,
		** we need to do a full redraw.
		*/

		return 0L;

	    case WM_PAINT:
		BeginPaint(hwnd,&ps);
		AppPaint (hwnd,ps.hdc);
		EndPaint(hwnd,&ps);
		return 0L;

	}
    }
    return DefWindowProc(hwnd,msg,wParam,lParam);
}

#ifdef __BORLANDC__
#pragma argsused
#endif /* __BORLANDC__ */
LONG NEAR PASCAL AppCommand (hwnd, msg, wParam, lParam)
    HWND     hwnd;
    unsigned msg;
    WORD     wParam;
    long     lParam;
{
    HMENU hmenu;
    int    fh;
    HAA local;

    fCommand++;
    if (fCommand > 1)
	fCommand--;

    switch(wParam)
    {
	/*
	 *  Stop the animation
	 */
	case MENU_STOP:
	    if (hAnim != NULL) {
		aaStop(hAnim);
		DisplayCaption();
	    }
	    break;

	/*
	 *  Pause the animation, if it is playing
	 *  otherwise Play the animation
	 */
	case MENU_PLAY:
	    if (hAnim != NULL) {
		if (aaGetParm(hAnim, AA_STATUS) != AA_PLAYING) {
		    aaPlay(hAnim);
		} else {
		    aaPause(hAnim);
		    DisplayCaption();
		}
	    }
	    break;

	/*
	 *  show the about box
	 */
        case MENU_ABOUT:
            fDialog("ABOUTBOX",hwnd,AppAbout);
            break;

        case MENU_EXIT:
	    PostMessage(hwnd,WM_CLOSE,0,0L);
            break;

	/*
	 *  Show next frame
	 */
	case MENU_NEXT:
	    if (hAnim != NULL) {
		aaSetParm(hAnim, AA_POSITION, SEEK_CUR, 1);
		DisplayCaption();
	    }
	    break;

	/*
	 *  Show previous frame
	 */
	case MENU_PREV:
	    if (hAnim != NULL) {
		if (aaGetParm(hAnim, AA_POSITION) == 0)
		    aaSetParm(hAnim, AA_POSITION, SEEK_SET,
			AA_LONGPOS(1,0));
		aaSetParm(hAnim, AA_POSITION, SEEK_CUR, -1);
		DisplayCaption();
	    }
	    break;

	/*
	 *  Change load into memory option
	 */
	case MENU_MEMORY:
	    wMode ^= AA_MEMORYLOAD;
	    if (hAnim != NULL
		    && !aaSetParm(hAnim, AA_MODE, wMode, (DWORD)~AA_MEMORYLOAD)) {
		MessageBox(hwndApp,
		    "Could not change for the loaded animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    }
	    break;

	/*
	 *  Change loop frame option
	 */
	case MENU_LOOPFRAME:
	    wMode ^= AA_LOOPFRAME;
	    if (hAnim != NULL
		    && !aaSetParm(hAnim, AA_MODE, wMode, (DWORD)~AA_LOOPFRAME)) {
		MessageBox(hwndApp,
		    "Could not change for the loaded animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    }
	    DisplayCaption();
	    break;

	/*
	 *  Change full screen option
	 */
	case MENU_FULLSCREEN:
	    wMode ^= AA_FULLSCREEN;
	    if (hAnim != NULL
		    && !aaSetParm(hAnim, AA_MODE, wMode, (DWORD)~AA_FULLSCREEN)) {
		MessageBox(hwndApp,
		    "Could not change for the loaded animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    }
	    break;

	/*
	 *  Change hide window option
	 */
	case MENU_HIDEWINDOW:
	    wMode ^= AA_HIDEWINDOW;
	    if (hAnim != NULL
		    && !aaSetParm(hAnim, AA_MODE, wMode, ~AA_HIDEWINDOW)) {
		MessageBox(hwndApp,
		    "Could not change for the loaded animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    } else
		aaShow(hAnim, wMode & AA_HIDEWINDOW ? FALSE : TRUE);
	    break;

	/*
	 *  Change palette animation option
	 */
	case MENU_NOPALETTE:
	    wMode ^= AA_NOPALETTE;
	    if (hAnim != NULL
		    && !aaSetParm(hAnim, AA_MODE, wMode, (DWORD)~AA_NOPALETTE)) {
		MessageBox(hwndApp,
		    "Could not change for the loaded animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    }
	    break;

	/*
	 * Change reserve palette animation option
	 */
	case MENU_RESERVEPALETTE:
	    wMode ^= AA_RESERVEPALETTE;
	    if (hAnim != NULL
		    && !aaSetParm(hAnim, AA_MODE, wMode,
		        (DWORD)~AA_RESERVEPALETTE)) {
		MessageBox(hwndApp,
		    "Could not change for the loaded animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    }
	    break;

	/*
	 *  Change option to show normal status messages
	 */
	case MENU_SHOWSTOPS:
	    ShowStops = !ShowStops;
	    break;

	/*
	 *  Change stop status option
	 */
	case MENU_STOPSTATUS:
	    wMode ^= AA_STOPSTATUS;
	    if (hAnim != NULL
		    && !aaSetParm(hAnim, AA_MODE, wMode, (DWORD)~AA_STOPSTATUS)) {
		MessageBox(hwndApp,
		    "Could not change for the loaded animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    }
	    break;

	/*
	 *  Change stop frame synchornization option
	 */
	case MENU_STOPNOTIFY:
	    wMode ^= AA_STOPNOTIFY;
	    if (hAnim != NULL
		    && !aaSetParm(hAnim, AA_MODE, wMode, (DWORD)~AA_STOPNOTIFY)) {
		MessageBox(hwndApp,
		    "Could not change for the loaded animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    }
	    break;

	/*
	 *  Add frame synchronization request
	 */
	case MENU_ADD_NOTIFY:
	    if (hAnim != NULL)
		fDialog("AA_ADDNOTS",hwnd,AppAddNots);
	    break;

	/*
	 *  Cancel frame synchronization request
	 */
	case MENU_CANCEL_NOTIFY:
	    if (hAnim != NULL)
		fDialog("AA_CANNOTS",hwnd,AppCanNots);
	    break;

	/*
	 *  Change the parent window
	 */
	case MENU_CHANGE_WINDOW:
	    if (hAnim != NULL)
		aaSetParm(hAnim, AA_WINDOW,
		    (HWND)fDialog("AA_CHGWDW",hwnd,AppChgWdw), 0L);
	    break;

	/*
	 *  Create a new empty script, and edit it
	 */
	case MENU_NEW:
	    if ((local = aaLoad("", hwndApp, wMode | AA_BUILDSCRIPT,
		    0, 0, 0, 0, 0, 0)) != NULL) {
		aaUnload(hAnim);
		hAnim = local;
		aaPrompt(hAnim, AnimFileName);

		/*
		 *  If there are no animation in the script,
		 *  grey words on menu bar
		 */
		if (aaGetParm(hAnim, AA_ANIMATIONCOUNT) != 0) {
		    fh = MF_ENABLED;
		} else {
		    fh = MF_GRAYED;
		}
		EnableMenuItem(hmenu = GetMenu(hwndApp),
		    MENU_PLAY, fh);
		EnableMenuItem(hmenu, MENU_STOP, fh);
		EnableMenuItem(hmenu, MENU_PREV, fh);
		EnableMenuItem(hmenu, MENU_NEXT, fh);
		DrawMenuBar(hwndApp);
		DisplayCaption();
	    } else {
		MessageBox(hwndApp,
		    "AALoad could not load the animation.",
		    "Animation Error",
		    MB_APPLMODAL | MB_ICONEXCLAMATION);
	    }
	    break;

	/*
	 *  Save a script
	 */
	case MENU_SAVE:
	    aaSave(hAnim, 0);
	    break;

	/*
	 *  Save a script under a new name
	 */
	case MENU_SAVEAS:
	    aaSave(hAnim, AA_SAVE_AS);
	    break;

	/*
	 *  Edit animation parameters and scripts
	 */
	case MENU_SET:
	    aaPrompt(hAnim, AnimFileName);

	    /*
	     *  If all of the animations are removed
	     *  from a script, grey menu bar
	     */
	    if (aaGetParm(hAnim, AA_ANIMATIONCOUNT) != 0) {
		fh = MF_ENABLED;
	    } else {
		fh = MF_GRAYED;
	    }
	    EnableMenuItem(hmenu = GetMenu(hwndApp),
		MENU_PLAY, fh);
	    EnableMenuItem(hmenu, MENU_STOP, fh);
	    EnableMenuItem(hmenu, MENU_PREV, fh);
	    EnableMenuItem(hmenu, MENU_NEXT, fh);
	    DrawMenuBar(hwndApp);
	    DisplayCaption();
	    break;

	/*
	 *  Open an animation
	 */
        case MENU_OPEN:
	    /*
	     *  Get the animation name
	     */
            fh = aaGetFile(AA_GETFILE_OPEN | AA_GETFILE_MUSTEXIST
		| AA_GETFILE_USEDIR, AnimFileName, sizeof(AnimFileName),
		NULL, 0
            );

            if (fh != 0)
            {
		/*
		 *  Open the animation. If it works unload
		 *  previous animation
		 */
                if ((local = aaLoad(AnimFileName, hwndApp,
			wMode, 0, 0, 0, 0, 0, 0)) == NULL) {
		    MessageBox(hwndApp,
			"AALoad could not load the animation.",
			"Animation Error",
			MB_APPLMODAL | MB_ICONEXCLAMATION);
		    break;
		} else {
		    aaUnload(hAnim);
		    hAnim = local;
		}

		/*
		 *  Grey menu bar if no animations
		 */
		if (aaGetParm(hAnim, AA_ANIMATIONCOUNT) != 0) {
		    fh = MF_ENABLED;
		} else {
		    fh = MF_GRAYED;
		}
		EnableMenuItem(hmenu = GetMenu(hwndApp),
		    MENU_PLAY, fh);
		EnableMenuItem(hmenu, MENU_STOP, fh);
		EnableMenuItem(hmenu, MENU_PREV, fh);
		EnableMenuItem(hmenu, MENU_NEXT, fh);
		DrawMenuBar(hwndApp);
		DisplayCaption();
            }
            break;

	/*
	 *  Add sound to the animation
	 */
	case MENU_SOUND:
	    if (hAnim != NULL && fSound
		    && aaGetParm(hAnim, AA_FILETYPE) != AA_SCRIPT) {
		fh = aaGetFile(AA_GETFILE_OPEN | AA_GETFILE_MUSTEXIST
		    | AA_GETFILE_USEDIR | AA_GETFILE_SOUND,
		    SndFileName, sizeof(SndFileName),
		    SndDevice, sizeof(SndDevice)
		);

		if (fh != 0)
		{
		    if (!aaSound(hAnim, SndDevice, SndFileName, 0)) {
			MessageBox(hwndApp,
			    "AALoad could not load the sound.",
			    "Animation Error",
			    MB_APPLMODAL | MB_ICONEXCLAMATION);
		    }
		}
	    }
            break;

	default:
	    break;
    }
    return 0L;
}

/*----------------------------------------------------------------------------*\
|   fDialog(id,hwnd,fpfn)						       |
|									       |
|   Description:                                                               |
|	This function displays a dialog box and returns the exit code.	       |
|	the function passed will have a proc instance made for it.	       |
|									       |
|   Arguments:                                                                 |
|	id		resource id of dialog to display		       |
|	hwnd		parent window of dialog 			       |
|	fpfn		dialog message function 			       |
|                                                                              |
|   Returns:                                                                   |
|	exit code of dialog (what was passed to EndDialog)		       |
|                                                                              |
\*----------------------------------------------------------------------------*/
int fDialog(LPSTR id,HWND hwnd,FARPROC fpfn)
{
    int		f;
    HANDLE	hInst;

    hInst = GetWindowWord(hwnd,GWW_HINSTANCE);
    fpfn  = MakeProcInstance(fpfn,hInst);
    f = DialogBox(hInst,id,hwnd,fpfn);
    FreeProcInstance (fpfn);
    return f;
}

/*----------------------------------------------------------------------------*\
|   AppAbout( hDlg, uiMessage, wParam, lParam ) 			       |
|									       |
|   Description:							       |
|	This function handles messages belonging to the "About" dialog box.    |
|	The only message that it looks for is WM_COMMAND, indicating the use   |
|	has pressed the "OK" button.  When this happens, it takes down	       |
|	the dialog box. 						       |
|									       |
|   Arguments:								       |
|	hDlg		window handle of about dialog window		       |
|	uiMessage	message number					       |
|	wParam		message-dependent				       |
|	lParam		message-dependent				       |
|									       |
|   Returns:								       |
|	TRUE if message has been processed, else FALSE			       |
|									       |
\*----------------------------------------------------------------------------*/
#ifdef __BORLANDC__
#pragma argsused
#endif /* __BORLANDC__ */
BOOL FAR PASCAL AppAbout(HWND hdlg, short msg, WORD wParam, LONG lParam)
{
    switch (msg) {
    case WM_CLOSE:
	EndDialog(hdlg, TRUE);
	return(TRUE);

    case WM_COMMAND:
	switch (wParam) {
	case IDOK:
	    EndDialog(hdlg, TRUE);
	    return(TRUE);
	}
	break;

    case WM_INITDIALOG:
	return TRUE;
    }
    return FALSE;
}

/*
 *  Process an edit control message.
 *  hdlg is the dialog window handle,
 *  item is the edit control control id.
 *  lpV is a pointer to where the data is to be stored.
 *  type is the message to process
 *  active is item if the control was changed.
 *  proc is a procedure for storing the data at *lpV.
 *  active is returned for most messages. 0 is returned for
 *	SETFOCUS and KILLFOCUS messages, and item is returned
 *	when the control is changed. Data is not stored
 *	until the KILLFOCUS message is receivced.
 */
WORD
EditControl(HWND hdlg, WORD item, LPVOID lpV, WORD type, WORD active,
    GETPROC proc)
{
    char buffer[32];
    WORD size;

    switch (type) {
    case EN_CHANGE:
	return(item);
    case EN_SETFOCUS:
	return(0);
    case EN_KILLFOCUS:
	if (active == item) {
	    /*
	     *  Control changed, get string, conver and store
	     */
	    size = GetDlgItemText(hdlg, item, buffer, sizeof(buffer) - 1);
	    (*proc)(buffer, lpV, &buffer[size]);
	    SetDlgItemText(hdlg, item, buffer);
	}
	return(0);
    }
    return(active);
}

/*
 *   Trim leading and trailing blanks
 */
static LPSTR
trim(LPSTR s, LPSTR FAR *ep)
{
    LPSTR c;
    LPSTR end = *ep;

    while (s < end && isspace(*s))
	s++;

    if (s < end) {
	for (c = end; --c != s && isspace(*c); );
    }

    *ep = end;
    return(s);
}

/*
 *  convert string to int
 */
static WORD
atouint(LPSTR s, LPVOID c, LPSTR end)
{
    WORD n = 0, d;

    while (s < end && isdigit(*s)) {
	d = *s - '0';

	/*
	 *  stop assembling the number if this iteration
	 *  will cause it to overflow an unsigned integer
	 */
	if (n > (UINT_MAX / 10) || (n == (UINT_MAX / 10)
		&& d > (UINT_MAX % 10)))
	    break;
	n = (n * 10) + d;
	s++;
    }

    *(LPSTR FAR *)c = s;
    return(n);
}

/*
 *  convert string to long
 */
static void
atoulong(LPSTR s, LPVOID c, LPSTR end)
{
    DWORD n = 0;
    WORD d;

    while (s < end && isdigit(*s)) {
	d = *s - '0';

	/*
	 *  stop assembling the number if this iteration
	 *  will cause it to overflow an unsigned integer
	 */
	if (n > (ULONG_MAX / 10) || (n == (ULONG_MAX / 10)
		&& d > (ULONG_MAX % 10)))
	    break;
	n = (n * 10) + d;
	s++;
    }

    *(LPDWORD)c = n;
    *s = '\0';
}


/*
 *  convert string to loops value
 *  this string is formatted loops:frames
 *  The number of frames is optional.
 */
static void
atoloops(LPSTR s, LPVOID dwp, LPSTR end)
{
    LPSTR c;
    WORD loops, frames;

    s = trim(s, &end);

    if (lstrlen(AAForever) == (end - s)
	    && _fstrnicmp(s, AAForever, (WORD)(end - s)) == 0) {
	*(LPDWORD)dwp = 0;
	*end = '\0';
	return;
    }

    for (c = s; c < end && *c != ':'; c++);
    if (lstrlen(AALoopSound) == (c - s)
	    && _fstrnicmp(s, AALoopSound, (WORD)(c - s)) == 0) {
	loops = AA_LOOPSOUND;
	frames = AA_LOOPSOUND;
    } else {
	/*
	 *  Convert loops, and frames, if present
	 */
	loops = atouint(s, &c, end);
	frames = 0;
    }

    if (*c == ':')
	frames = atouint(c + 1, &c, end);

    /*
     *  Set the loop value, and check for valid string
     */
    *(LPDWORD)dwp = AA_LONGPOS(loops, frames);
    *c = '\0';
}

/*
 *  Set the string representing loops in an edit control
 */
static void
SetDlgItemLoops(HWND hDlg, WORD wId, DWORD dwLoops)
{
    char string[20];

    wsprintf(string, "%u:%u", AA_POSLOOPS(dwLoops), AA_POSFRAME(dwLoops));
    SetDlgItemText(hDlg, wId, string);
}

/*
 *  Set the string representing Long in an edit control
 */
static void
SetDlgItemLong(HWND hDlg, WORD wId, DWORD dwLong, BOOL bSigned)
{
    char string[20];

    wsprintf(string, bSigned ? "%ld" : "%lu", dwLong);
    SetDlgItemText(hDlg, wId, string);
}

/*
 *  Handle dialog for adding frame synchronization messages
 */
BOOL FAR PASCAL AppAddNots(HWND hdlg, short msg, WORD wParam, LONG lParam)
{
    static DWORD loop, param;
    static WORD active = 0;

    switch (msg) {
    case WM_COMMAND:
	switch (wParam) {
	case IDOK:
	    switch (active) {
	    case TN_LOOP:
		active = EditControl(hdlg, TN_LOOP, &loop,
		    EN_KILLFOCUS, active, atoloops);
		break;

	    case TN_PARAM:
		active = EditControl(hdlg, TN_PARAM, &param,
		    EN_KILLFOCUS, active, atoulong);
		break;
	    }

	    if (!aaNotify(hAnim, loop, param))
		MessageBox(hdlg, "aaNotify failed.", "Notify Error",
		    MB_OK | MB_ICONEXCLAMATION);
	    EndDialog(hdlg, TRUE);
	    return(TRUE);

	case IDCANCEL:
	    EndDialog(hdlg, FALSE);
	    return(TRUE);

	case TN_LOOP:
	    active = EditControl(hdlg, TN_LOOP, &loop,
		HIWORD(lParam), active, atoloops);
	    return(FALSE);

	case TN_PARAM:
	    active = EditControl(hdlg, TN_PARAM, &param,
		HIWORD(lParam), active, atoulong);
	    return(FALSE);
	}
	break;

    case WM_INITDIALOG:
	loop = 0;
	param = 0;
	SetDlgItemLoops(hdlg, TN_LOOP, loop);
	SetDlgItemLong(hdlg, TN_PARAM, param, FALSE);
	active = 0;
	return TRUE;
    }
    return FALSE;
}


/*
 *  Handle dialog for canceling frame synchronization messages
 */
BOOL FAR PASCAL AppCanNots(HWND hdlg, short msg, WORD wParam, LONG lParam)
{
    static DWORD loop, eloop;
    static WORD active;
    char buffer[80];

    switch (msg) {
    case WM_COMMAND:
	switch (wParam) {
	case IDOK:
	    switch (active) {
	    case TN_LOOP:
		active = EditControl(hdlg, TN_LOOP, &loop,
		    EN_KILLFOCUS, active, atoloops);
		break;

	    case TN_ELOOP:
		active = EditControl(hdlg, TN_ELOOP, &eloop,
		    EN_KILLFOCUS, active, atoloops);
		break;
	    }

	    wsprintf(buffer, "aaCancel removed %d notifications.",
		aaCancel(hAnim, loop, eloop));
	    MessageBox(hdlg, buffer, "Notifications Canceled",
		MB_OK | MB_ICONEXCLAMATION);
	    EndDialog(hdlg, TRUE);
	    return(TRUE);

	case IDCANCEL:
	    EndDialog(hdlg, -1);
	    return(TRUE);

	case TN_LOOP:
	    active = EditControl(hdlg, TN_LOOP, &loop,
		HIWORD(lParam), active, atoloops);
	    return(FALSE);

	case TN_ELOOP:
	    active = EditControl(hdlg, TN_ELOOP, &eloop,
		HIWORD(lParam), active, atoloops);
	    return(FALSE);
	}
	break;

    case WM_INITDIALOG:
	loop = 0;
	eloop = AA_LONGPOS(999,65535U);
	SetDlgItemLoops(hdlg, TN_LOOP, loop);
	SetDlgItemLoops(hdlg, TN_ELOOP, eloop);
	active = 0;
	return TRUE;
    }
    return FALSE;
}

/*
 *  Display the window info in dialog box, and
 *  Outline window. (At least try).
 */
void
ShowWindowSelect(HWND hdlg, HWND window, BOOL bOn)
{
    HDC hDc;
    RECT wrect;
    HWND parent;
    HRGN clip;
    LPSTR cp;
    char buffer[80];
    static char undef[] = "<    Undefined   >";

    if (bOn) {
	if (window == NULL) {
	    SetDlgItemText(hdlg, CW_HANDLE, undef);
	    SetDlgItemText(hdlg, CW_CAPTION, undef);
	    SetDlgItemText(hdlg, CW_CLASS, undef);
	    SetDlgItemText(hdlg, CW_PARENT, undef);
	} else {
	    wsprintf(buffer, "%X(%u)", (WORD)window, (WORD)window);
	    SetDlgItemText(hdlg, CW_HANDLE, buffer);
	    buffer[GetWindowText(window, buffer, sizeof(buffer) - 1)] = '\0';
	    SetDlgItemText(hdlg, CW_CAPTION, buffer);
	    buffer[GetClassName(window, buffer, sizeof(buffer) - 1)] = '\0';
	    SetDlgItemText(hdlg, CW_CLASS, buffer);
	    if ((parent = GetParent(window)) == NULL)
		SetDlgItemText(hdlg, CW_PARENT, "<    No Parent   >");
	    else {
		wsprintf(buffer, "%X(%u):", (WORD)parent, (WORD)parent);
		cp = &buffer[lstrlen((LPSTR)buffer)];
		cp[GetWindowText(parent, cp,
		    sizeof(buffer) - 1 - (WORD)(cp - (LPSTR)buffer))] = '\0';
		SetDlgItemText(hdlg, CW_PARENT, buffer);
	    }
	}
    }

    if (window != NULL) {
	if ((hDc = GetDC(window)) == NULL)
	    return;

	GetWindowRect(window, &wrect);
	ScreenToClient(window, (LPPOINT)&wrect);
	ScreenToClient(window, ((LPPOINT)&wrect) + 1);
	clip = CreateRectRgnIndirect(&wrect);
	SelectObject(hDc, GetStockObject(WHITE_BRUSH));
	SelectClipRgn(hDc, clip);
	DeleteObject(clip);

	PatBlt(hDc, wrect.left, wrect.top,
	    wrect.right - wrect.left, CW_WIDTH, PATINVERT);
	PatBlt(hDc, wrect.right - CW_WIDTH, wrect.top + CW_WIDTH,
	    CW_WIDTH, wrect.bottom - wrect.top - (2 * CW_WIDTH), PATINVERT);
	PatBlt(hDc, wrect.left, wrect.bottom - CW_WIDTH,
	    wrect.right - wrect.left, CW_WIDTH, PATINVERT);
	PatBlt(hDc, wrect.left, wrect.top + CW_WIDTH,
	    CW_WIDTH, wrect.bottom - wrect.top - (2 * CW_WIDTH), PATINVERT);
	ReleaseDC(window, hDc);
    }
}

/*
 *  Return the handle to window under the mouse
 */
HWND
GetWindowUnderMouse(HWND hdlg, HWND was, HWND none)
{
    POINT p;
    HWND is, child;
    RECT wrect;

    GetCursorPos(&p);

    for (is = GetWindow(hwndApp, GW_HWNDFIRST); is != NULL;
	    is = GetWindow(is, GW_HWNDNEXT)) {
	GetWindowRect(is, &wrect);
	if (IsWindowVisible(is) && PtInRect(&wrect, p))
	    break;
    }

    if (is != NULL) {
	ScreenToClient(is, &p);
	while ((child = ChildWindowFromPoint(is, p)) != is
		&& child != NULL) {
	    ClientToScreen(is, &p);
	    is = child;
	    ScreenToClient(is, &p);
	}
    } else
	is = none;

    if (was != is) {
	ShowWindowSelect(hdlg, was, FALSE);
	ShowWindowSelect(hdlg, is, TRUE);
    }
    return(is);
}

/*
 *  Return new parent window for animation
 */
BOOL FAR PASCAL AppChgWdw(HWND hdlg, short msg, WORD wParam, LONG lParam)
{
    static HWND capture;
    static HWND hWndPick;
    static BOOL bFrozen;

    switch (msg) {
    case WM_SETCURSOR:
	switch(HIWORD(lParam)) {
	case WM_MOUSEMOVE:
	    if (!bFrozen)
		hWndPick = GetWindowUnderMouse(hdlg,
		    hWndPick, hWndPick);
	    break;

	case WM_LBUTTONDOWN:
	    if (!bFrozen) {
		hWndPick = GetWindowUnderMouse(hdlg, hWndPick, NULL);
		bFrozen = TRUE;
	    }
	    break;

	default:
	    return(FALSE);
	}
	return(FALSE);

    case WM_COMMAND:
	switch (wParam) {
	case IDCANCEL:
	    ShowWindowSelect(hdlg, hWndPick, FALSE);
	    hWndPick = NULL;

	    /*  Fall into ... */
	case IDOK:
	    ShowWindowSelect(hdlg, hWndPick, FALSE);


	    if (capture == NULL)
		ReleaseCapture();
	    else
		SetCapture(capture);

	    EndDialog(hdlg, hWndPick);
	    break;

	default:
	    return(FALSE);
	}
	break;

    case WM_INITDIALOG:
	hWndPick = NULL;
	bFrozen = FALSE;
	capture = SetCapture(hdlg);
	ShowWindowSelect(hdlg, NULL, TRUE);
	break;

    default:
	return(FALSE);
    }

    return(TRUE);
}
