/*
 *  aaplay.h
 *
 *	Copyright (C) 1990, Autodesk, Inc.
 *
 *  This file contains the public definitions for AAPLAY.DLL. This file
 *  should be included by applications that use AAPLAY.DLL.
 */

#ifndef AAPLAY_INCLUDE
#define AAPLAY_INCLUDE

#define	AAEXPORT _export

typedef WORD		HAA;		/* Handle to an animation */

/*
 *  Type that holds speed for animation.
 */
typedef WORD		AASPEED;

/*
 *  Type that holds transition for animation.
 */
typedef WORD		AATRN;

/*
 *  AAOpen and AAClose are not really needed, except there is
 *  a bug in Windows which prevents Libraries from being
 *  freed in the Windows Exit Proc (WEP). So we use AAClose
 *  to free the Libraries when the last task closes the
 *  AAPlay DLL.
 */
BOOL FAR AAEXPORT PASCAL aaOpen(void);
void FAR AAEXPORT PASCAL aaClose(void);

/*
 *  aaLoad loads an animation. The file name is in lpzFileName
 *  and the loading mode is in wMode.
 */
HAA FAR AAEXPORT PASCAL aaLoad(LPSTR lpzFileName, HWND hWnd, WORD wMode,
    int x, int y, int width, int height, int orgx, int orgy);

/*
 *  aaReLoad will load a new animation file into an old animation
 *  handle. Notifications are lost, but the palette and window are
 *  retained.
 */
BOOL FAR AAEXPORT PASCAL aaReLoad(HAA hAa, LPSTR lpzFileName, WORD wMode, WORD wMask);

/*
 *  Valid flags in wMode. The low order eight bits of wMode
 *  is used in aa_flags.
 */
#define AA_MEMORYLOAD		0x0001	/* Load into memory */
#define AA_HIDEWINDOW		0x0002	/* Hide the animation window */
#define AA_NOPALETTE		0x0004	/* Prevent palette animation */
#define AA_RESERVEPALETTE	0x0008	/* Reserve entire palette at start */
#define AA_LOOPFRAME		0x0010	/* Loaded with a loop frame */
#define AA_FULLSCREEN		0x0020	/* Use full screen for playing */
#define AA_STOPNOTIFY		0x0040	/* Prevent any notification messages */
#define AA_STOPSTATUS		0x0080	/* Prevent status messages */
#define AA_NOFAIL		0x0100	/* Reduce load type on fail */
#define AA_DONTPAINT		0x0200	/* Don't paint animation when loading */
#define AA_BUILDSCRIPT		0x0400	/* lpzFileName is script not name */
#define AA_EMBEDDEDFILE		0x0800	/* lpzFileName is imbedded struct */
#define AA_ALLMODES		0x00FF


typedef	LONG (FAR AAEXPORT PASCAL *AASEEKPROC)(LONG instance,
    LONG offset, int origin);
typedef DWORD (FAR AAEXPORT PASCAL *AAREADPROC)(LONG instance,
    LPSTR buffer, DWORD length);

typedef struct tagAAEMBED {
    LONG	ae_instance;
    AASEEKPROC	ae_seek;
    AAREADPROC	ae_read;
} AAEMBED, FAR *LPAAEMBED;

/*
 *  aaUnload unloads a loaded animation. FALSE is returned if
 *  hAa is not the handle of a loaded animation.
 */
BOOL FAR AAEXPORT PASCAL aaUnload(HAA hAa);

/*
 *  aaSound will open or close a sound for the animation. The sound
 *  is opened if the file arguement is not null and not blank,
 *  otherwise the sound is closed. If device is null, the format
 *  of the file is used to select an appropriate device.
 */
BOOL FAR AAEXPORT PASCAL aaSound(HAA hAa, LPSTR device, LPSTR file, WORD wMode);

/*
 *  Valid flags for the sound mode.
 */
#define AA_SNDFREEZE		0x0001	/* Freeze frames until sound begins */
#define AA_SNDDEVICEID		0x0100	/* device is an ID not a name */
#define AA_SNDBUILDALIAS	0x0200	/* create sound device alias */

/*
 *  aaPlay plays a loaded animation. TRUE is returned if the
 *  animation is not stopped when aaPlay returns.
 */
BOOL FAR AAEXPORT PASCAL aaPlay(HAA hAa);


/*
 *  aaNotify allows an application to be notified at specific frames
 *  when an animation is playing. lPosition is the position at which
 *  the notification is to take place. The wParam for this message is
 *  hAa, and the lParam is copied from this call. Returns TRUE if the
 *  notification is set.
 */
BOOL FAR AAEXPORT PASCAL aaNotify(HAA hAa, DWORD lPosition, DWORD lParam);

#define AA_LONGPOS(l,f)		MAKELONG(f,l)
#define AA_POSLOOPS(l)		HIWORD(l)
#define AA_POSFRAME(l)		LOWORD(l)

/*
 *  This value for the loops will cause the animation to end when
 *  the sound has finished. If no sound is playing, the animation
 *  will loop forever.
 */
#define AA_LOOPSOUND		0xFFFF

/*
 *  An automatic notify is sent when an animation in a script is
 *  reloaded. The lParam of this message is defined below
 */
#define AA_ANIMATIONLOADED	0L

/*
 *  aaCancel allows an applicatin to cancel notifications set by aaNotify.
 *  lLoPos and lHiPos give lower and upper limits on positions. Returns
 *  the number of notifications canceled.
 */
WORD FAR AAEXPORT PASCAL aaCancel(HAA hAa, DWORD lLoPos, DWORD lHiPos);


/*
 *  aaStop will stop a playing animation. TRUE is returned if the
 *  animation is stopped when aaStop returns.
 */
BOOL FAR AAEXPORT PASCAL aaStop(HAA hAa);


/*
 *  aaPause will pause an animation. TRUE is returned if the animation
 *  is paused when aaPause returns. To resume a paused animation, use
 *  aaPlay.
 */
BOOL FAR AAEXPORT PASCAL aaPause(HAA hAa);


/*
 *  aaPrompt will produce a dialog to prompt for user input. When
 *  input is accepted, the handle is changed to reflect the new
 *  parameters. The old handle is not destroyed until the new handle
 *  has been created. aaPrompt returns NULL if the new handle cannot
 *  be created, otherwise the new handle is returned.
 */
BOOL FAR AAEXPORT PASCAL aaPrompt(HAA hAa, LPSTR lpName);


/*
 *  The parameter structure to be used for the animation.
 */
typedef struct {
    BYTE	aa_status;	/*  Current status of animation */
    BYTE	aa_filetype;	/*  Type of file on disk  */
    BYTE	aa_mode;	/*  Some flags */
    BYTE	aa_bitpix;	/*  bits per pixel */

    HWND	aa_window;	/*  Handle of window for status messages */
    int		aa_x,		/*  Left of display window */
		aa_y,		/*  Top of display window */
		aa_cx,		/*  Width of display window */
		aa_cy;		/*  Height of display window */
    int		aa_orgx,	/*  Point in the animation displayed */
		aa_orgy;	/*  in the upper left corner */

    AASPEED	aa_speed;	/*  Speed of animation, msec per frame */
    AASPEED	aa_designspeed;	/*  Designed milliseconds per frame */

    WORD	aa_width;	/*  Width of animation in pixels */
    WORD	aa_height;	/*  Height of animation in pixels */

    WORD	aa_frames;	/*  Number of frames in animation */

    DWORD	aa_position;	/*  Current frame position */
    DWORD	aa_loops;	/*  End of animation position */
    WORD	aa_rptsound;	/*  Number of times to repeat sound */
    WORD	aa_pause;	/*  Number of milliseconds to hold last frame */
    LONG	aa_delaysnd;	/*  Delay Sound in milliseconds  */
    BYTE	aa_transin;	/*  Transition at start of animation */
    BYTE	aa_transout;	/*  Transition at end of animation */
    WORD	aa_timein;	/*  Length of transition in, milliseconds */
    WORD	aa_timeout;	/*  Length of transition out, milliseconds */
    HWND	aa_callback;	/*  message callback window  */
    HWND	aa_animwnd;	/*  Animation Window Handle */
    WORD	aa_fsdelay;	/*  Full screen Mode delay in milliseconds */
} AAPARMS, FAR *LPAAPARMS;


/*
 *  aaGetParm will return information on an animation. Some of the
 *  information can be set with aaSetParm, and other information is
 *  state information maintined by AAPLAY.
 */
DWORD FAR AAEXPORT PASCAL aaGetParm(HAA hAa, WORD wType);

/*
 *  aaGetParmIndirect returns the same information as aaGetParm, in a
 *  structure for easy access by C applications.
 */
BOOL FAR AAEXPORT PASCAL aaGetParmIndirect(HAA hAa, LPAAPARMS lpAp, WORD wSize);


/*
 *  aaSetParm will set information on an animation
 */
BOOL FAR AAEXPORT PASCAL aaSetParm(HAA hAa, WORD wType, WORD wValue1, DWORD lValue2);

/*
 *  aaSetParmIndirect will set animation parameters from a structure
 */
BOOL FAR AAEXPORT PASCAL aaSetParmIndirect(HAA hAa, DWORD dwType, LPAAPARMS lpAp,
    WORD wMask);


/*
 *  The parameter types that can be used with aaGetParm, and aaSetParm.
 */
#define AA_STATUS		  1	/* Get current status */
#define AA_FILETYPE		  2	/* Get Type of animation on disk */
#define AA_MODE			  3	/* Get/Set Animation Flags */
#define AA_WINDOW		  4	/* Set/Get animation window */
#define AA_SPEED		  5	/* Set/Get current speed */
#define AA_DESIGNSPEED		  6	/* Get design speed */
#define AA_FRAMES		  7	/* Get Number of frames */
#define AA_POSITION		  8	/* Set/Get current frame position */
#define AA_LOOPS		  9	/* Set/Get number of loops */
#define AA_X			 10	/* Set/Get Pos of display window */
#define AA_Y			 11	/* Set/Get Pos of display window */
#define AA_CX			 12	/* Set/Get extents of display window */
#define AA_CY			 13	/* Set/Get extents of display window */
#define AA_ORGX			 14	/* Set/Get Origin of display window */
#define AA_ORGY			 15	/* Set/Get Origin of display window */
#define AA_WIDTH		 16	/* Get Width of animation */
#define AA_HEIGHT		 17	/* Get Height of animation */
#define AA_RPTSOUND		 18	/* Set/Get sound repeats */
#define AA_PAUSE		 19	/* Set/Get pause time */
#define AA_DELAYSND		 20	/* Set/Get sound delay time */
#define AA_TRANSIN		 21	/* Set/Get Transition In type */
#define AA_TRANSOUT		 22	/* Set/Get Transition Out type */
#define AA_TIMEIN		 23	/* Set/Get Transition In time */
#define AA_TIMEOUT		 24	/* Set/Get Transition Out Time */
#define AA_CALLBACK		 25	/* Set/Get CallBack window */
#define AA_ANIMWND		 26	/* Get Animation Window Handle */
#define AA_FILE			 27	/* Get file handle */
#define AA_FSDELAY		 28	/* Get/Set fullscreen switch delay */
#define AA_MODFLAG		100	/* Set/Get Script is modified flag */
#define AA_SCRIPTNAME		101	/* Set/Get Script name */
#define AA_ANIMATION		102	/* Get/Set Script Animation */
#define AA_ANIMATIONCOUNT	103	/* Get Script Animation Count */
#define AA_SCRIPTCONTENTS	104	/* Get Script Contents */
#define AA_LASTERROR	       1001	/* Get last error code */
#define AA_LASTERRORMESSAGE    1002	/* Get/Set last error messsage */
#define AA_FRAMEBITMAP	       2000	/* Get/Set frame bitmap */


/*
 *  The parameter type that can be used with aaSetParmIndirect
 */
#define AA_SETMODE	0x00000001UL	/* Get/Set Animation Flags */
#define AA_SETWINDOW	0x00000002UL	/* Set/Get animation window */
#define AA_SETSPEED	0x00000004UL	/* Set/Get current speed */
#define AA_SETPOSITION	0x00000008UL	/* Set/Get current frame position */
#define AA_SETLOOPS	0x00000010UL	/* Set/Get number of loops */
#define AA_SETX		0x00000020UL	/* Set/Get left of display window */
#define AA_SETY		0x00000040UL	/* Set/Get left of display window */
#define AA_SETCX	0x00000080UL	/* Set/Get top of display window */
#define AA_SETCY	0x00000100UL	/* Set/Get top of display window */
#define AA_SETORGX	0x00000200UL	/* Set/Get width of display window */
#define AA_SETORGY	0x00000400UL	/* Set/Get width of display window */
#define AA_SETRPTSOUND	0x00000800UL	/* Set/Get sound repeats */
#define AA_SETPAUSE	0x00001000UL	/* Set/Get pause time */
#define AA_SETDELAYSND	0x00002000UL 	/* Set/Get sound delay time */
#define AA_SETTRANSIN	0x00004000UL	/* Set/Get Transition In type */
#define AA_SETTRANSOUT	0x00008000UL	/* Set/Get Transition Out type */
#define AA_SETTIMEIN	0x00010000UL	/* Set/Get Transition In time */
#define AA_SETTIMEOUT	0x00020000UL	/* Set/Get Transition Out Time */
#define AA_SETCALLBACK	0x00040000UL	/* Set/Get Callback window */
#define AA_SETFSDELAY	0x00100000UL	/* Set/Get fullscreen mode switch delay */
#define AA_ALL		0xFFFFFFFFUL	/* Get/Set all parameters */


/*
 *  Status values for an animation
 */
#define AA_STOPPED		 1	/* Loaded but not playing */
#define AA_QUEUED		 2	/* Animation is waiting to play */
#define AA_PLAYING		 3	/* Animation is playing */
#define AA_PAUSED		 4	/* Animation is paused */
#define AA_DONE			 5	/* Animation has ended playing,
					   and is awaiting an aaStop call */


/*
 *  File type definitions
 */
#define AA_FLI			0x01	/* Autodesk Animator Fli format */
#define AA_DIB			0x02	/* Windows DIB format */
#define AA_NUMTYPES		0x02	/* Number of types */
#define AA_SCRIPT		0x03	/* Script without an animation */

/*
 *  Transition types
 */
#define AA_CUT			0x00	/* Just stop one and start another */
#define AA_FADEBLACK		0x01	/* Fade In/Out from to black */
#define AA_FADEWHITE		0x02	/* Fade In/Out from to white */


/*
 *  Error codes returned by aaGetParm(xxx, AA_LASTERROR)
 */
#define AA_ERR_NOERROR			0x0000	/*  Unknown error */

#define AA_ERR_NOMEMORY			0x0100	/*  Out of memory error */
#define AA_ERR_BADHANDLE		0x0101	/*  Bad handle */
#define AA_ERR_NOTIMERS			0x0102	/*  Cannot start timer */
#define AA_ERR_BADSOUND			0x0103	/*  Bad sound spec */
#define AA_ERR_NOSCRIPT			0x0104	/*  Requires a script */
#define AA_ERR_WRITEERR			0x0105	/*  Write error for script */
#define AA_ERR_BADANIMATION		0x0106  /*  Can not opem animation */

#define AA_ERR_BADWINDOWHANDLE		0x0200	/*  Bad Window Handle */
#define AA_ERR_WINDOWCREATE		0x0201	/*  Cannot create window */
#define AA_ERR_DLGERROR			0x0202	/*  Dialog error */

#define AA_ERR_INVALIDSTATUS		0x0300	/*  Invalid status */
#define AA_ERR_BADDIBFORMAT		0x0301	/*  Bad dib file */
#define AA_ERR_BADFLIFORMAT		0x0302	/*  Bad fli file */
#define AA_ERR_UNRECOGNIZEDFORMAT	0x0303	/*  Unrecognized format */
#define AA_ERR_NOSOUND			0x0304	/*  Sound not supported */
#define AA_ERR_NOTVALIDFORSCRIPTS	0x0305	/*  Not valid for scripts */
#define AA_ERR_INVALIDFILE		0x0306	/*  Bad file handle */
#define AA_ERR_NOSCRIPTS		0x0307	/*  No Script files */

#define AA_ERR_SPEED			0x0400	/*  Invalid speed */
#define AA_ERR_LOOPS			0x0401	/*  invalid loops */
#define AA_ERR_RPTSOUND			0x0402	/*  invalid repeat sound */
#define AA_ERR_PAUSE			0x0403	/*  invalid pause */
#define AA_ERR_TRANSIN			0x0404	/*  invalid transition */
#define AA_ERR_TIMEIN			0x0405	/*  invalid transition */
#define AA_ERR_TRANSOUT			0x0406	/*  invalid transition time */
#define AA_ERR_TIMEOUT			0x0407	/*  invalid transition time */
#define AA_ERR_DELAYSND			0x0408	/*  invalid sound delay */
#define AA_ERR_INVALIDTYPE		0x0409	/*  invalid parameter type */
#define AA_ERR_FSDELAY			0x040a	/*  invalid full screen delay */

#define AA_ERR_DUPLICATENOTIFY		0x0500	/*  duplicate notify  */

#define AA_ERR_NOSWITCH			0x0600	/*  no switch in script */
#define AA_ERR_PARSELOOPS		0x0601	/*  Bad loops in script */
#define AA_ERR_PARSESPEED		0x0602	/*  Bad speed in script */
#define AA_ERR_BADRPTSOUND		0x0603	/*  Bad repeat sound in script */
#define AA_ERR_PARSEPAUSE		0x0604	/*  Bad pause in script */
#define AA_ERR_PARSETRANS		0x0605	/*  Bad tranisition in script */
#define AA_ERR_PARSEDELAYSND		0x0606	/*  Bad delay sound in script */
#define AA_ERR_TOOMANYLINKS		0x0607  /*  Too many links */


/*
 *  aaShow will show a single frame of an animation in a window.
 *  The mode determines how the animation is drawn. The window used
 *  is set using either aaSetParm or aaSetParmIndirect. aaShow returns
 *  TRUE if the animation was successfully drawn.
 */
BOOL FAR AAEXPORT PASCAL aaShow(HAA hAa, BOOL bShow);


/* result = OpenFileDialog(wFlags, lpszPath, wBufLen, lpszDriver, wDrvLen)
 *
 * Opens a dialog box querying the user for a file in the standard windows
 * file open box style.  <dwFlags> determines how the dialog box will appear.
 * It may be the following:
 *	AA_GETFILE_MUSTEXIST	Selected file must satisfy conditions of
 *				OpenFile() flags, else dialog beeps.
 *	AA_GETFILE_NOSHOWSPEC	DO NOT show the search spec in the edit box.
 *				Default IS to show the spec.
 *	AA_GETFILE_SAVE		Ok button will show "Save".
 *	AA_GETFILE_OPEN		Ok button will show "Open".
 *	AA_GETFILE_USEFILE	Set the filename to the file in lpszPath
 *	AA_GETFILE_UDEDIR	Change to the directory in lpszPath
 *	AA_GETFILE_SOUND	Get sound file and driver
 *	AA_GETFILE_SCRIPT	Get script file
 *	AA_GETFILE_ANIMATION	Get Animation File (no scripts)
 *
 * <lpszPath> is a LPSTR to a string buffer into which the final fully
 * qualified pathname will be written.  <wBufLen> is the length of this
 * buffer.
 *
 * <lpszDriver> is a LPSTR to a string buffer into which a sound device
 * selection is placed. <wDrvLen> is the length of this buffer.
 *
 * Return value is:	0 if cancel was hit
 *			-1 if OpenFile() failed but AA_GETFILE_MUSTEXIST was
 *			not specified.
 *			DOS file handle of selected file, otherwise. This
 *			    handle is not open when aaOpenFile returns.
 */

int FAR AAEXPORT PASCAL aaGetFile(WORD dwFlags,          /* Flag values */
                         LPSTR lpszPath,
                         WORD  wBufLen,
			 LPSTR lpszDriver,	      /* Selected item */
			 WORD  wDrvLen);	      /* Length of buffer */

/*
 *  flags:
 *     can be any of the following
 */
#define AA_GETFILE_MUSTEXIST	0x0001
#define AA_GETFILE_NOSHOWSPEC	0x0002
#define AA_GETFILE_SAVE		0x0004
#define AA_GETFILE_OPEN		0x0008
#define AA_GETFILE_USEDIR	0x0010
#define AA_GETFILE_USEFILE	0x0020
#define AA_GETFILE_SOUND	0x0040
#define AA_GETFILE_SCRIPT	0x0080
#define AA_GETFILE_ANIMATION	0x0100

/*
 *  Save a script
 */
int  FAR AAEXPORT PASCAL aaSave(HAA hAa, WORD wMode);

/*
 *  wMode Values
 */
#define AA_SAVE_IFMODIFIED	0x0001
#define AA_SAVE_AS		0x0002
#define AA_SAVE_CANCEL		0x0004

/*
 *  Get Capabitities
 */
WORD FAR AAEXPORT PASCAL aaGetCaps(WORD wType);

/*
 * Capabitlities
 */
#define AA_CAP_TIMER		 1
#define AA_CAP_SOUND		 2
#define AA_CAP_SCRIPT		 3
#define AA_CAP_VERSION		 4
#define AA_CAP_BITMAP		 5
#define AA_CAP_EMBEDDING	 6

/*
 *  Animation status messages
 *  Use RegisterWindowMessage to get the real message
 *  numbers.
 */
#define AA_NOTIFY		"AAPLAY Notify"	/* notification message */
#define AA_STOP			"AAPLAY Stop"	/* stop message */

/*
 *  These are sent in the low word of lParam with the AA_ERROR
 *  message. They indicate the error that occured
 */
#define AA_BADPLAY		1	/* Error attempting to play */
#define AA_BADNOTIFY		2	/* Error attempting to notify */
#define AA_BADSCRIPT		3	/* Error attempting to animation
					   in script */


#endif /* AAPLAY_INCLUDE */
