/*
 * $Header:   K:/21vcs/srccmd/samples.win/techos.c_v   1.3   06 May 1992 16:03:20   arnoff  $
 */

/*
 * Copyright (C) 1991-1992 by FTP Software, Inc.
 * 
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 * 
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 * 
 * EDIT HISTORY:
 * 20-Sep-90  msd@ayuda	Original author.  Adapted from 'tdll.c'.
 * 04-Oct-90  msd@ayuda	First pre-release.
 * 01-Apr-91  msd@ayuda	To FTP for 2.05 pl 2 beta.
 * 06-May-92  ftp	DevKit 2.1 beta.
 */

/* "techos.c" -- Service application for PC/TCP under Windows 3.x.
   Implements a TCP ECHO server. */

/* Disclaimer: FTP Software provides these sample Windows applications as 
   simple examples of how the PC/TCP PCTCPAPI.DLL can be used. We cannot 
   ensure that all coding conventions used herein are appropriate for all 
   Windows applications. Please consult an official Windows programming
   reference (i.e. Microsoft SDK) for recommended coding conventions which
   best suit your needs.
*/
   
#include <stdio.h>
#include <dos.h>
#include <memory.h>

/* The following #defines prune the symbol table! */
#define	NORASTEROPS
#define NOSOUND
#define NOCOMM
#define NOKANJI

#include <windows.h>
#include <pctcp/winapp.h>
#include <pctcp/asynch.h>
#include "fmters.h"
#include "plaints.h"
#include "techos.h"

/* TCP connection states */
#define CLOSED		0	/* cradle & grave state */
#define LISTEN		1	/* listen but not connected */
#define ESTAB		2	/* established for normal i/o */
#define CLOSING		3	/* remote EOF came -- shutting down */

/* Verify mode types. */
#define	V_NONE		0	/* not verifying input (default) */
#define V_BP		1	/* verifying 'tstpat' "barberpole" */

#define nextBP(c)	((c) >= '~' ? ' ' : (c) + 1)

#ifndef	OURPORT
#define	OURPORT		7	/* std ECHO service port # */
#endif


/* local function prototypes */

void AbortNet(int, char *);
void HandleAsync(WORD, DWORD);
int InitListen(HWND);
int InitNet(HWND);
void PaintLine(char *, HDC, TEXTMETRIC *, int *, int *);
void PaintWnd(HWND);
int ParseCmdLine(LPSTR);
void Process1Segment();
void Repaint();


/* global / local variables */

HANDLE hInst;			/* current instance */
HWND hOurWnd = NULL;		/* handle for our root window */

unsigned long byteCounter;	/* # of bytes processed */
int error;			/* errno of last i/o error */
unsigned char expectNext;	/* next expected for pattern verify mode */
int nd = -1;			/* datagram net descriptor */
unsigned netVsn;		/* PC/TCP version # */
struct addr msgAddr;		/* addresses */
char msgBuf[2048];		/* buffer for a segment */
unsigned long phaseError;	/* state machine phase errors */
char prbuf[256];		/* message assembly buffer */
int readNow;			/* Boolean that tells main loop to read */
unsigned long sndNomemError;	/* # of NOMEM errors on write requests */
unsigned long sndSpinCounter;	/* # of Yield spins waiting for write space */
int state;			/* TCP connection state */
unsigned tcpPort = OURPORT;	/* service port number */
unsigned long verifyError;	/* # of verify mode errors */
int verifyMode;			/* in verify mode; which type */

char *state_name[] = {
	"CLOSED", "LISTENING", "ESTABLISHED", "CLOSING"
};

char *verify_name[] = {
	"NONE", "BARBERPOLE"
};

/* function bodies */

int PASCAL
WinMain (hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;
HANDLE hPrevInstance;
LPSTR lpCmdLine;
int nCmdShow;
{
	MSG msg;

	if (!hPrevInstance)
		if (!InitApplication(hInstance))
			return FALSE;

	if (!InitInstance(hInstance, lpCmdLine, nCmdShow))
		return FALSE;

	/******************************************************************** 
	   Modified main MessageLoop:  Because there is no relationship 
	   between the arrival of NET_AS_RCV notifications and what 'net_read'
	   will return, we can ofttimes get several asynchronous notifications
	   which are dealt with by a single 'net_read'. (Note that the 
	   opposite sometimes happens as well). Therefore, the asynchronous 
	   arrival simply sets a flag telling this logic to read now (this 
           flag is cleared when 'net_read' sees NET_ERR_WOULD_BLOCK). This
	   also works well with the need to avoid processing network I/O in 
           the WndProc itself, because doing that can cause nesting and 
	   ordering problems.  Doing the I/O here serializes things nicely.

	   The Modified Main MessageLoop allows polling for inbound
	   data segments while yielding to other applications.  This
	   is necessary because NET_AS_RCV messages are SEND, not
	   POST to the WinProc (SendMessage() vs. PostMessage()). 
	*********************************************************************/
	
	do {
		if (readNow)
			Process1Segment();
		Yield();
		if (!PeekMessage(&msg, NULL, NULL, NULL, PM_REMOVE|PM_NOYIELD))
			continue;
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	} while (msg.message != WM_QUIT);
	return msg.wParam;
}

BOOL
InitApplication (hInstance)
HANDLE hInstance;
{
	WNDCLASS wc;

	wc.style = NULL;
	wc.lpfnWndProc = MainWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = GetStockObject(WHITE_BRUSH);
	wc.lpszMenuName = "TECHOSMenu";
	wc.lpszClassName = "TECHOSWClass";

	return RegisterClass(&wc);

}

BOOL
InitInstance (hInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;
LPSTR lpCmdLine;
int nCmdShow;
{
	HWND hWnd;		/* main window */
	char prbuf[128];

	hInst = hInstance;

	hWnd = CreateWindow("TECHOSWClass", "TCP ECHO Server"
	  , WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT
	  , CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, hInstance, NULL);

	if (!hWnd)
		return FALSE;

	hOurWnd = hWnd;			/* export for global use */
	if (!ParseCmdLine(lpCmdLine)	/* parse command line */
	  ||  InitNet(hWnd) < 0) {	/* initialize network--must be last */
		PostQuitMessage(0);
		return FALSE;
	}

	ShowWindow(hWnd, nCmdShow);
	UpdateWindow(hWnd);	/* sends WM_PAINT message */
	return TRUE;
}

long FAR PASCAL
MainWndProc (hWnd, message, wParam, lParam)
HWND hWnd;
unsigned message;
WORD wParam;
LONG lParam;
{
	FARPROC lpProcAbout;

	switch (message) {
        case WM_COMMAND:    /* message: command from application menu */
		if (wParam == IDM_ABOUT) {
			lpProcAbout = MakeProcInstance(About, hInst);
			DialogBox(hInst, "TECHOSAboutBox", hWnd, lpProcAbout);
			FreeProcInstance(lpProcAbout);
			break;
		} else
			return DefWindowProc(hWnd, message, wParam, lParam);
	case WM_DESTROY:
		/* Terminate access to PCTCPAPI.DLL. */
		net_taskDestroy();
		nd = -1;
		PostQuitMessage(0);
		break;
	case WM_PAINT:
		PaintWnd(hWnd);
		break;
	case WM_TIMER:
		/* Force main loop to try reading.  This forestalls hanging
		   indefinitely on a lost asynch (highly unlikely). */
		if (state == ESTAB)
			readNow = 1;
		/* FALLTHRU */
	/* specials in this app */
	case TECHOSM_POLL:
		/* Forced full repaint. */
		InvalidateRect(hWnd, NULL, TRUE);
		break;
	default:
		/* See if this is an asynch notice from PC/TCP. */
		if (message == net_msgType) {
			if (state == CLOSED)
				readNow = 0;	/* ignore it */
			else if (net_msgEvent(wParam) == NET_AS_RCV
			  ||  net_msgEvent(wParam) == NET_AS_FCLOSE)
				readNow = 1;	/* tell main loop to read */
			else
				HandleAsync(wParam, lParam);
			break;
		}
		return DefWindowProc(hWnd, message, wParam, lParam);
	}
	return NULL;
}

BOOL FAR PASCAL
About (hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WORD wParam;
LONG lParam;
{
	switch (message) {
	case WM_INITDIALOG:
		return TRUE;
	case WM_COMMAND:	             /* message: received a command */
		if (wParam == IDOK	      /* "OK" box selected? */
		   ||  wParam == IDCANCEL) {  /* System menu close command? */
			Repaint();
			EndDialog(hDlg, TRUE);
			return TRUE;
		}
		break;
	}
	return FALSE;		/* Didn't process a message */
}

/* Abort the network connection with an error. */
void
AbortNet (nd, plaint)
int nd;
char *plaint;
{
	int n;

	if (plaint)
		perrorTypePlaint(plaint, 1);
	state = CLOSED;
	error = net_errno;
	net_abort((n = nd, nd = -1, n));
	Repaint();
}

/* Handle a non-arrival-type asynch message notification. */
void
HandleAsync (wParam, lParam)
WORD wParam;
DWORD lParam;
{
	int n;

	switch (net_msgEvent(wParam)) {
	case NET_AS_ERROR:
		/* deal summarily with errors */
		net_errno = (int) lParam;
		AbortNet(nd, "net_asynchw ERROR arrival");
		break;
	case NET_AS_CLOSE:
		if (state == CLOSING) {
			state = CLOSED;
			net_release((n = nd, nd = -1, n));
			Repaint();
		} else
			++phaseError;
		break;
	case NET_AS_OPEN:
		if (state == LISTEN) {
			state = ESTAB;
			Repaint();
		} else
			++phaseError;
		break;
	default:
		++phaseError;
		break;
	}
}

/* Create a network descriptor and start listening on it. */
int
InitListen (hWnd)
HWND hWnd;
{
	char *errP;
	int n;
	
	/* Get a socket. */
	if ((nd = net_getdesc()) == -1)
		return perrorTypePlaint("net_getdesc", 0);
	/* Register for notification of connection establishment. */
	if (net_asynchw(nd, NET_AS_OPEN, hWnd, NET_ASWM_POST) == -1L) {
		errP = "net_asynchw OPEN";
		goto abortPlaintNQuit;
	}
	/* Register for notification of connection error. */
	if (net_asynchw(nd, NET_AS_ERROR, hWnd, NET_ASWM_POST) == -1L) {
		errP = "net_asynchw ERROR";
		goto abortPlaintNQuit;
	}
	/* Register for notification of EOF arrival. */
	if (net_asynchw(nd, NET_AS_FCLOSE, hWnd, NET_ASWM_POST) == -1L) {
		errP = "net_asynchw FCLOSE";
		goto abortPlaintNQuit;
	}
	/* Register for notification of connection end. */
	if (net_asynchw(nd, NET_AS_CLOSE, hWnd, NET_ASWM_POST) == -1L) {
		errP = "net_asynchw CLOSE";
		goto abortPlaintNQuit;
	}
	/* Register for notification of data arrival.  Note that for TCP,
	   we SEND the AS_RCV messages (via SendMessage()).  This avoids 
	   POSTed messages clogging the message queue, rendering it deaf to
	   operator commands. */
	if (net_asynchw(nd, NET_AS_RCV, hWnd, NET_ASWM_SEND) == -1L) {
		errP = "net_asynchw RCV";
		goto abortPlaintNQuit;
	}
	/* Listen on the STREAM echo socket. */
	msgAddr.lsocket = tcpPort;
	msgAddr.protocol = STREAM;
	if (net_listen(nd, STREAM, &msgAddr) == -1) {
		errP = "net_listen";
		goto abortPlaintNQuit;
	}
	state = LISTEN;
	return nd;

abortPlaintNQuit:;
	net_abort((n = nd, nd = -1, n));
	return perrorTypePlaint(errP, 0);
}

int
InitNet (hWnd)
HWND hWnd;
{
	/* Initialize access to PCTCPAPI.DLL. */
	if (net_taskInit("techos") == -1) {
		perrorTypePlaint("net_taskInit", 0);
		goto deregNQuit;
	}
	/* Get the PC/TCP version. */
	netVsn = (unsigned) get_netversion();
	/* Listen for a connection. */
	if (InitListen(hWnd) == -1)
		goto deregNQuit;

	/* Register for a Windows timer on 30 second intervals. */
	if (SetTimer(hWnd, 0, 30000, (FARPROC) 0) == NULL) {
		otherPlaint("SetTimer failed", 0);
		goto deregNQuit;
	}
	return 0;

	/* Deregister access to PCTCPAPI.DLL and quit. */
deregNQuit:;
	net_taskDestroy();
	return -1;
}

int
ParseCmdLine (lpCmdLine)
LPSTR lpCmdLine;
{
	register char c;
	char *errP;
	LPSTR fcP;
	auto LPSTR afcP;

	/* Parse the command line.  Look for "/P<portNo>" and/or "/C"
	   options. */
	for (fcP = lpCmdLine; c = *fcP; ++fcP) {
		switch (c) {
		case '/':	/* option delimiter -- MS-DOS style */
			switch (*++fcP) {
			case 'p':  case 'P':	/* TCP service port number */
				afcP = ++fcP;
				tcpPort = (unsigned) wstrtol(fcP, &afcP, 10);
				if (afcP == fcP)
					goto usage_err_out;
				fcP = afcP - 1;	/* at end of number parsed */
				break;
			case 'c':  case 'C':	/* 'check' (verify) mode */
				verifyMode = V_BP;
				expectNext = ' ';
				break;
			default:
				goto usage_err_out;
			}
			break;
		}
	}
	return TRUE;

usage_err_out:;
	errP = "Invalid command line: options are /P<portNo> /C";
	otherPlaint(errP, 0);
	return FALSE;
}

/* Process a 'segment' of data (actually, whatever 'net_read' gives us in a
   single request.  Then, push it back at the client.  Note, the sending
   logic isn't particularly clever; it *could* leave unsent data in a buffer
   somewhere and then await NET_AS_SND. */
void
Process1Segment () {
	register char c;
	register int i;
	 int n, n_left;

	if (state != ESTAB  ||  nd == -1)
		return;
	if ((n = net_read(nd, msgBuf, sizeof msgBuf, &msgAddr, 0)) == -1) {
		if (neterrno == NET_ERR_EOF)
			n = 0;
		else {
			if (neterrno != NET_ERR_WOULD_BLOCK)
				AbortNet(nd, "net_read");
			readNow = 0;
			return;
		}
	}
	if (n == 0) {
		state = CLOSING;
		if (net_eof(nd) == -1)
			AbortNet(nd, "net_eof");
		return;
	}
	switch (verifyMode) {
	case V_BP:
		for (i = 0; i < n; ++i) {
			if ((c = msgBuf[i]) != expectNext)
				++verifyError;
			expectNext = nextBP(c);
		}
		break;
	}
	for (i = 0, n_left = n; n_left > 0; i += n, n_left -= n) {
		if ((n = net_write(nd, &msgBuf[i], n_left, 0)) > 0) {
			byteCounter += n;
			continue;
		}
		if (n < 0) {
			if (neterrno == NET_ERR_NOMEM)
				/* Just count NOMEM errors. */
				++sndNomemError;
			else if (neterrno != NET_ERR_WOULD_BLOCK) {
				AbortNet(nd, "net_write");
				return;
			}
			n = 0;
		}
		++sndSpinCounter;
		Yield();
	}
}

void
PaintLine (txtP, hDC, tmP, nDrawXP, nDrawYP)
char *txtP;
HDC hDC;
TEXTMETRIC *tmP;
int *nDrawXP, *nDrawYP;
{
	unsigned len = strlen(txtP);

	/* Send characters to the screen.  After displaying each line of
	   text, advance the vertical position for the next line of text.
	   The pixel distance between the top of each line of text is equal
	   to the standard height of the font characters (tmHeight), plus
	   the standard amount of spacing (tmExternalLeading) between
	   adjacent lines. 
	*/
	TextOut(hDC, *nDrawXP, *nDrawYP, txtP, len);
	*nDrawYP += tmP->tmExternalLeading + tmP->tmHeight;
}

void
PaintWnd (hWnd)
HWND hWnd;
{
	HDC hDC;
	PAINTSTRUCT ps;
	TEXTMETRIC textmetric;
	int nDrawX, nDrawY;
	struct addr peerAddr;

	hDC = BeginPaint(hWnd, &ps);
	/* Get the size characteristics of the current font.
	   This information will be used for determining the
	   vertical spacing of text on the screen. */
        GetTextMetrics(hDC, &textmetric);
	/* Initialize drawing position to 1/4 inch from the
	   top and from the left of the top, left corner of
	   the client area of the main windows. */
	nDrawX = GetDeviceCaps(hDC, LOGPIXELSX) / 4;
	nDrawY = GetDeviceCaps(hDC, LOGPIXELSY) / 4;

	/* paint out basic app stuff */
	sprintf(prbuf, "PC/TCP Version %u.%02u port %d"
	  , netVsn / 256, netVsn % 256, tcpPort);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	/* basic stats */
	sprintf(prbuf
	   , "#rcv PhaseErrors %lu -- #snd yields %lu -- #snd nomemErrors %lu"
	   , phaseError, sndSpinCounter, sndNomemError);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	/* connection stats */
	if (state == CLOSED  ||  state == LISTEN)
		sprintf(prbuf, "%s -- #bytes %lu", state_name[state]
		  , byteCounter);
	else if (get_peer(nd, &peerAddr) == -1)
		sprintf(prbuf, "%s *UNKNOWN PEER* -- #bytes %lu"
		  , state_name[state], byteCounter);
	else
		sprintf(prbuf, "%s to %s,%u -- #bytes %lu", state_name[state]
		  , pr_in_name(peerAddr.fhost), peerAddr.fsocket, byteCounter);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	if (error) {
		sprintf(prbuf, "** ERROR CODE %d **", error);
		PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);
	}

	if (verifyMode != V_NONE) {
		sprintf(prbuf, "Verifying %s -- #errors %lu"
		  , verify_name[verifyMode], verifyError);
		PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);
	}

	EndPaint(hWnd, &ps);
}

void
Repaint () {
	/* Tell our WinProc to repaint. */
	PostMessage(hOurWnd, TECHOSM_POLL, 0, 0L);
}

/* eof */

/*
 * $Log:   K:/21vcs/srccmd/samples.win/techos.c_v  $
 * 
 *    Rev 1.3   06 May 1992 16:03:20   arnoff
 *  * 06-May-92  ftp	DevKit 2.1 beta.
 * 
 *    Rev 1.2   03 Feb 1992 22:01:26   arnoff
 * pre beta-2 testing freeze
 * 
 *    Rev 1.1   29 Jan 1992 22:47:20   arnoff
 *  
 */
