/*
 * $Header:   J:/22vcs/srclib/socket/accept.c_v   1.3   02 Oct 1992 18:22:34   rcq  $
 */

/*
 * ACCEPT.C - Emulate 4bsd accept() call.
 *
 * Copyright (C) 1987-1992 by FTP Software, Inc.  All rights reserved.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 * Edit History
 * 14-Oct-87	jbvb	malloc() isn't guaranteed to return zeroed data.
 * 18-Jan-88	jbvb	Don't panic when we run out of descriptors, instead
 *			 just don't finish refilling the queue to "backlog".
 *			Do return EMFILE if the pending queue is empty.  These
 *			 should fix HP #938, and maybe 647, too.
 *			Propagate use of NET_FLG_NONBLOCKING to this routine
 *			 (Philip had confused this with NET_OPT_NONBLOCKING).
 *			 This ought to fix HP #647, finally.
 * 21-Jan-88	jbvb	Copy l_intrvl when swapping descriptors.
 * 23-Jan-88	jbvb	Make accept() robust when out of network descriptors.
 *			 Call bsd_fill_con_q to fill/refill the queue, call
 *			 bsd_relisten() to restart net_listen, etc.
 * 04-Feb-88	jbvb	Fix arguments to calloc().
 * 12-Feb-88	jbvb	Be sure to do a relisten if backlog is 1.
 * 18-Feb-88	jbvb	Catch all cases where a re-listen is needed (whenever
 *			 all queued descriptors were used, and more become
 *			 available).  HP #940.
 * 06-Jun-88	jbvb	Disable loop optimization, so MSC 5.1 won't cache
 *			 volatile values in registers...
 * 03-Aug-89	jbvb	Handle case where not listening, queue re-fills but
 *			 bsd_relisten() fails; return error i/o hanging.
 * 23 AUG 89	stev	fixed compiler warning about types mismatch on 
 *			 routines using sockaddr_in here but sockaddr 
 *			 in the header file.
 * 29aug89	STEV	added EBADF to accept() for invalid descriptor
 * 07-Nov-91	paul	changed to new-style function declarators,
 *			added function return types,
 *			changed forever loops from while(1) to for(;;)
 * 14-Aug-92    rcq     updated the copyright in comments
 * 09-Sep-92	rcq	replaced call to panic() with return(-1)
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#include <pctcp/error.h>
#include <pctcp/types.h>
#include <pctcp/pctcp.h>
#include <pctcp/options.h>

#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <debug.h>

#include "4bsd.h"

extern int bsd_fill_con_q(SOCKET *sp);
extern int bsd_relisten(SOCKET *sp);
/* extern void panic(char *s, int a); */

#pragma loop_opt(off)		/* MSC 5.1 would never find a connection... */

int
accept(int s, struct sockaddr *addr_passed, int *addrlen)
{
	struct sockaddr_in	*addr;
	SOCKET	*sp = NSOCK(s), *nsp;
	NCONN	*nc;
	int	wason, i;

	addr = (struct sockaddr_in *) addr_passed;
#ifdef	DEBUG
	printf("accept(s = %d, addr = x%Np, *addrlen = %d)",
		s, addr, *addrlen);
#endif

	if(sp == NULL)
		bomb(EBADF);		/* invalid descriptor		*/

	if(is_netnd(sp->nd))		/* not a socket			*/
		bomb(ENOTSOCK);

	if(! (STATE(sp) & SI_LISTENING))
		bomb(EINVAL);		/* not listening on the socket	*/

	if(addr != NULL && *addrlen < sizeof(*addr))
		bomb(EFAULT);		/* wrong size address		*/

	if(sp->type != STREAM)
		bomb(EOPNOTSUPP);	/* not stream (TCP) connection	*/

	if (sp->nconn_q == NULL) {	/* No queued descriptors... 	*/
		bsd_fill_con_q(sp);	/* Try to refill queue 		*/
#ifdef DEBUG
		printf("Q emptied: refill %s\n", sp->nconn_q ? "ok": "bad");
#endif
		if (sp->nconn_q) {	/* If we found at least one 	*/
		    if (i = bsd_relisten(sp)) /*  try listening again. 	*/
			bomb(i);	/* Report whatever failed 	*/
		} else {		/* My cupboard is bare, 	*/
		    bomb(EMFILE);	/*  throw them a bone... 	*/
		}
	}

	if(sp->cur_q_len == 0) {		/* No queued TCPs 	*/
		if(SOPTIONS(sp) & NET_FLG_NONBLOCKING)	
			bomb(EWOULDBLOCK);	/* set for nonblocking	*/

		if ((STATE(sp) & SI_LISTEN_ERR) &&	/* If not listening */
		    (i = bsd_relisten(sp)))		/*  & can't start, */
			bomb(i);			/*  don't wedge. */

		while(sp->cur_q_len == 0)	/* We're a PC, so we can */
			;			/*  just spin for a while. */
	}

	/* Extract the pending connection
	*/
	wason = disable_asynch();		/* Don't interrupt us 	*/

	/* search connection queue for unconnected descriptor
	 * (marked by being negative).
	 */
	for(nc = sp->nconn_q; nc && nc->nd >= 0; nc = nc->next)
		;
	i = (int) nc;			/* Save old queue state */

	nc = sp->nconn_q;
	sp->nconn_q = nc->next;
	--(sp->cur_q_len);

	bsd_fill_con_q(sp);			/* Try to refill queue */
	
	if(wason)
		enable_asynch();

	if((nsp = (SOCKET *) calloc(1, sizeof(*nsp))) == NULL) {
		net_abort(nc->nd);	/* blow the connection away 	*/
		free(nc);		/* Free the memory 		*/
		bomb(ENOBUFS);		/* Tell caller we're out of memory */
	}

	nsp->state = (SI_CONNECTED|SI_BOUND|SI_OPEN|SI_CAN_SEND|SI_CAN_RECV);
	nsp->flags = sp->flags;
	nsp->options = sp->options;
	nsp->af = sp->af;
	nsp->type = sp->type;
	nsp->nd = nc->nd;
	nsp->lhost = nc->lhost;
	nsp->fhost = nc->fhost;
	nsp->fsocket = nc->fsocket;
	nsp->lsocket = nc->lsocket;
	nsp->protocol = nc->protocol;
	nsp->l_intrvl = nc->l_intrvl;
	free(nc);
/*	nsp->max_q_len = nsp->cur_q_len = 0;	*/
/*	nsp->nconn_q = NULL;			*/

/*	printf("nd = %d\n", nsp->nd);	*/

	/* we know that there is a `hole' in the descriptor space
	 * where the new listening descriptor is, so we plug the
	 * new connected descriptor into this slot.
	 */
	if(nsp->nd < 0 || NSOCK(nsp->nd) != NULL)
	/* panic("accept: internal error on descriptor %d!\n", nsp->nd); */
	    return(-1);

	NSOCK(nsp->nd) = nsp;	/* install it in the table */

	if(addr != NULL) {
		addr->sin_family = nsp->af;
		addr->sin_port = htons(nsp->fsocket);
		addr->sin_addr.s_addr = nsp->fhost;
		*addrlen = sizeof(*addr);

#ifdef	DEBUG
		printf("accept: ");
		dump_sockaddr(addr);
#endif
	}

	/* search connection queue for unconnected descriptor
	 * (marked by being negative).
	 */
	wason = disable_asynch();		/* Don't interrupt us */
	for(nc = sp->nconn_q; nc && nc->nd >= 0; nc = nc->next)
		;

	/* May get error if # of nd's exceeds # of TCPs, but shouldn't
	 * scrap this connection just because of that.  Set a bit so we
	 * can tell this happened later.
	 */
	if ((i == 0) &&				/* If not listening before */
	    (nc != NULL) &&			/*  & have an nd, we can */
	    (bsd_relisten(sp)))			/*  try listening again. */
		sp->state |= SI_LISTEN_ERR;	/* Indicate it failed... */

	if(wason)
		enable_asynch();

	dreturn(" = %d\n", nsp->nd);
}

/*
 * $Log:   J:/22vcs/srclib/socket/accept.c_v  $
 * 
 *    Rev 1.3   02 Oct 1992 18:22:34   rcq
 * merged updates done in 2.1
 * 
 *    Rev 1.2   27 Aug 1992 15:41:00   arnoff
 *  * 14-Aug-92    rcq     updated the copyright in comments
 * 
 *    Rev 1.1   30 Jan 1992 00:50:44   arnoff
 *  
 */
