/*
 * $Header:   H:/devkit.dos/vcs/srcsmpl/dns_smpl/src/dns_smpl.c_v   1.0   20 Apr 1993 15:08:44   paul  $
 */

/************************************************************************
 * Copyright (C) 1986-1993 by FTP Software, Inc.  All rights reserved.
 * 
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 * 
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 ************************************************************************
 *
 * Name:  dns_smpl.c
 *
 ************************************************************************
 * History:
 *      22-Feb-93       olga    module creation
 *      08-Apr093       Arnoff  cleaning.
 *
 ************************************************************************
 * This sample program was created to demonstrate the proper usage of the
 * function calls in the Domain library.
 *
 * The program takes as input a domain name of a host on the
 * network.  It uses the Domain library to discover the following
 * information about the host:
 *
 *      IP Address(s)                           A query
 *      Name Servers                            NS query
 *      Zone Information                        SOA query
 *      Hardware Information                    HINFO query
 *      Canonical Name                          CNAME query
 *      Mail Exchange Information               MX query
 *
 * The Domain Name System (DNS) works on the client / server model.  Servers
 * are called Name Servers.  Clients ask them for information about machines
 * on the network.  This information includes machine names, IP addresses,
 * etc.  The client will send the Name Server a DNS query with one question.
 * The Name Server will reply with one or more answers and, often, additional
 * pertinant information.  A DNS query consists of a DNS header, and the
 * specific query type.  A DNS reply has the same type of DNS header, with a
 * number of Resource Records (RR) following it.  Each RR contains a bit of
 * information pertinant to the query (see the array "types" below).  Some
 * RRs are a direct answer to the query, others contain related information.
 *
 ************************************************************************
 */
 
#include <stdio.h>                      /* for input output                 */
#include <stdlib.h>                     /* for ltoa function                */
#include <string.h>                     /* for string function              */
#include <process.h>                    /* for exit()                       */
#include <pctcp/types.h>                /* Typedef in_name                  */
#include <ctype.h>                      /* for testing characters funct.    */
#include <sys/types.h>                  /* data types for C                 */
#include <fcntl.h>                      /* to flush the keyboard buffer     */

#include <pctcp/rwconf.h>               /* for configuration library stuff  */
#include <pctcp/ipconfig.h>             /* struct def needed for config.h   */
#include <pctcp/config.h>               /* for TAGS                         */
#include <pctcp/pctcp.h>                /* Function prototipes              */
#include <pctcp/error.h>                /* Error code definitions           */
#include <pctcp/dns_lib.h>              /* For library queries              */

/********************< definitions >*********************************/
#define NOERROR         (0)             /* dns call successful return       */

#define SIZE_DOMAIN_Q   (768+36)        /* size of structure dq to allocate */
#define SIZE_LINE       (256)           /* storage of results in struct info*/
#define SIZE_DNAME      (100)           /* storage of domain name & buffers */

#define SERV_ARRAY_LEN  (4)             /* Number of name servers           */

                        /* Types query flags:                           */
#define FL_IPADDR (1)                   /* IP-address                   */
#define FL_HINFO  (2)                   /* Hardware information         */
#define FL_CNAME  (4)                   /* Canonical name               */
#define FL_SOA    (8)                   /* Time Zone                    */
#define FL_MX    (16)                   /* Mail exchange                */
#define FL_NS    (32)                   /* Name server                  */

#define NUM_REQ   (6)                   /* Number of requests           */
                                        /* Mode flags:                  */
#define FL_NOMOD  (0)                   /* No mode specified            */
#define FL_SIMPL  (1)                   /* Simple mode                  */
#define FL_VERBO  (2)                   /* Verbose mode                 */
#define FL_ALGOR  (4)                   /* Algorithm mode               */

/* Macro for Algorithm mode */
#define ALGORITHM(a) if (mode & FL_ALGOR) printf(a)

/********************< prototypes >**********************************/
#ifdef _MSC_VER                                 /* Borland has it's own     */
int getopt(int, char**, char*);                 /* iterator and getopt() ret*/
#endif /* _MSC_VER */
void dns_get(struct domain_resp *);
static void usage(void);

/********************< External variables >*****************************/
extern  char    *optarg;                /* Argument from getopt()      */
extern  int     optind;                 /* Index where parsing stopped */


/********************< Global variables >***********************************/
char    *optlist        = "svA?h";              /* Flags for getopt()      */
static char not_avail[]={"<not available>"};    /* "not-available" message */

/***< data structure for received information from Domain Name Server >*****/
struct info {
     unsigned int flag_type;                    /* type query flags         */
     char cname_str[SIZE_LINE];                 /* Canonical name           */
     char ip_addrs_str[SIZE_LINE];              /* IP-addresses             */
     char mx_str[SIZE_LINE];                    /* Mail exchange            */
     char soa_str[SIZE_LINE];                   /* Time Zone                */
     char hinfo_str[SIZE_LINE];                 /* Hardware information     */
     char ns_str[SIZE_LINE];                    /* Name server              */
} info;

             /***************************************************
             * This array lists the different types of DNS      *
             * queries.  It is used during Verbose mode display.*
             ***************************************************/

static char    *types[] = {
                           "Unknown",           /* Not a valid DNS type     */
                           "ADDRESS",           /* ip (or other) ADDRESS    */
                           "NS",                /* Name Server              */
                           "MD",                /* Mail Destination <*>     */
                           "MF",                /* Mail Forwarder   <*>     */
                           "CNAME",             /* Canonical NAME           */
                           "SOA",               /* Start Of Authority       */
                           "MB",                /* MailBox          <!>     */
                           "MG",                /* Mail Group       <!>     */
                           "MR",                /* Mail Rename      <!>     */
                           "RT NULL",           /* Null RR          <!>     */
                           "WKS",               /* Well Known Services      */
                           "PTR",               /* Pointer                  */
                           "HINFO",             /* Hardware INFOrmation     */
                           "MINFO",             /* Mail list INFO   <!>     */
                           "MX",                /* Mail eXchange            */
                           "TXT",               /* TeXT strings             */
                           ""                   /* extra element, in case...*/
};
                                                /* <*> == obsolete, use MX  */
                                                /* <!> == experimental      */

/********************< Program usage >****************************/
static void usage(void)
{
        pr_notice("dns_smpl");                  /* Program Title        */
        printf("\
usage:  dns_smpl [ -s | -v | -A ]  <host>\n\
        dns_smpl -version | -? | -h \n\
                -s                Simple output mode\n\
                -v                Verbose output mode\n\
                -A                Algorithm output mode\n\
                -version          Display program version\n\
                -? | -h           Display this help message");
} /* end usage() */


/***********************< main >****************************/
void main (int argc, char *argv[])
{
    int         i, j;                           /* scratch storage          */
    int         mode = 0;                       /* output option            */
    int         name_dot;                       /* domain name searching    */
    struct      domain_q *dq;                   /* DNS query struct         */
    struct      domain_resp *drr;               /* DNS response struct      */

    int         size=SIZE_DNAME;                /* size of domain name      */
    char        buffer[SIZE_DNAME];             /* internal buffer          */
    char        name_buf[SIZE_DNAME];           /* buffer for original name */

    in_name     server[SERV_ARRAY_LEN];         /* name server addresses    */

                                                /* DNS types we query for   */
    unsigned    dns_qtype[NUM_REQ]={A, NS, SOA, HINFO, CNAME, MX};

                                                /* check for -version       */
    if((argc == 2) && (strcmpi(argv[1], "-version") == 0)) {
        pr_notice("dns_smpl");
        exit(0);                                /* normal return            */
    }

        /***** get the command line arguments, set up running modes *****/

    while ((i = getopt(argc, argv, optlist)) != EOF) {
        switch(i) {
            case 's':                           
                mode |= FL_SIMPL;               /* SIMPLE ouput mode        */
                break;
            case 'v':
                mode |= FL_VERBO;               /* VERBOSE ouput mode       */
                break;
            case 'A':
                mode |=FL_ALGOR;                /* ALGORITHM ouput mode     */
                break;
            case '?':
            case 'h':
            default:
                usage();                        /* Usage/Help option        */
                exit(0);                        /* normal return            */
                break;
        }
    }
    if (mode == FL_NOMOD) mode = FL_SIMPL;      /* default if no option     */

    if (optind >= argc) {                       /* not enough arguments?    */
            usage();                            /* no name to resolve       */
            exit(1);            
    }

    strcpy(name_buf, argv[optind]);             /* save original name arg   */

    ALGORITHM("The DOMAIN library sample does the following:\n");

    ALGORITHM("1.  Make a Fully Qualified Domain Name (name.domain)\n");

         /*************************************************************
         * The DNS system will resolve only fully qualified domain    *
         * names.  We have to check the name from the command line.   *
         *                                                            *
         * We determine if a name is fully qualified as a domain name *
         * by searching for one or more dots.  A host name will have  *
         * none, while a fully qualified domain name will have at     *
         * least one.                                                 *
         *      An example of a host name would be:  "foo"            *
         *      Examples of fully qualified domain names would be:    *
         *              "nic.ddn.mil" -or- "foo.bar.com"              *
         *************************************************************/

    for (i=0, name_dot = FALSE; i < (int)strlen(name_buf) ; i++) {
        if (name_buf[i] == '.') {
            name_dot = TRUE;                    /* seaching for dot         */
            break;
         }
    }

         /*************************************************************
         * If it's not a fully qualified domain name, we get the      *
         * default domain name from the PCTCP kernel and append it to *
         * the host name.                                             *
         *************************************************************/

    if (!name_dot) {
        if (get_kernel_info (0, TAG_DOMAIN, 0, (void far *)&buffer,
                                 (unsigned far *)&size) != NOERROR) {
            puts("No default domain set");
            exit(1);
        }
        strcat(name_buf,".");                     /* fully qualified        */
        strcat(name_buf, buffer);                 /*  domain name           */
    }

    ALGORITHM("2.  Get the list of Domain Name Servers from the PC/TCP Kernel\n");

             /***************************************************
             * We now have to get the address of a Domain Name  *
             * Server which can answer our queries.  The kernel *
             * has up to 3, only one of which we use in this    *
             * sample program.  A more robust technique would be*
             * to fall back to other Name Servers if the first  *
             * one failed to answer our query.                  *
             ***************************************************/

    size = sizeof (in_name) * SERV_ARRAY_LEN;
                                                  /* get domain name servers*/
    if (get_kernel_info (0, TAG_DNS_SRVS, 0, (void far *)&server,
                                 (unsigned far *)&size) != NOERROR) {
        puts("No domain servers set");
        exit(1);
    }
         
    /* We allocate a query structure. The structure allocated includes a
     * header structure as well as a response structure.
     */

    ALGORITHM("3.  Allocate a DNS query structure\n");
        
    if ((dq = alloc_domain_q(SIZE_DOMAIN_Q)) == 0) {
        puts("Couldn't allocate query structure.");
        exit(1);
    }

    dq->a.fhost = server[0];                      /* Name Server's IP addr  */

             /***********************************************************
             * We will do recursive queries.  A recursive query means   *
             * that if the Name Server does not have an answer to the   *
             * query, it will ask another Name Server.  The way this    *
             * works is that the Name Servers are organized on the      *
             * net into a tree based on Domain.  We can illustrate      *
             * this by looking at the domain name "boris.org.com" and   *
             * following the recursion:                                 *
             *                                                          *
             *   Lets say our host is "natasia.baltimore.us"            *
             *                                                          *
             *   1st:   We ask the local Domain Name Server, the one    *
             *          for the "baltimore.us" domain (or zone).  We    *
             *          are querying for the IP Address of              *
             *          "boris.org.com" so we can send boris a          *
             *          message.                                        *
             *                                                          *
             *   2nd:   It doesn't know so it asks the Name Server      *
             *          for the "us" domain (for the purposes of this   *
             *          discussion a zone and a domain are the same     *
             *          thing).                                         *
             *                                                          *
             *   3rd:   It doesn't know so it asks the Name Server      *
             *          for the "com" domain.                           *
             *                                                          *
             *   4th:   It doesn't know so it asks the Name Server      *
             *          for the "org" domain.                           *
             *                                                          *
             *   5th:   It knows, and passes the answer back to the     *
             *          4th Name Server, who passes it back to the      *
             *          3rd Name Server, and so on until the 1st Name   *
             *          server passes it back an answer.  In the        *
             *          process each Name Server "caches" the           *
             *          information for future reference, should        *
             *          anyone happen to ask for it.                    *
             *                                                          *
             *  If we hadn't set the recursive bit the query would      *
             *  have stopped with the 1st Name Server and we wouldn't   *
             *  have gotten an answer.                                  *
             *                                                          *
             *  If you look at the flags in Verbose Output Mode you     *
             *  will see the recursion flag.  You can try setting it    *
             *  to 0 here to see what it will do to the query, and      *
             *  the response you get back.                              *
             ***********************************************************/


    dq->recurse = 1;                              /* Do DNS recursive query */

             /***************************************************
             * Below is the main loop where we do all of our    *
             * queries.  NUM_REQ is the number of requests we   *
             * have to make.  The array dns_qtype[] contains the*
             * the list of actual query types.                  *
             ***************************************************/

    ALGORITHM("4.  For all DNS Queries (A, NS, SOA, HINFO, CNAME, MX):\n");

    for (j = 0; j < NUM_REQ; ++j) {               /* work w/ NUM_REQ queries*/
        dq->nq[0]->q_type = dns_qtype[j];         /* Next type              */
        dq->nq[0]->name = name_buf;

        ALGORITHM("5.    Send a DNS query to a foreign Name Server.\n");

        if (dns_query(dq) != NOERROR) {           /* send query             */
            pneterror("sending query");
            continue;
        }

        if (mode & FL_VERBO) {                     /* work in VERBOSE mode  */
            printf("\n\t\t-- <<< **** The DNS query going out: **** >>> --\n\n");
            (void)dns_h_print(dq->dmp);             /* display header       */
                        
            printf(" Machine name to resolve %s\n",dq->nq[0]->name);
            printf(" Query type %d -- %s\n",
                dq->nq[0]->q_type, types[(int)dq->nq[0]->q_type]);
        }                                         /* end of VERBOSE mode */

        ALGORITHM("6.    Receive a Resource Record list response from the Name Server\n");

        if (dns_rcv(dq) != NOERROR) {             /* receive answer      */
                pneterror("receiving DNS response");
                continue;
        }

        if (dq->partial)                        /* partial answer?       */
                puts("\n\t\t\t--->>> Partial DNS answer received\n");

        ALGORITHM("7.    Print the response from the Name Server (Verbose mode)\n");

        if( mode & FL_VERBO ) {                   /* header print        */
            printf("\n\t\t  <<< **** The DNS response received: **** >>>\n\n");
            if ((dns_h_print(dq->dmp) != NOERROR) && (mode & FL_ALGOR))
                printf("Error dns_h_print()\n");   /* dq->dmp == null!   */

            drr = dq->dr;                          /* print all responses*/
            do {
                dns_print(drr);                    /* print response     */
            } while ((drr = drr->next) != (struct domain_resp *)0);

            printf("\n\t\t\t\t<Hit Enter to continue.>\n");
            while (!kbhit());                       /* wait for readability */
            i = getchar();                          /* clear the buffer     */
        } /* end of VERBOSE mode */

        if( mode & FL_SIMPL ) {                   /* work in SIMPLE mode */
            drr = dq->dr;

            do {
                dns_get(drr);                     /* fill structure info */
            } while ((drr = drr->next) != (struct domain_resp *)0);
        } /* end of SIMPLE mode */

        ALGORITHM("8.    Free the received Resource Record list\n");

        if ((free_resp_list(dq) != NOERROR) && ( mode & FL_ALGOR ))
            printf(" Error in free_resp_list()\n");
    }

    ALGORITHM("9.  Display the DNS response(s) (Simple mode)\n");

    if (mode & FL_SIMPL) {

        printf("\
  Host Name:            %s\n\
  IP Address(s):        %s\n\
  Name Server(s):       %s\n\
  Zone Information:     %s\n\
  Hardware Information: %s\n\
  Canonical Name:       %s\n\
  Mail Exchange Info:   %s\n",
        argv[optind],
        (info.flag_type & FL_IPADDR) ?  info.ip_addrs_str : not_avail,
        (info.flag_type & FL_NS)     ?  info.ns_str       : not_avail,
        (info.flag_type & FL_SOA)    ?  info.soa_str      : not_avail,
        (info.flag_type & FL_HINFO)  ?  info.hinfo_str    : not_avail,  
        (info.flag_type & FL_CNAME)  ?  info.cname_str    : name_buf,      
        (info.flag_type & FL_MX)     ?  info.mx_str       : not_avail);
    } /* end of SIMPLE mode */

    ALGORITHM("10. Free the DNS query structure\n");

    if ((free_domain_q(dq) != NOERROR) && (mode & FL_ALGOR))
        printf("Error in free_domain()\n");
} /* end main() */

/*****************************************************************************/

             /*******************< dns_get() >***********************
             * This function analyses the contents of answer we get *
             * from Domain Name Server and fills the structure info *
             * for display.                                         *
             *******************************************************/

void dns_get(struct domain_resp *dr)
{
    char bufin[10];                               /* internal buffer        */
    
    switch (dr->type) {
        case CNAME:                               /* canonical  name response*/
            info.flag_type |= FL_CNAME;
            info.cname_str[0]='\0';
            strcat(info.cname_str, dr->data.qcname.qcname);
            break;
        case A:                                   /* IP address response    */
            info.flag_type |= FL_IPADDR;
            strcat(info.ip_addrs_str,
                inet_ntoa(dr->data.qa.address));
            strcat(info.ip_addrs_str," ");
            break;
        case MX:
            info.flag_type |=FL_MX;               /* Mail exchange response */
            strcat(info.mx_str, dr->data.mx.exchange);
            strcat(info.mx_str, "(");
            strcat(info.mx_str, ltoa(dr->data.mx.preference, bufin, 10));
            strcat(info.mx_str, ") ");
            break;
        case SOA:                                 /* Zone information       */
                                                  /*   response             */
            info.flag_type |= FL_SOA;
            strcat(info.soa_str, dr->data.soa.mname);
            strcat(info.soa_str, ", ");
            strcat(info.soa_str, dr->data.soa.rname);
            strcat(info.soa_str, ", ");
            strcat(info.soa_str, ltoa(dr->data.soa.serial, bufin, 10));
            strcat(info.soa_str, ", ");
            strcat(info.soa_str, ltoa(dr->data.soa.refresh, bufin, 10));
            strcat(info.soa_str, ", ");
            strcat(info.soa_str, ltoa(dr->data.soa.retry, bufin, 10));
            strcat(info.soa_str, ", ");
            strcat(info.soa_str, ltoa(dr->data.soa.expire, bufin, 10));
            strcat(info.soa_str, ", ");
            strcat(info.soa_str, ltoa(dr->data.soa.minimum, bufin, 10));
            break;

        case HINFO:                                /* Hardware information  */
            info.flag_type |= FL_HINFO;
            strcat(info.hinfo_str, dr->data.hinfo.machine);
            strcat(info.hinfo_str, ";  ");
            strcat(info.hinfo_str, dr->data.hinfo.os);
            break;
        case NS:                                  /* Domain Name Server     */
            info.flag_type |= FL_NS;
            strcat(info.ns_str, dr->data.nsdname.nsdname);
            strcat(info.ns_str, " ");
            break;
    }
} /* end dns_get() */

/***************** 
 * $Log:   H:/devkit.dos/vcs/srcsmpl/dns_smpl/src/dns_smpl.c_v  $ 
 * 
 *    Rev 1.0   20 Apr 1993 15:08:44   paul
 * Initial revision.
 */
