/* The following Neural Net code is designed to control
 * a cart and inverted pole system. The neural net and pole system are
 * described in:
 * Barto, Sutton, and Anderson, "Neuronlike Adaptive Elements That Can
 *        Solve Difficult Learning Control Problems", IEEE Trans. Syst.
 *        , Man, Cybern., Vol. SMC-13, pp. 834-846, Sept.-Oct. 1983
 * 
 * The following algorithm was adapted by Dr. Charles Anderson to run
 * with a continuous error feedback.
 */

#include "vsuser.h"
#include <stdio.h>

typedef double NEURALBOX[3][3][6][3];
NEURALBOX nb;

/*
 * The getBox routine separates the cart-pole state space into a 4 dimensional
 * space 3x3x6x3 on X, X dot, theta, and theta dot respectively.
 */
static double *getBox( pstate)
  double FAR*pstate;
{
  int w,x,y,z;
  /* Partition X state */
  if (pstate[0] < -.8)	   w = 0;
  else if (pstate[0] < .8) w = 1;
  else			   w = 2;
  
  /* Partition X Dot state */
  if (pstate[1] < -.5)	    x = 0;
  else if (pstate[1] < .5)  x = 1;
  else			    x = 2;
  
  /* Partition Theta state */
  /* Note, angles are in radians */
  if (pstate[2] < -.1047192)	  y = 0;
  else if (pstate[2] < -.0174532) y = 1;
  else if (pstate[2] < 0)	  y = 2;
  else if (pstate[2] < .0174532)  y = 3;
  else if (pstate[2] < .1047192)  y = 4;
  else                            y = 5;

  /* Partition Theta Dot state */
  if (pstate[3] < -.87266)	z = 0;
  else if (pstate[3] < .87266)  z = 1;
  else				z = 2;
  return &(nb[w][x][y][z]);
}

static double *lastBox;

/* First inSig is error signal, next are x, xDot, theta, thetaDot
 * respectively.
 * Note that, as written, the neural net itself is statically
 * allocated. Thus if more than one instance of a neural net occurs
 * in a diagram, the nets will overwrite each other.  Future versions
 * of the VSDK will allow the user to globally allocate memory and store
 * the pointer in the 3rd (param) arg to the user function.
 */
void PASCAL neuralNet(  param, inSig, outSig )
  double FAR inSig[], FAR outSig[], FAR param[];
{
  int box;
    
  /* Accumulate weighted error in sensor measurement space */ 
  if (lastBox) *lastBox += param[0] * inSig[0];

  /* inSig[0] is error signal, sensor data starts at inSig[1] */
  lastBox = getBox( &inSig[1]);
  outSig[0] = *lastBox;
}

#include <stdio.h>

/* Called at Sim Start */
void PASCAL neuralNetSS( double FAR param[], long FAR*runCount )
{
  FILE *f;
  int cnt;

  lastBox = 0;
  if (*runCount > param[1])
    {
      stopSimulation( 2 );
      return;
    }
  /* read state from a file */
  f = fopen( "nnet.dat", "rb");
  if (!f) return;
  cnt = fread( nb, 1, sizeof(NEURALBOX), f);
#if DEBUGGING
  if (cnt != sizeof(NEURALBOX) && ferror(f)) debMsg("read Error");
  else 
    debMsg("Successfully read %d bytes of neural net weights (rc = %d)"
	    , cnt, *runCount );
#endif
  fclose( f );
}

/* Called at Sim End */
void PASCAL neuralNetSE( double FAR param[], long FAR*runCount )
{
  FILE *f;

  /* Don't write to file until last iteration */
  if (*runCount < param[1]) return;
  /* write net state to file */
  f = fopen( "nnet.dat", "w+b");
  if (!f) return;
  fwrite( nb, 1, sizeof(NEURALBOX), f);
  fclose( f );
}

#define NNPARAMS 2

/* Parameter Alloc   
 * paramCount is number of default, VisSim prompted parameters   
 * Return total bytes of parameter space for VisSim to alloc,
 * including above params.
 */
long PASCAL neuralNetPA( pCount )
  int FAR*pCount;
{
  *pCount = NNPARAMS; /* one prompted param, the rest are unprompted */
  return sizeof(NEURALBOX) + sizeof(double)*NNPARAMS;
}

/* Parameter Init
 * provide initial values for parameters
 */
void PASCAL neuralNetPI( DOUBLE *param)
{
  param[0] = .5; /* set default learning rate */
  param[1] = 10; /* Default iterations */
}

/* Param Change */
char FAR* PASCAL neuralNetPC( DOUBLE *param)
{
  return "Learning Rate;Iteration Count";
}

int DLLInst;

/* Required Windows DLL stuff  */
int FAR PASCAL LibMain( hInst, dataSeg, heapSize, cmd)
   int hInst; unsigned dataSeg, heapSize; char FAR* cmd;
{
  DLLInst = hInst;
  return TRUE;
}

int FAR PASCAL WEP( parm )
{
  return 1;
}
