/*  @(#)uprmeans.c	5.1 as of 3/19/93 15:33:32  */

/*
 * uprmeans - SPSS User Proc to report means of selected variables
 *
 * Command syntax
 *      USER PROC NAME UPRMEANS <variable list>.
 */

#include <string.h>

#include "usercode.h"
#include "cmnset.h"
#include "cmnusr.h"
#include "cmnpar.h"

#define MAX_LAB     40          /* Maximum var label length */

/*
 * Local function prototypes
 */

static int parse(               /* Returns non-0 if parse error */
    SI* indexs,                 /* Returned as vars' indexes */
    INTEGER* maxvars,           /* Capacity of indexs */
    INTEGER* nvar);             /* Returned as number of vars */

static int compute(             /* Returns non-0 at end of file */
    OBS* obsvec,                /* Observation vector */
    SI* indexs,                 /* Indexes of vars to process */
    LR* sum,                    /* Returned as sums of vars */
    LR* xn,                     /* Returned as case counts */
    OBS* xobvec,                /* Values of variables */
    INTEGER* nvar);             /* Number of variables */

static void report(             /* Print the results */
    SI* indexs,                 /* Indexes of vars processed */
    LR* sum,                    /* Sum of each var's values */
    LR* xn,                     /* Case counts */
    INTEGER* nvar);             /* Number of variables */


/*
 * Usrset - Initialization for a user procedure
 */

SUBROUTINE Usrset(void)         /* User code for initialization */
{

    /* Elect to use SPSS's lexical analyzer */
    LOG_SET_TRUE(CmnSet->qstack);

    /* We will not be adding any new variables */
    CmnSet->mxnewv = 0L;

}


/*
 * Usrpgm - Business part of a user procedure
 */

SUBROUTINE Usrpgm(              /* User code for user proc */
    OBS obsvec[],               /* Observation vector */
    OBS wrkspc[],               /* Workspace */
    INTEGER* ndbles)            /* Size of workspace */
{
    INTEGER maxvars;            /* Maximum vars we can process */
    INTEGER nvar;               /* Number of vars to process */
    SI* pindexs;                /* Pointer to indexs vector */
    LR* psum;                   /* Pointer to vector of sums */
    LR* pxn;                    /* Pointer to vector of counts */
    OBS* pxobvec;               /* Pointer to vector of values */

    /* Calculate program capacity insuring doubleword alignment */
    maxvars = (*ndbles*sizeof(OBS))/
        (sizeof(SI)+2*sizeof(LR)+sizeof(OBS));
    maxvars &= -(long)sizeof(OBS);

    /* Partition the workspace */
    pindexs = (SI*)wrkspc;
    psum = (LR*)(pindexs+maxvars);
    pxn = (LR*)(psum+maxvars);
    pxobvec = (OBS*)(pxn+maxvars);

    /* Parse the command */
    if (!parse(pindexs,&maxvars,&nvar) &&
        ISFALSE(CmnUsr->qfatal) && ISFALSE(CmnUsr->qedit)) {

        /* Data pass initialization */
        LOG_SET_TRUE(CmnUsr->qfinal);
        Upinit();

        /* Read the cases in one split-file group */
        while (!compute(obsvec,pindexs,psum,pxn,pxobvec,&nvar)) {

            /* Print the results for that group */
            report(pindexs,psum,pxn,&nvar);
        }
    }
}


/*
 * parse - Parse the command.
 *
 * Returns non-zero if an error is detected 
 */

static int parse(               /* Returns non-0 if parse error */
    SI* indexs,                 /* Returned as vars' indexes */
    INTEGER* maxvars,           /* Capacity of indexs */
    INTEGER* nvar)              /* Returned as number of vars */
{
    static INTEGER erdisym =    /* Display current token */
        ERDISYM;
    INTEGER errnum;             /* Error number */
    static INTEGER ersers =     /* Serious error */
        ERSERS;
    LOGICAL qerr;               /* Set by Usvars if any error */
    static INTEGER vartyp = 1L; /* Acceptable var types - numeric */

    /* Get first token - should be first variable name */
    Usnext();

    /* Be tolerant and skip any preceding slash */
    if (ISTRUE(CmnPar->qslash)) {
        Usnext();
    }

    /* Process the list of variable names */
    *nvar = 0L;
    Usvars(&vartyp,maxvars,indexs,nvar,&qerr);
    if (ISFALSE(qerr)) {

        /* We should be at end of command */
        if (ISTRUE(CmnPar->qslash)) {
            Usnext();
        }
        if (ISFALSE(CmnPar->qeocmd)) {
            errnum = 13652L;
            Uerror(&errnum,&erdisym,&ersers);
            LOG_SET_TRUE(qerr);
        }
    }
    return ISTRUE(qerr);
}


/*
 * compute - Accumulate the information for a single split-file
 *
 * Returns non-zero if end of file reached.
 */

static int compute(             /* Returns non-0 at end of file */
    OBS* obsvec,                /* Observation vector */
    SI* indexs,                 /* Indexes of var to process */
    LR* sum,                    /* Returned as sums of vars */
    LR* xn,                     /* Returned as case counts */
    OBS* xobvec,                /* Values of variables */
    INTEGER* nvar)              /* Number of variables */
{
    int ix;                     /* Index of indexs, sum, and xn */
    INTEGER nmiss;              /* Number of missing values */
    LOGICAL qignor;             /* Set to ignore user missing */

    /* Initialize for next splitfile group */
    for (ix = 0; ix < *nvar; ix++) {
        sum[ix] = 0.;
        xn[ix] = 0.;
    }
    LOG_SET_FALSE(qignor);

    /* Loop thru the cases in one splitfile group */
    do {

        /* Obtain one case */
        Uread(obsvec);
        if (ISFALSE(CmnUsr->qfinf) && ISFALSE(CmnUsr->qfing)) {

            /* Filter selected variables for missing values */
            Umscop(obsvec,&qignor,&nmiss,xobvec,nvar,indexs);

            /* Accumulate sums and n's */
            for (ix = 0; ix < *nvar; ix++) {
                if (xobvec[ix] != CmnUsr->sysmis) {
                    xn[ix] += CmnUsr->caswgt;
                    sum[ix] += CmnUsr->caswgt*xobvec[ix];
                }
            }
        }
    } while (ISFALSE(CmnUsr->qfinf) && ISFALSE(CmnUsr->qfing));
    return ISTRUE(CmnUsr->qfinf);
}


/*
 * report - Print the results for a single split-file group.
 */

static void report(             /* Print the results */
    SI* indexs,                 /* Indexes of vars processed */
    LR* sum,                    /* Sum of each var's values */
    LR* xn,                     /* Case counts */
    INTEGER* nvar)              /* Number of variables */
{
    static INTEGER decn = 0L;   /* Fractional digits for count */
    INTEGER fmtcod;             /* Format code */
    static INTEGER fmccomma =   /* "COMMA" format code */
        FMCCOMMA;
    INTEGER fmtdec;             /* Fractional digits */
    INTEGER fmtwid;             /* Field width */
    INTEGER ierr;               /* Set by Uoutcv on an error */
    INTEGER ivar;               /* Index of a variable */
    int ix;                     /* Index of indexs, sum, and xn */
    static INTEGER k2 = 2L;     /* Constant 2 */
    static INTEGER k7 = 7L;     /* Constant 7 */
    static INTEGER len0 = 0L;   /* To print 0 characters */
    INTEGER lenx;               /* To print length of pline */
    char pline[81];             /* Print buffer */
    static INTEGER skpone = 1L; /* To skip 1 line */
    static INTEGER skptwo = 2L; /* To skip 2 lines */
    static INTEGER widlab =     /* Width of variable label */
        MAX_LAB;
    static INTEGER widmn = 20L; /* Width of encoded mean */
    static INTEGER widn = 10L;  /* Width of encoded n */
    LR xmean;                   /* Mean of one variable */

    /* Print results for one splitfile group */
    Uheadr();
    Ulinen(&skptwo);
    lenx = 79L;
    Uprint(
        "Variable "
        "Label                                   "
        "         N"
        "                Mean",
        &lenx,&skptwo);
    Ulinen(&skpone);
    Uprint("",&len0,&skpone);
    for (ix = 0; ix < *nvar; ix++) {

        /* Calculate the mean */
        if (xn[ix] == 0.) {
            xmean = CmnUsr->sysmis;
        } else {
            xmean = sum[ix]/xn[ix];
        }

        /* Get the name, label, and format of the variable */
        ivar = indexs[ix];
        Udicgt(&ivar);
        memcpy(&pline[0],CmaUsr->diname,8);
        Uclowr(pline,&k2,&k7);
        pline[8] = ' ';
        Uvarlb(&widlab,&pline[9]);
        Ufmts(&CmnUsr->dipfmt,&fmtcod,&fmtwid,&fmtdec);

        /* Encode n using "comma" format */
        Uoutcv(&xn[ix],&fmccomma,&widn,&decn,&pline[49],&ierr);

        /* Encode mean using print format + 1 fractional digit */
        fmtdec++;
        Uoutcv(&xmean,&fmtcod,&widmn,&fmtdec,&pline[59],&ierr);
        Ulinen(&skpone);
        lenx = 79L;
        Uprint(pline,&lenx,&skpone);
    }
}
