/******************************************/
/* xwinged.c                              */
/* Copyright (C) 1993 David Bollinger     */
/* A Ship Editor For Lucas Arts' X-Wing   */
/* compiled w/ Borland C++ 3.1 & TASM 2.0 */
/******************************************/

#pragma inline
#pragma option -ms
#include <mem.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include "xwinged.h"

extern int _wscroll = 0;

/***************************************************************************/
main()
   {
   if (ReadShips() == SUCCESS)
      {
      DrawScreen();
      Editor();
      clrscr();
      if (WriteShips() != SUCCESS)
         printf("Error writing ship data.\r\n");
      }
   else
      printf("Error reading ship data.\r\n");

   return SUCCESS;
   }

/***************************************************************************/
void Editor(void)
   {
   editmode = HEXEDITOR;

   while (editmode != QUITEDITOR)
      {
      if (editmode == HEXEDITOR)
         HexEditor();
      if (editmode == FIELDEDITOR)
         FieldEditor();
      }
   }


void HexEditor(void)
   {
   int c, cx, cy;

   curship = curbyte = curnibble = 0;

   /****************************/
   /* recalculate ship pointer */
   /****************************/
   ChangeShip(0);

   while (editmode == HEXEDITOR)
      {
      DisplayShip();
      PositionCursor();
      c = dgetch();
      switch(c)
         {
         case ESC  : editmode = QUITEDITOR; break;
         case TAB  :
         case BKTB : editmode = FIELDEDITOR; break;
         case PGUP : ChangeShip(-1); break;
         case PGDN : ChangeShip( 1); break;
         case UP   : ChangeByte(-16); break;
         case DOWN : ChangeByte(+16); break;
         case LEFT : ChangeNibble(-1); break;
         case RGHT : ChangeNibble( 1); break;
         default   : if (isxdigit(c))
                        SaveByte((uchar) c);
                     break;
         }
      }
   }


void FieldEditor(void)
   {
   int curfield=0, retval;

   /****************************/
   /* recalculate ship pointer */
   /****************************/
   ChangeShip(0);

   while (editmode == FIELDEDITOR)
      {
      DisplayShip();
      switch(curfield)
         {
         case  0 : retval = DVgets(16, 1, 3,ship->shortname); break;
         case  1 : retval = DVgets(16, 2,24,ship->longname); break;
         case  2 : retval = DVgets(16, 3, 5,ship->filename);  break;
         case  3 : retval = DVgeti(16, 4, 5,&ship->power);    break;
         case  4 : retval = DVgeti(16, 5, 5,&ship->speed);    break;
         case  5 : retval = DVgeti(16, 6, 5,&ship->shieldsf); break;
         case  6 : retval = DVgeti(16, 7, 5,&ship->shieldsr); break;
         case  7 : retval = DVgeti(16, 8, 5,&ship->hull);     break;
         default : curfield = 0; break;
         }
      switch(retval)
         {
         case QUIT      : editmode = QUITEDITOR; break;
         case NEXTBLOCK : editmode = HEXEDITOR; break;
         case PREVSHIP  : ChangeShip(-1); break;
         case NEXTSHIP  : ChangeShip( 1); break;
         case PREVFIELD : if (curfield > 0) curfield--; break;
         case NEXTFIELD : if (curfield < NUMFIELDS-1) curfield++; break;
         }
      }
   }


/***************************************************************************/
/* NOTE: dir video is fast enough we can redisplay entire screen each time */
/***************************************************************************/
void DisplayShip(void)
   {
   register int i;
   uchar *sptr, c;
   int byte, x=0, y=0;

   /***************/
   /* STATUS LINE */
   /***************/
   DVprintf(0, 24, TITLECOLOR, " XWingEd (C) 1993 David Bollinger  %03d %-39s ", curbyte, descript[curbyte]);

   /*************************************************************************/
   /* DECODED DISPLAY - note space at end of format string to clean display */
   /*************************************************************************/
   DVprintf( 2, 0, DATACOLOR, "Ship Number : %-2d ",  curship);
   DVprintf( 2, 1, DATACOLOR, "Short Name  : %-4s ",  ship->shortname);
   DVprintf( 2, 2, DATACOLOR, "Long Name   : %-24s ", ship->longname);
   DVprintf( 2, 3, DATACOLOR, "CP FileName : %-5s ",  ship->filename);
   DVprintf( 2, 4, DATACOLOR, "Power       : %-5d ",  ship->power);
   DVprintf( 2, 5, DATACOLOR, "Speed       : %-5d ",  ship->speed);
   DVprintf( 2, 6, DATACOLOR, "Shields (F) : %-5d ",  ship->shieldsf);
   DVprintf( 2, 7, DATACOLOR, "Shields (R) : %-5d ",  ship->shieldsr);
   DVprintf( 2, 8, DATACOLOR, "Hull        : %-5d ",  ship->hull);


   DVputs(WEAPONX, 0, DATACOLOR, "W#  X-axis Y-axis Z-axis PrjCnt");
   for (i=0; i<NUMWEAPONS; i++)
      DVprintf(WEAPONX, i+1, DATACOLOR, "%1d   %-6d %-6d %-6d %-6d ", i+1,
               ship->weapon[i].xaxis,
               ship->weapon[i].yaxis,
               ship->weapon[i].zaxis,
               (int) ship->weapon[i].projectilecount);

   /************/
   /* HEX DUMP */
   /************/
   sptr = (uchar *) ship;
   for (byte=0; byte<224; byte++)
      {
      y = (int) (byte/16);
      x = (byte%16);
      c = *(sptr++);
      DVprintf(DUMPX+x*3, DUMPY+y, HEXCOLOR, "%02X", c);
      DVprintf(DUMPTXTX+x, DUMPY+y, TEXTCOLOR, "%c", (c==0) ? '.' : c );
      }
   }

/***************************************************************************/
/* PositionCursor figures out the screen position for the hex editor       */
/***************************************************************************/
void PositionCursor(void)
   {
   gotoxy(DUMPX+(curbyte%16)*3+curnibble+1, DUMPY+(int)(curbyte/16)+1);
   }

/***************************************************************************/
/* NAVIGATION THROUGH THE HEX DATA                                         */
/***************************************************************************/
void ChangeShip(int inc)
   {
   if ((curship+inc < NUMSHIPS) AND (curship+inc >= 0))
      curship += inc;
   ship = &ships[curship];
   }


void ChangeByte(int inc)
   {
   if ((curbyte+inc < BYTESPERSHIP) AND (curbyte+inc >= 0))
      curbyte += inc;
   }


void ChangeNibble(int inc)
   {
   if (curnibble+inc > 1)
      {
      curnibble = 0;
      ChangeByte( 1);
      }
   else if (curnibble+inc < 0)
      {
      curnibble = 1;
      ChangeByte(-1);
      }
   else
      curnibble+=inc;
   }


void SaveByte(uchar c)
   {
   static uchar mask[2] = {0x0f,0xf0};
   static uchar shft[2] = {   4,   0};
   uchar *sptr;

   c = ((c > '9') ? c-55 : c-48) & 0x0f;
   sptr = ((uchar *) ship) + curbyte;
   *sptr = (*sptr & mask[curnibble]) | (c << shft[curnibble]);
   ChangeNibble(1);
   }

/***************************************************************************/
/* FILE STUFF                                                              */
/***************************************************************************/
int ReadShips(void)
   {
   FILE *file;

   if ((file=fopen(DATAFILE, "rb+")) != NULL)
      if (fseek(file, SHIPOFFSET, SEEK_SET) == 0)
         if (fread(ships, sizeof(ships), 1, file) == 1)
            if (fclose(file) == 0)
               return SUCCESS;
   return FAILURE;
   }


int WriteShips(void)
   {
   FILE *file;

   if ((file=fopen(DATAFILE, "rb+")) != NULL)
      if (fseek(file, SHIPOFFSET, SEEK_SET) == 0)
         if (fwrite(ships, sizeof(ships), 1, file) == 1)
            if (fclose(file) == 0)
               return SUCCESS;
   return FAILURE;
   }

/**************************************************************************/
/* DV - Direct Video stuff - output - hardcoded for color monitors        */
/**************************************************************************/
void DVprintf(int x, int y, int attr, char *fmt, ...)
   {
   va_list argptr;
   char str[160];

   va_start(argptr, fmt);
   vsprintf(str, fmt, argptr);
   DVputs(x, y, attr, str);
   }


void DVputs(int x, int y, int attr, char *str)
   {
   while (*str != '\0')
      DVputc(x++, y, attr, *str++);
   }


void DVputc(int x, int y, int attr, uchar chr)
   {
   asm   mov   bx, y
   asm   mov   ax, 80
   asm   imul  bx
   asm   add   ax, x
   asm   shl   ax, 1
   asm   mov   bx, ax
   asm   mov   ax, 0b800h
   asm   mov   es, ax
   asm   mov   ah, byte ptr attr
   asm   mov   al, byte ptr chr
   asm   mov   word ptr es:[bx], ax
   /********************************************************************/
   /* if TASM is not available you can use this code instead,          */
   /* although it so slow you will want to rewrite the display section */
   /********************************************************************/
   /*
   textattr(attr);
   gotoxy(x+1, y+1);
   if (chr >= 32)
      putch(chr);
   else
      putch('.');
   */
   }

/***************************************************************************/
/* DVget? - direct video input - to get around the limitations of gets()   */
/***************************************************************************/
int DVgets(int x, int y, int len, char *dest)
   {
   uchar tempstr[80];
   int c, curx=0, edited=FALSE, retval=STILLEDITING;

   memset(tempstr, 0, 80);
   strcpy(tempstr, dest);
   while (retval == STILLEDITING)
      {
      gotoxy(x+curx+1, y+1);
      c = dgetch();
      switch(c)
         {
         case ESC  : retval = QUIT;      break;
         case ENTR : retval = NEXTFIELD; break;
         case TAB  :
         case BKTB : retval = NEXTBLOCK; break;
         case PGUP : retval = PREVSHIP;  break;
         case PGDN : retval = NEXTSHIP;  break;
         case UP   : retval = PREVFIELD; break;
         case DOWN : retval = NEXTFIELD; break;
         case BKSP :
         case LEFT : if (curx > 0) curx--; break;
         case RGHT : if (curx < len-1) curx++; break;
         default   : if (isprint(c))
                        {
                        edited = TRUE;
                        DVputc(x+curx, y, DATACOLOR, tempstr[curx] = c);
                        if (curx < len-1) curx++;
                        }
                     break;
         }
      }
   if (edited)
      strcpy(dest, tempstr);
   return retval;
   }


int DVgeti(int x, int y, int len, int *dest)
   {
   int retval;
   char tempstr[16];

   itoa(*dest, tempstr, 10);
   if ((retval = DVgets(x, y, len, tempstr)) == NEXTFIELD)
      *dest = atoi(tempstr);
   return retval;
   }


int DVgetc(int x, int y, int len, uchar *dest)
   {
   int retval;
   char tempstr[16];

   tempstr[0] = *dest;
   if ((retval = DVgets(x, y, len, tempstr)) == NEXTFIELD)
      *dest = tempstr[0];
   return retval;
   }

/***************************************************************************/
/* dgetch() - a simple way to get most extended keys                       */
/***************************************************************************/
int dgetch(void)
   {
   int c;

   if ((c = getch()) == 0)
      c = -getch();
   return c;
   }

/***************************************************************************/
void DrawScreen(void)
   {
   register int i;

   clrscr();
   for (i=1; i<79; i++)
      DVputc(i, DUMPY-1, BORDERCOLOR, '');

   for (i=0; i<24; i++)
      {
      DVputc( 0, i, BORDERCOLOR, '');
      DVputc(79, i, BORDERCOLOR, '');
      }
   }

/***************************************************************************/
/* end of xwinged.c */


