/*
 * samlogic.c
 *
 * This program takes a callsign from the command line and searches
 * the sam database for a match.  If found the call data is
 * displayed in a 1 line comma delimited format:
 *
 * call,class,first m,last,address,city,state,zip,yearborn,[,county]
 *
 * If /c follows the call then the county is displayed if possible
 *
 * findcall depends on the API to the sam db provided by SAMAPI.EXE.  
 *
 * Any error messagess are 1 line long and start with "***" (in case
 * some other program wants to use the output of this program).
 *
 * to compile with Turbo-C
 *  tcc findcall.c samlib.c
 *
 * Other C compilers can be used, but a "pack structs" option may be
 * required.
 *
 * Copyright 1991 by Tom Thompson, Athens, AL and by RT Systems
 *                   [n4yos]
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#include "samapi.h"			/* samapi interface spec */

/*
 * functions in samlib.c
 */

int LocateSam(void);
int CallSam(int cmd, void far *cmdbuf, void far *rspbuf);

/*
 * decls for functions contained here
 */

void display_call(datarec_t * d, int wantcty, char *county);


/*******************
 * main
 *******************
 *
 * in: argv[1] is callsign
 *
 * returns (or exits()) non-zero if error, zero if call found and displayed
 */

main(int argc, char *argv[])
{
	int err;
	int wantcty = 0, callarg;
	cmdfindcall_t sam_in;	/* buffer for samapi find command */
	rspdatarec_t sam_out;	/* buffer for result of samapi find command */
	cmdfindcounty_t cty_in;	/* buffer for find county command */
	rspfindcounty_t cty_out;	/* buffer for find county response */

	/*
	 * make sure got arg
	 */

	if (argc < 2)
	{
		printf("*** No call specified");
		exit(3);
	}

	/*
	 * make sure the resident code (SAMAPI.EXE)
	 * has been installed
	 */

	if (LocateSam())
	{
		printf("*** SAMAPI not loaded\n");
		exit(2);
	}

	/*
	 * check for county requested
	 */

	callarg = 1;
	if (argc > 2)
	{
		if (!stricmp(argv[1], "/c"))
		{
			wantcty = 1;
			callarg = 2;
		}
		else if (!stricmp(argv[2], "/c"))
		{
			wantcty = 1;
		}
	}
			

	/*
	 * build command block and call SAMAPI, function SamFindCall
	 */

	sam_in.packflags = 0;	/* 0 to unpack all data record fields */
	strncpy(sam_in.call, argv[callarg], 6);
	sam_in.call[6] = 0;
	err = CallSam(SamFindCall, &sam_in, &sam_out);

	/*
	 * check for unusual error
	 * something other that plain ole not found
	 */

	if (err != 0 && err != SerrNotFound)
	{
		printf("*** SAMAPI error %d\n", err);
		exit(2);
	}

	/*
	 * check for just not found
	 */

	if (err == SerrNotFound)
	{
		printf("*** Call not found\n");
		exit(1);
	}


	/* try to get county if /c option */
	if (wantcty)
	{
		cty_out.county[0] = 0;
		if (sam_out.d.Call[sam_out.d.Call[0] == ' '] != 'V')
		{
			/* it's a US call */
			strcpy(cty_in.zip, sam_out.d.Zip);
			memset(cty_in.reserved, 0, sizeof(cty_in.reserved));
			err = CallSam(SamFindCounty, &cty_in, &cty_out);
			if (err)
				cty_out.county[0] = 0;
		}
	}

	/*
	 * got a match, display the call data
	 */

	display_call(&sam_out.d, wantcty, cty_out.county);

	/*
	 * exit with 0 for success
	 */
	return 0;
}

/*******************
 * display_call
 *******************
 *
 * displays call format:
 * call,class,first m,last,address,city,state,zip,yearborn,[,county]
 * county shown only if 
 * input is pointer to data record
 *
 * returns nothing
 */

void display_call(datarec_t * d, int wantcty, char *county)
{
	/*
     * display call and class
     */
	printf("%s,%s,", d->Call + (d->Call[0] == ' '), d->Class);

	/* display first middle */
	printf("%s", d->FirstName);
	if (d->MidInitial[0] != ' ')
		printf(" %s", d->MidInitial);
	/*
	 * display last,addr,city,st,zip,yearborn
	 */
	printf(",%s,%s,%s,%s,%s,%s",
				d->LastName, d->Address, d->City, d->State, d->Zip,
				d->Dob);
	if (wantcty)
		printf(",%s", county);
	printf("\n");
}
