/*
 * findcall.c
 *
 * This program takes a callsign from the command line and searches
 * the sam database for a match.  If found the call data is
 * displayed in 4 lines as follows:
 *
 *  1: callsign
 *  2: first [m ]last
 *  3: address
 *  4: city, st zip
 *
 * If /c follows the call then the county is displayed if possible
 *
 * findcall depends on the API to the sam db provided by SAMAPI.EXE.  
 *
 * Any error messagess are 1 line long and start with "***" (in case
 * some other program wants to use the output of this program).
 *
 * to compile with Turbo-C
 *  tcc findcall.c samlib.c
 *
 * Other C compilers can be used, but a "pack structs" option may be
 * required.
 *
 * Copyright 1991 by Tom Thompson, Athens, AL and by RT Systems
 *                   [n4yos]
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#include "samapi.h"			/* samapi interface spec */

/*
 * functions in samlib.c
 */

int LocateSam(void);
int CallSam(int cmd, void far *cmdbuf, void far *rspbuf);

/*
 * decls for functions contained here
 */

void display_call_4lines(datarec_t * d);


/*******************
 * main
 *******************
 *
 * in: argv[1] is callsign
 *
 * returns (or exits()) non-zero if error, zero if call found and displayed
 */

main(int argc, char *argv[])
{
	int err;
	cmdfindcall_t sam_in;	/* buffer for samapi find command */
	rspdatarec_t sam_out;	/* buffer for result of samapi find command */
	cmdfindcounty_t cty_in;	/* buffer for find county command */
	rspfindcounty_t cty_out;	/* buffer for find county response */

	/*
	 * make sure got arg
	 */

	if (argc < 2)
	{
		printf("*** No call specified");
		exit(3);
	}

	/*
	 * make sure the resident code (SAMAPI.EXE)
	 * has been installed
	 */

	if (LocateSam())
	{
		printf("*** SAMAPI not loaded\n");
		exit(2);
	}

	/*
	 * build command block and call SAMAPI, function SamFindCall
	 */

	sam_in.packflags = 0;	/* 0 to unpack all data record fields */
	strncpy(sam_in.call, argv[1], 6);
	sam_in.call[6] = 0;
	err = CallSam(SamFindCall, &sam_in, &sam_out);

	/*
	 * check for unusual error
	 * something other that plain ole not found
	 */

	if (err != 0 && err != SerrNotFound)
	{
		printf("*** SAMAPI error %d\n", err);
		exit(2);
	}

	/*
	 * check for just not found
	 */

	if (err == SerrNotFound)
	{
		printf("*** Call not found\n");
		exit(1);
	}

	/*
	 * got a match, display the call data
	 */

	display_call_4lines(&sam_out.d);

	/* try to display county */
	if (argc == 3 && !stricmp(argv[2], "/c"))
	{
		if (sam_out.d.Call[sam_out.d.Call[0] == ' '] != 'V')
		{
			/* it's a US call */
			strcpy(cty_in.zip, sam_out.d.Zip);
			memset(cty_in.reserved, 0, sizeof(cty_in.reserved));
			err = CallSam(SamFindCounty, &cty_in, &cty_out);
			if (!err)
				printf("%s\n", cty_out.county);
		}
	}

	/*
	 * exit with 0 for success
	 */
	return 0;
}

/*******************
 * display_call
 *******************
 *
 * displays call in 4 line format:
 *  1: callsign
 *  2: first [m ]last
 *  3: address
 *  4: city, st zip
 *
 * input is pointer to data record
 *
 * returns nothing
 */

void display_call_4lines(datarec_t * d)
{
	/*
     * display call with leading space stripped
     */
	printf("%s\n", d->Call + (d->Call[0] == ' '));

	/*
	 * display first m last, but leave out middle and space
	 * if no middle initial
	 */
	printf("%s ", d->FirstName);
	if (d->MidInitial[0] != ' ')
		printf("%s ", d->MidInitial);
	printf("%s\n", d->LastName);

	/*
	 * address line, then city, st zip
	 */
	printf("%s\n", d->Address);
	printf("%s, %s %s\n", d->City, d->State, d->Zip);
}
